#include "chattestmanager.h"
#include "fixtures/chatapitest.h"
#include "testchatobjectfactory.h"
#include "testutilities.h"
#include "twitchsdk/chat/chatapi.h"
#include "twitchsdk/chat/ichatcommentmanager.h"
#include "twitchsdk/chat/ichatobjectfactory.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/socket.h"

#include <thread>

#include "gtest/gtest.h"

using namespace ttv;
using namespace ttv::chat;
using namespace ttv::chat::test;
using namespace ttv::test;

namespace {
/**
 * A helper IChatCommentListener implementation that proxies events into optional lambda calls.
 */
class TestChatCommentListener : public ttv::chat::IChatCommentListener {
 public:
  using ChatCommentManagerStateChangedFunc =
    std::function<void(UserId userId, const std::string& vodId, IChatCommentManager::PlayingState state)>;
  using ChatCommentsReceivedFunc =
    std::function<void(UserId userId, const std::string& vodId, std::vector<ChatComment>&& messageList)>;
  using ChatCommentsErrorReceivedFunc = std::function<void(const std::string& errorMsg, TTV_ErrorCode ec)>;

  void SetChatCommentManagerStateChanged(ChatCommentManagerStateChangedFunc cb) {
    mChatCommentManagerStateChanged = cb;
  }
  void SetChatCommentsReceived(ChatCommentsReceivedFunc cb) { mChatCommentsReceived = cb; }

 public:
  TestChatCommentListener() {}

  virtual void ChatCommentManagerStateChanged(
    UserId userId, const std::string& vodId, IChatCommentManager::PlayingState state) override {
    if (mChatCommentManagerStateChanged != nullptr) {
      mChatCommentManagerStateChanged(userId, vodId, state);
    }
  }

  virtual void ChatCommentsReceived(
    UserId userId, const std::string& vodId, std::vector<ChatComment>&& messageList) override {
    if (mChatCommentsReceived != nullptr) {
      mChatCommentsReceived(userId, vodId, std::move(messageList));
    }
  }

  virtual void ChatCommentsErrorReceived(const std::string& errorMsg, TTV_ErrorCode ec) override {
    if (mChatCommentsErrorReceived != nullptr) {
      mChatCommentsErrorReceived(errorMsg, ec);
    }
  }

 private:
  ChatCommentManagerStateChangedFunc mChatCommentManagerStateChanged;
  ChatCommentsReceivedFunc mChatCommentsReceived;
  ChatCommentsErrorReceivedFunc mChatCommentsErrorReceived;
};

const char* kVodId = "123";
const uint32_t kDefaultChatResponseLimit = 60;
}  // namespace

TEST_F(ChatApiTest, ChatComment_GetVodSuccess) {
  TestChatApiInitializationCallback(TokenizationOptions());

  MockResponse& response = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                             .SetResponseBodyFromFile("core/twitch_vod.json")
                             .SetType(ttv::HTTP_POST_REQUEST)
                             .AddJsonValue({"variables", "id"}, kVodId);

  auto result = mChatApi->CreateChatCommentManager(0, kVodId, std::make_shared<TestChatCommentListener>());
  ASSERT_FALSE(result.IsError());

  std::shared_ptr<ttv::chat::IChatCommentManager> chatCommentManager = result.GetResult();

  std::function<bool()> chatReplayVodName = [&chatCommentManager]() {
    auto channelIdResult = chatCommentManager->GetChannelId();
    return (!channelIdResult.IsError() && channelIdResult.GetResult() == 1001);
  };

  std::function<void()> updateFunction = [this]() { mChatApi->Update(); };

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, chatReplayVodName, updateFunction));
  response.AssertRequestsMade();
}

TEST_F(ChatApiTest, ChatComment_GetVodErrorRetryTimer) {
  TestChatApiInitializationCallback(TokenizationOptions());

  MockResponse& response = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                             .SetStatusCode(500)
                             .SetType(ttv::HTTP_POST_REQUEST)
                             .AddJsonValue({"variables", "id"}, kVodId);

  auto result = mChatApi->CreateChatCommentManager(0, kVodId, std::make_shared<TestChatCommentListener>());
  ASSERT_FALSE(result.IsError());

  std::shared_ptr<ttv::chat::IChatCommentManager> chatCommentManager = result.GetResult();

  std::function<bool()> requestCount = [&response]() { return response.GetRequestCount() == 2; };

  std::function<void()> updateFunction = [this]() { mChatApi->Update(); };

  ASSERT_TRUE(WaitUntilResultWithPollTask(3000, requestCount, updateFunction));
}

TEST_F(ChatApiTest, ChatComment_FetchNextCommentsByCursor) {
  TokenizationOptions options;
  options.emoticons = true;
  TestChatApiInitializationCallback(options);

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("core/twitch_vod.json")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "id"}, kVodId);

  MockResponse& response1 = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                              .SetType(ttv::HTTP_POST_REQUEST)
                              .AddJsonValue({"variables", "cursor"}, "")
                              .SetResponseBodyFromFile("chat/chatcomment_messages_cursor_next.json");

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();

  MockResponse& response2 = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                              .SetType(ttv::HTTP_POST_REQUEST)
                              .AddJsonValue({"variables", "cursor"}, "cursorUrl")
                              .SetResponseBodyFromFile("chat/chatcomment_messages.json");

  std::function<bool()> response1Requested = [&response1]() { return response1.GetRequestCount() == 1; };

  std::function<bool()> response2Requested = [&response2]() { return response2.GetRequestCount() == 1; };

  std::function<void()> updateFunction = [this]() { mChatApi->Update(); };

  auto result = mChatApi->CreateChatCommentManager(0, kVodId, std::make_shared<TestChatCommentListener>());
  ASSERT_FALSE(result.IsError());

  std::shared_ptr<ttv::chat::IChatCommentManager> chatCommentManager = result.GetResult();

  // Fetch the comments
  chatCommentManager->UpdatePlayhead(0);

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, response1Requested, updateFunction));
  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, response2Requested, updateFunction));
}

TEST_F(ChatApiTest, ChatComment_NoComments) {
  TokenizationOptions options;
  options.emoticons = true;
  TestChatApiInitializationCallback(options);

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("core/twitch_vod.json")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "id"}, kVodId);

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .AddJsonValue({"variables", "limit"}, 60)
    .SetResponseBodyFromFile("chat/chatcomment_messages_empty.json")
    .Done();

  auto listener = std::make_shared<TestChatCommentListener>();
  bool finishedPlaying = false;
  listener->SetChatCommentManagerStateChanged(
    [&finishedPlaying](UserId /*userId*/, const std::string& /*vodId*/, IChatCommentManager::PlayingState state) {
      if (state == IChatCommentManager::PlayingState::Finished) {
        finishedPlaying = true;
      }
    });

  std::function<bool()> finishedPlayingFunc = [&finishedPlaying]() { return finishedPlaying; };

  uint64_t playhead = 0;
  auto result = mChatApi->CreateChatCommentManager(0, kVodId, listener);
  ASSERT_FALSE(result.IsError());

  std::shared_ptr<ttv::chat::IChatCommentManager> chatCommentManager = result.GetResult();

  std::function<void(uint64_t)> updateFunction = [this, &playhead, chatCommentManager](uint64_t delta) {
    mChatApi->Update();
    playhead += delta;
    chatCommentManager->UpdatePlayhead(playhead);
  };

  ASSERT_TRUE(TTV_SUCCEEDED(chatCommentManager->Play()));
  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, finishedPlayingFunc, updateFunction));
}

TEST_F(ChatApiTest, ChatComment_AdvanceMessages) {
  TokenizationOptions options;
  options.emoticons = true;
  TestChatApiInitializationCallback(options);

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("core/twitch_vod.json")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "id"}, kVodId);

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .AddJsonValue({"variables", "limit"}, 60)
    .SetResponseBodyFromFile("chat/chatcomment_messages.json");

  auto listener = std::make_shared<TestChatCommentListener>();

  size_t messagesReceivedCount = 0;
  listener->SetChatCommentsReceived(
    [&messagesReceivedCount](UserId /*userId*/, const std::string& /*vodId*/,
      const std::vector<ChatComment>& messageList) { messagesReceivedCount += messageList.size(); });

  std::function<bool()> checkFunc = [&messagesReceivedCount]() { return messagesReceivedCount == 5; };

  uint64_t playhead = 0;
  auto result = mChatApi->CreateChatCommentManager(0, kVodId, listener);
  ASSERT_FALSE(result.IsError());

  std::shared_ptr<ttv::chat::IChatCommentManager> chatCommentManager = result.GetResult();

  std::function<void(uint64_t)> updateFunction = [this, &playhead, chatCommentManager](uint64_t delta) {
    mChatApi->Update();
    playhead += delta;
    chatCommentManager->UpdatePlayhead(playhead);
  };

  ASSERT_TRUE(TTV_SUCCEEDED(chatCommentManager->Play()));
  ASSERT_TRUE(WaitUntilResultWithPollTask(5000, checkFunc, updateFunction));
}

TEST_F(ChatApiTest, ChatComment_SeekInBuffer) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("core/twitch_vod.json")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "id"}, kVodId);

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();

  MockResponse& response1 = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                              .SetType(ttv::HTTP_POST_REQUEST)
                              .AddJsonValue({"variables", "cursor"}, "")
                              .SetResponseBodyFromFile("chat/chatcomment_messages_halfminute.json");

  MockResponse& response2 =
    mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
      .SetType(ttv::HTTP_POST_REQUEST)
      .AddJsonValue({"variables", "cursor"},
        "eyJpZCI6ImE1OGYxNjBlLTBjOWYtNDQyOS04ZGVlLTg1NzU4OTY5MDYyNCIsImhrIjoiYnJvYWRjYXN0OjM1MzMzMjg4NDMyIiwic2siOiJBQUFBQW9rWU5vQVZ1OHpJNFgxemdBIn0")
      .SetResponseBodyFromFile("chat/chatcomment_messages_after_halfminute.json");

  TokenizationOptions options;
  options.emoticons = true;
  TestChatApiInitializationCallback(options);

  auto listener = std::make_shared<TestChatCommentListener>();

  size_t messagesReceivedCount = 0;
  listener->SetChatCommentsReceived(
    [&messagesReceivedCount](UserId /*userId*/, const std::string& /*vodId*/,
      const std::vector<ChatComment>& messageList) { messagesReceivedCount += messageList.size(); });

  IChatCommentManager::PlayingState currentState;
  listener->SetChatCommentManagerStateChanged([&currentState](UserId /*userId*/, const std::string& /*vodId*/,
                                                IChatCommentManager::PlayingState state) { currentState = state; });

  std::function<bool()> response1Requested = [&response1]() { return response1.GetRequestCount() == 1; };

  std::function<bool()> response2Requested = [&response2]() { return response2.GetRequestCount() == 1; };

  std::function<bool()> currentStatePlaying = [&currentState]() {
    return currentState == IChatCommentManager::PlayingState::Playing;
  };

  std::function<bool()> messageInFirstThreeSecondsReceivedFunc = [&messagesReceivedCount]() {
    return messagesReceivedCount == 4;
  };

  std::function<bool()> totalMessagesReceivedFunc = [&messagesReceivedCount]() { return messagesReceivedCount == 6; };

  uint64_t playhead = 0;
  auto result = mChatApi->CreateChatCommentManager(0, kVodId, listener);
  ASSERT_FALSE(result.IsError());

  std::shared_ptr<ttv::chat::IChatCommentManager> chatCommentManager = result.GetResult();

  std::function<void(uint64_t)> updateFunction = [this, &playhead, chatCommentManager](uint64_t delta) {
    mChatApi->Update();
    playhead += delta;
    chatCommentManager->UpdatePlayhead(playhead);
  };

  ASSERT_TRUE(TTV_SUCCEEDED(chatCommentManager->Play()));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, response1Requested, updateFunction));
  ASSERT_TRUE(WaitUntilResultWithPollTask(500, currentStatePlaying, updateFunction));

  // Get 4 messages at timestamp 1 second and 1 message at timestamp 31 seconds.
  ASSERT_TRUE(WaitUntilResultWithPollTask(3000, messageInFirstThreeSecondsReceivedFunc, updateFunction));
  playhead = 29500;

  // We are now within 5 seconds of our last loaded time, must fetch again for more messages
  ASSERT_TRUE(WaitUntilResultWithPollTask(3000, response2Requested, updateFunction));
  ASSERT_TRUE(WaitUntilResultWithPollTask(3000, totalMessagesReceivedFunc, updateFunction));
}

TEST_F(ChatApiTest, ChatComment_SeekPastBuffer) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("core/twitch_vod.json")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "id"}, kVodId);

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();

  MockResponse& response1 = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                              .SetType(ttv::HTTP_POST_REQUEST)
                              .AddJsonValue({"variables", "contentOffsetSeconds"}, 0)
                              .SetResponseBodyFromFile("chat/chatcomment_messages_halfminute.json");

  MockResponse& response2 = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                              .SetType(ttv::HTTP_POST_REQUEST)
                              .AddJsonValue({"variables", "contentOffsetSeconds"}, 32)
                              .SetResponseBodyFromFile("chat/chatcomment_messages_after_halfminute.json");

  TokenizationOptions options;
  options.emoticons = true;
  TestChatApiInitializationCallback(options);

  auto listener = std::make_shared<TestChatCommentListener>();

  size_t messagesReceivedCount = 0;
  listener->SetChatCommentsReceived(
    [&messagesReceivedCount](UserId /*userId*/, const std::string& /*vodId*/,
      const std::vector<ChatComment>& messageList) { messagesReceivedCount += messageList.size(); });

  IChatCommentManager::PlayingState currentState;
  listener->SetChatCommentManagerStateChanged([&currentState](UserId /*userId*/, const std::string& /*vodId*/,
                                                IChatCommentManager::PlayingState state) { currentState = state; });

  std::function<bool()> response1Requested = [&response1]() { return response1.GetRequestCount() == 1; };

  std::function<bool()> response2Requested = [&response2]() { return response2.GetRequestCount() == 1; };

  std::function<bool()> currentStateBuffering = [&currentState]() {
    return currentState == IChatCommentManager::PlayingState::Buffering;
  };

  std::function<bool()> currentStatePlaying = [&currentState]() {
    return currentState == IChatCommentManager::PlayingState::Playing;
  };

  std::function<bool()> messageInBufferReceivedFunc = [&messagesReceivedCount]() { return messagesReceivedCount > 1; };

  uint64_t playhead = 0;
  auto result = mChatApi->CreateChatCommentManager(0, kVodId, listener);
  ASSERT_FALSE(result.IsError());

  std::shared_ptr<ttv::chat::IChatCommentManager> chatCommentManager = result.GetResult();

  std::function<void(uint64_t)> updateFunction = [this, &playhead, chatCommentManager](uint64_t delta) {
    mChatApi->Update();
    playhead += delta;
    chatCommentManager->UpdatePlayhead(playhead);
  };

  ASSERT_TRUE(TTV_SUCCEEDED(chatCommentManager->Play()));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, response1Requested, updateFunction));
  ASSERT_TRUE(WaitUntilResultWithPollTask(500, currentStatePlaying, updateFunction));

  playhead = 32 * 1000;
  updateFunction(0);  // This makes sure a fetch at 32 sec playhead time happens

  ASSERT_TRUE(WaitUntilResultWithPollTask(500, currentStateBuffering, updateFunction));
  ASSERT_TRUE(WaitUntilResultWithPollTask(500, response2Requested, updateFunction));
  ASSERT_TRUE(WaitUntilResultWithPollTask(500, currentStatePlaying, updateFunction));
  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, messageInBufferReceivedFunc, updateFunction));
}

TEST_F(ChatApiTest, ChatComment_SeekBehind) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("core/twitch_vod.json")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "id"}, kVodId);

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();

  MockResponse& response1 = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                              .SetResponseBodyFromFile("chat/chatcomment_messages.json")
                              .SetType(ttv::HTTP_POST_REQUEST)
                              .AddJsonValue({"variables", "limit"}, 60)
                              .AddJsonValue({"variables", "videoID"}, kVodId)
                              .AddJsonValue({"variables", "cursor"}, "");

  TokenizationOptions options;
  options.emoticons = true;
  TestChatApiInitializationCallback(options);

  auto listener = std::make_shared<TestChatCommentListener>();

  size_t messagesReceivedCount = 0;
  listener->SetChatCommentsReceived(
    [&messagesReceivedCount](UserId /*userId*/, const std::string& /*vodId*/,
      const std::vector<ChatComment>& messageList) { messagesReceivedCount += messageList.size(); });

  IChatCommentManager::PlayingState currentState;
  listener->SetChatCommentManagerStateChanged([&currentState](UserId /*userId*/, const std::string& /*vodId*/,
                                                IChatCommentManager::PlayingState state) { currentState = state; });

  std::function<bool()> response1Requested = [&response1]() { return response1.GetRequestCount() == 1; };

  std::function<bool()> response1RequestedTwice = [&response1]() { return response1.GetRequestCount() == 2; };

  std::function<bool()> currentStateBuffering = [&currentState]() {
    return currentState == IChatCommentManager::PlayingState::Buffering;
  };

  std::function<bool()> currentStatePlaying = [&currentState]() {
    return currentState == IChatCommentManager::PlayingState::Playing;
  };

  std::function<bool()> messageInBufferReceivedFunc = [&messagesReceivedCount]() { return messagesReceivedCount > 0; };

  std::function<bool()> messagesAfterSeekReceivedFunc = [&messagesReceivedCount]() {
    return messagesReceivedCount == 5;
  };

  uint64_t playhead = 0;
  auto result = mChatApi->CreateChatCommentManager(0, kVodId, listener);
  ASSERT_FALSE(result.IsError());

  std::shared_ptr<ttv::chat::IChatCommentManager> chatCommentManager = result.GetResult();

  std::function<void(uint64_t)> updateFunction = [this, &playhead, chatCommentManager](uint64_t delta) {
    mChatApi->Update();
    playhead += delta;
    chatCommentManager->UpdatePlayhead(playhead);
  };

  ASSERT_TRUE(TTV_SUCCEEDED(chatCommentManager->Play()));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, response1Requested, updateFunction));
  ASSERT_TRUE(WaitUntilResultWithPollTask(500, currentStatePlaying, updateFunction));
  ASSERT_TRUE(WaitUntilResultWithPollTask(1500, messageInBufferReceivedFunc, updateFunction));

  messagesReceivedCount = 0;
  playhead = 0;  // A seek backwards should clear any messages in the buffer.

  ASSERT_TRUE(WaitUntilResultWithPollTask(500, currentStateBuffering, updateFunction));
  ASSERT_TRUE(WaitUntilResultWithPollTask(500, response1RequestedTwice, updateFunction));
  ASSERT_TRUE(WaitUntilResultWithPollTask(500, currentStatePlaying, updateFunction));
  ASSERT_TRUE(WaitUntilResultWithPollTask(5000, messagesAfterSeekReceivedFunc, updateFunction));
}

TEST_F(ChatApiTest, ChatComment_RefetchMessagesOnError) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("core/twitch_vod.json")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "id"}, kVodId);

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();

  MockResponse& response1 = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                              .SetType(ttv::HTTP_POST_REQUEST)
                              .AddJsonValue({"variables", "limit"}, 60)
                              .SetResponseBodyFromFile("chat/chatcomment_messages_invalid_json.json");

  TokenizationOptions options;
  options.emoticons = true;
  TestChatApiInitializationCallback(options);

  auto listener = std::make_shared<TestChatCommentListener>();

  IChatCommentManager::PlayingState currentState;
  listener->SetChatCommentManagerStateChanged([&currentState](UserId /*userId*/, const std::string& /*vodId*/,
                                                IChatCommentManager::PlayingState state) { currentState = state; });

  std::function<bool()> response1RequestedTwice = [&response1]() { return response1.GetRequestCount() == 2; };

  std::function<bool()> buffering = [&currentState]() {
    return currentState == IChatCommentManager::PlayingState::Buffering;
  };

  auto result = mChatApi->CreateChatCommentManager(0, kVodId, listener);
  ASSERT_FALSE(result.IsError());

  std::shared_ptr<ttv::chat::IChatCommentManager> chatCommentManager = result.GetResult();

  uint64_t playhead = 0;
  std::function<void(uint64_t)> updateFunction = [this, &playhead, chatCommentManager](uint64_t delta) {
    mChatApi->Update();
    playhead += delta;
    chatCommentManager->UpdatePlayhead(playhead);
  };

  ASSERT_TRUE(TTV_SUCCEEDED(chatCommentManager->Play()));
  ASSERT_TRUE(WaitUntilResultWithPollTask(5000, response1RequestedTwice, updateFunction));
  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, buffering, updateFunction));
}

TEST_F(ChatApiTest, ChatComment_PrefetchDelay) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("core/twitch_vod.json")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "id"}, kVodId);

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();

  MockResponse& response1 = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                              .SetType(ttv::HTTP_POST_REQUEST)
                              .AddJsonValue({"variables", "cursor"}, "")
                              .SetResponseBodyFromFile("chat/chatcomment_messages_halfminute.json");

  MockResponse& response2 =
    mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
      .SetType(ttv::HTTP_POST_REQUEST)
      .AddJsonValue({"variables", "cursor"},
        "eyJpZCI6ImE1OGYxNjBlLTBjOWYtNDQyOS04ZGVlLTg1NzU4OTY5MDYyNCIsImhrIjoiYnJvYWRjYXN0OjM1MzMzMjg4NDMyIiwic2siOiJBQUFBQW9rWU5vQVZ1OHpJNFgxemdBIn0")
      .SetResponseBodyFromFile("chat/chatcomment_messages_after_halfminute.json")
      .SetDelay(800);

  TokenizationOptions options;
  options.emoticons = true;
  TestChatApiInitializationCallback(options);

  auto listener = std::make_shared<TestChatCommentListener>();

  size_t messagesReceivedCount = 0;
  listener->SetChatCommentsReceived(
    [&messagesReceivedCount](UserId /*userId*/, const std::string& /*vodId*/,
      const std::vector<ChatComment>& messageList) { messagesReceivedCount += messageList.size(); });

  IChatCommentManager::PlayingState currentState;
  listener->SetChatCommentManagerStateChanged([&currentState](UserId /*userId*/, const std::string& /*vodId*/,
                                                IChatCommentManager::PlayingState state) { currentState = state; });

  std::function<bool()> response1Requested = [&response1]() { return response1.GetRequestCount() == 1; };

  std::function<bool()> response2Requested = [&response2]() { return response2.GetRequestCount() == 1; };

  std::function<bool()> currentStateBuffering = [&currentState]() {
    return currentState == IChatCommentManager::PlayingState::Buffering;
  };

  std::function<bool()> currentStatePlaying = [&currentState]() {
    return currentState == IChatCommentManager::PlayingState::Playing;
  };

  std::function<bool()> messageInFirstTwoSecondsReceivedFunc = [&messagesReceivedCount]() {
    return messagesReceivedCount == 4;
  };

  std::function<bool()> totalMessagesReceivedFunc = [&messagesReceivedCount]() { return messagesReceivedCount == 6; };

  auto result = mChatApi->CreateChatCommentManager(0, kVodId, listener);
  ASSERT_FALSE(result.IsError());

  std::shared_ptr<ttv::chat::IChatCommentManager> chatCommentManager = result.GetResult();

  uint64_t playhead = 0;
  std::function<void(uint64_t)> updateFunction = [this, &playhead, chatCommentManager](uint64_t delta) {
    mChatApi->Update();
    playhead += delta;
    chatCommentManager->UpdatePlayhead(playhead);
  };

  ASSERT_TRUE(TTV_SUCCEEDED(chatCommentManager->Play()));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, response1Requested, updateFunction));
  ASSERT_TRUE(WaitUntilResultWithPollTask(500, currentStatePlaying, updateFunction));

  // Need to wait for playhead to reach the comment's time to receive them.
  ASSERT_TRUE(WaitUntilResultWithPollTask(1500, messageInFirstTwoSecondsReceivedFunc, updateFunction));
  playhead = 30 * 1000;

  // We're requesting new comments.
  ASSERT_TRUE(WaitUntilResultWithPollTask(2000, response2Requested, updateFunction));

  // Comments have been received, start playing.
  ASSERT_TRUE(WaitUntilResultWithPollTask(2000, currentStatePlaying, updateFunction));

  // Again, wait for playhead to reach comment's time.
  ASSERT_TRUE(WaitUntilResultWithPollTask(3000, totalMessagesReceivedFunc, updateFunction));
}

TEST_F(ChatApiTest, ChatComment_PlayingState) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("core/twitch_vod.json")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "id"}, kVodId);

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();

  MockResponse& response1 = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                              .SetType(ttv::HTTP_POST_REQUEST)
                              .AddJsonValue({"variables", "limit"}, 60)
                              .SetResponseBodyFromFile("chat/chatcomment_messages.json");

  TokenizationOptions options;
  options.emoticons = true;
  TestChatApiInitializationCallback(options);

  auto listener = std::make_shared<TestChatCommentListener>();

  IChatCommentManager::PlayingState currentState;
  listener->SetChatCommentManagerStateChanged([&currentState](UserId /*userId*/, const std::string& /*vodId*/,
                                                IChatCommentManager::PlayingState state) { currentState = state; });

  std::function<bool()> response1Requested = [&response1]() { return response1.GetRequestCount() == 1; };

  std::function<bool()> currentStateBuffering = [&currentState]() {
    return currentState == IChatCommentManager::PlayingState::Buffering;
  };

  std::function<bool()> currentStatePlaying = [&currentState]() {
    return currentState == IChatCommentManager::PlayingState::Playing;
  };

  std::function<bool()> currentStatePaused = [&currentState]() {
    return currentState == IChatCommentManager::PlayingState::Paused;
  };

  std::function<bool()> currentStateFinished = [&currentState]() {
    return currentState == IChatCommentManager::PlayingState::Finished;
  };

  auto result = mChatApi->CreateChatCommentManager(0, kVodId, listener);
  ASSERT_FALSE(result.IsError());

  std::shared_ptr<ttv::chat::IChatCommentManager> chatCommentManager = result.GetResult();

  uint64_t playhead = 0;
  std::function<void(uint64_t)> deltaUpdateFunction = [this, &playhead, chatCommentManager](uint64_t delta) {
    mChatApi->Update();
    playhead += delta;
    chatCommentManager->UpdatePlayhead(playhead);
  };

  std::function<void()> updateFunction = [this]() { mChatApi->Update(); };

  chatCommentManager->UpdatePlayhead(playhead);
  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, response1Requested, updateFunction));

  ASSERT_TRUE(TTV_SUCCEEDED(chatCommentManager->Play()));
  ASSERT_TRUE(WaitUntilResultWithPollTask(500, currentStatePlaying, deltaUpdateFunction));

  ASSERT_TRUE(TTV_SUCCEEDED(chatCommentManager->Pause()));
  ASSERT_TRUE(WaitUntilResultWithPollTask(500, currentStatePaused, updateFunction));

  // Seek right before last comment
  playhead = 9900;
  ASSERT_TRUE(WaitUntilResultWithPollTask(500, currentStatePaused, updateFunction));

  ASSERT_TRUE(TTV_SUCCEEDED(chatCommentManager->Play()));
  ASSERT_TRUE(WaitUntilResultWithPollTask(500, currentStatePlaying, deltaUpdateFunction));
  ASSERT_TRUE(WaitUntilResultWithPollTask(500, currentStateFinished, deltaUpdateFunction));

  playhead = 0;
  ASSERT_TRUE(WaitUntilResultWithPollTask(500, currentStateBuffering, deltaUpdateFunction));
  ASSERT_TRUE(WaitUntilResultWithPollTask(500, currentStatePlaying, deltaUpdateFunction));
}

TEST_F(ChatApiTest, ChatComment_FetchChatComments) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("core/twitch_vod.json")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "id"}, kVodId);

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .AddJsonValue({"variables", "contentOffsetSeconds"}, 15)
    .SetResponseBodyFromFile("chat/chatcomment_messages_halfminute.json")
    .Done();

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .AddJsonValue({"variables", "contentOffsetSeconds"}, 0)
    .SetResponseBodyFromFile("chat/chatcomment_messages_halfminute.json")
    .Done();

  TokenizationOptions options;
  options.emoticons = true;
  TestChatApiInitializationCallback(options);

  auto result = mChatApi->CreateChatCommentManager(0, kVodId, std::make_shared<TestChatCommentListener>());
  ASSERT_FALSE(result.IsError());

  std::shared_ptr<ttv::chat::IChatCommentManager> chatCommentManager = result.GetResult();

  std::string cursorUrl;
  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };

  uint32_t offsetSeconds = 15;
  TTV_ErrorCode ec = chatCommentManager->FetchComments(offsetSeconds * 1000, kDefaultChatResponseLimit,
    [&cursorUrl, &callbackCalled](TTV_ErrorCode ec, std::vector<ChatComment>&& result, std::string&& nextCursor) {
      callbackCalled = true;
      ASSERT_TRUE(TTV_SUCCEEDED(ec));

      ASSERT_EQ(result[0].commentId, "a6Q08ZUe0BSO1A");
      ASSERT_EQ(result[0].channelId, 22510310);
      ASSERT_EQ(result[0].contentId, "123");
      ASSERT_EQ(result[0].timestampMilliseconds, 0);
      ASSERT_EQ(result[0].commentSource, ChatCommentSource::Comment);
      ASSERT_EQ(result[0].publishedState, ChatCommentPublishedState::Published);
      ASSERT_EQ(result[0].messageInfo.userName, "twitch");
      ASSERT_EQ(result[0].messageInfo.displayName, "Twitch");
      ASSERT_EQ(result[0].messageInfo.userId, 12826);
      ASSERT_EQ(result[0].messageInfo.tokens.size(), 2);
      ASSERT_EQ(result[0].messageInfo.nameColorARGB, 4278222848);
      ASSERT_EQ(result[0].messageInfo.badges.size(), 1);

      cursorUrl = std::move(nextCursor);
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(callbackCalled);

  callbackCalled = false;
  ec = chatCommentManager->FetchComments(cursorUrl, kDefaultChatResponseLimit,
    [&cursorUrl, &callbackCalled](TTV_ErrorCode ec, std::vector<ChatComment>&& result, std::string&& nextCursor) {
      callbackCalled = true;
      ASSERT_TRUE(TTV_SUCCEEDED(ec));

      ASSERT_EQ(result[0].commentId, "a6Q08ZUe0BSO1A");
      ASSERT_EQ(result[0].channelId, 22510310);
      ASSERT_EQ(result[0].contentId, "123");
      ASSERT_EQ(result[0].timestampMilliseconds, 0);
      ASSERT_EQ(result[0].commentSource, ChatCommentSource::Comment);
      ASSERT_EQ(result[0].publishedState, ChatCommentPublishedState::Published);
      ASSERT_EQ(result[0].messageInfo.userName, "twitch");
      ASSERT_EQ(result[0].messageInfo.displayName, "Twitch");
      ASSERT_EQ(result[0].messageInfo.userId, 12826);
      ASSERT_EQ(result[0].messageInfo.tokens.size(), 2);
      ASSERT_EQ(result[0].messageInfo.nameColorARGB, 4278222848);
      ASSERT_EQ(result[0].messageInfo.badges.size(), 1);

      cursorUrl = std::move(nextCursor);
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(callbackCalled);
}

TEST_F(ChatApiTest, ChatComment_FetchChatComment) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("core/twitch_vod.json")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "id"}, kVodId);

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();

  mHttpRequest->AddResponse("https://api.twitch.tv/kraken/videos/comments/commentid")
    .SetResponseBodyFromFile("chat/chatcomment_singlecomment.json")
    .Done();

  TokenizationOptions options;
  options.emoticons = true;
  TestChatApiInitializationCallback(options);

  auto result = mChatApi->CreateChatCommentManager(0, kVodId, std::make_shared<TestChatCommentListener>());
  ASSERT_FALSE(result.IsError());

  std::shared_ptr<ttv::chat::IChatCommentManager> chatCommentManager = result.GetResult();

  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };

  TTV_ErrorCode ec =
    chatCommentManager->FetchComment("commentid", [&callbackCalled](TTV_ErrorCode ec, ChatComment&& result) {
      callbackCalled = true;
      ASSERT_TRUE(TTV_SUCCEEDED(ec));
      ASSERT_EQ(result.commentId, "commentid");
      ASSERT_EQ(result.channelId, 22510310);
      ASSERT_EQ(result.contentId, "158180399");
      ASSERT_EQ(result.timestampMilliseconds, 5000);
      ASSERT_EQ(result.commentSource, ChatCommentSource::Comment);
      ASSERT_EQ(result.publishedState, ChatCommentPublishedState::Published);
      ASSERT_EQ(result.messageInfo.userName, "twitch");
      ASSERT_EQ(result.messageInfo.displayName, "Twitch");
      ASSERT_EQ(result.messageInfo.userId, 12826);
      ASSERT_EQ(result.messageInfo.tokens.size(), 1);
      ASSERT_EQ(result.messageInfo.nameColorARGB, 4279874421);
      ASSERT_EQ(result.messageInfo.badges.size(), 2);
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(callbackCalled);
}

TEST_F(ChatApiTest, ChatComment_PostChatComment) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("core/twitch_vod.json")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "id"}, kVodId);

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(HTTP_POST_REQUEST)
    .SetResponseBodyFromFile("chat/chatcomment_singlecomment_post.json")
    .AddJsonValue({"variables", "message"}, "First")
    .Done();

  TokenizationOptions options;
  options.emoticons = true;
  TestChatApiInitializationCallback(options);

  auto result = mChatApi->CreateChatCommentManager(0, kVodId, std::make_shared<TestChatCommentListener>());
  ASSERT_FALSE(result.IsError());

  std::shared_ptr<ttv::chat::IChatCommentManager> chatCommentManager = result.GetResult();

  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };

  TTV_ErrorCode ec = chatCommentManager->PostComment(
    "First", 5, [&callbackCalled](TTV_ErrorCode ec, ChatComment&& result, std::string&& /*errorMessage*/) {
      callbackCalled = true;
      ASSERT_TRUE(TTV_SUCCEEDED(ec));
      ASSERT_EQ(result.commentId, "commentid");
      ASSERT_EQ(result.channelId, 22510310);
      ASSERT_EQ(result.contentId, "158180399");
      ASSERT_EQ(result.timestampMilliseconds, 5000);
      ASSERT_EQ(result.commentSource, ChatCommentSource::Comment);
      ASSERT_EQ(result.publishedState, ChatCommentPublishedState::Published);
      ASSERT_EQ(result.messageInfo.userName, "twitch");
      ASSERT_EQ(result.messageInfo.displayName, "Twitch");
      ASSERT_EQ(result.messageInfo.userId, 12826);
      ASSERT_EQ(result.messageInfo.tokens.size(), 1);
      ASSERT_EQ(result.messageInfo.nameColorARGB, 4279874421);
      ASSERT_EQ(result.messageInfo.badges.size(), 2);
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(callbackCalled);
}

TEST_F(ChatApiTest, ChatComment_PostChatCommentFailed) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("core/twitch_vod.json")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "id"}, kVodId);

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "message"}, "some profanity")
    .SetResponseBodyFromFile("chat/chatcomment_postcomment_error.json")
    .SetStatusCode(200)
    .Done();

  TokenizationOptions options;
  options.emoticons = true;
  TestChatApiInitializationCallback(options);

  auto result = mChatApi->CreateChatCommentManager(0, kVodId, std::make_shared<TestChatCommentListener>());
  ASSERT_FALSE(result.IsError());

  std::shared_ptr<ttv::chat::IChatCommentManager> chatCommentManager = result.GetResult();

  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };

  TTV_ErrorCode ec = chatCommentManager->PostComment(
    "some profanity", 5, [&callbackCalled](TTV_ErrorCode ec, ChatComment&& /*result*/, std::string&& errorMessage) {
      callbackCalled = true;
      ASSERT_FALSE(TTV_SUCCEEDED(ec));
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(callbackCalled);
}

TEST_F(ChatApiTest, ChatComment_DeleteChatComment) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("core/twitch_vod.json")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "id"}, kVodId);

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();

  mHttpRequest->AddResponse("https://api.twitch.tv/kraken/videos/comments/commentid")
    .SetType(HTTP_DELETE_REQUEST)
    .SetStatusCode(204)
    .Done();

  TokenizationOptions options;
  options.emoticons = true;
  TestChatApiInitializationCallback(options);

  auto result = mChatApi->CreateChatCommentManager(0, kVodId, std::make_shared<TestChatCommentListener>());
  ASSERT_FALSE(result.IsError());

  std::shared_ptr<ttv::chat::IChatCommentManager> chatCommentManager = result.GetResult();

  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };

  TTV_ErrorCode ec = chatCommentManager->DeleteComment("commentid", [&callbackCalled](TTV_ErrorCode ec) {
    callbackCalled = true;
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
  });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(callbackCalled);
}

TEST_F(ChatApiTest, ChatComment_ReportChatComment) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("core/twitch_vod.json")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "id"}, kVodId);

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();

  mHttpRequest->AddResponse("https://api.twitch.tv/kraken/videos/comments/commentid/reports")
    .SetType(HTTP_POST_REQUEST)
    .SetStatusCode(204)
    .Done();

  TokenizationOptions options;
  options.emoticons = true;
  TestChatApiInitializationCallback(options);

  auto result = mChatApi->CreateChatCommentManager(0, kVodId, std::make_shared<TestChatCommentListener>());
  ASSERT_FALSE(result.IsError());

  std::shared_ptr<ttv::chat::IChatCommentManager> chatCommentManager = result.GetResult();

  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };

  TTV_ErrorCode ec = chatCommentManager->ReportComment(
    "commentid", "report reason", "report description", [&callbackCalled](TTV_ErrorCode ec) {
      callbackCalled = true;
      ASSERT_TRUE(TTV_SUCCEEDED(ec));
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(callbackCalled);
}

TEST_F(ChatApiTest, ChatComment_GetChatCommentReplies) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("core/twitch_vod.json")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "id"}, kVodId);

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();

  mHttpRequest->AddResponse("https://api.twitch.tv/kraken/videos/comments/commentid/replies?limit=50")
    .SetType(HTTP_GET_REQUEST)
    .SetResponseBodyFromFile("chat/chatcomment_getrepliestask.json")
    .Done();

  TokenizationOptions options;
  options.emoticons = true;
  TestChatApiInitializationCallback(options);

  auto result = mChatApi->CreateChatCommentManager(0, kVodId, std::make_shared<TestChatCommentListener>());
  ASSERT_FALSE(result.IsError());

  std::shared_ptr<ttv::chat::IChatCommentManager> chatCommentManager = result.GetResult();

  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };

  TTV_ErrorCode ec = chatCommentManager->FetchCommentReplies(
    "commentid", [&callbackCalled](TTV_ErrorCode ec, std::vector<ChatComment>&& result) {
      callbackCalled = true;
      ASSERT_TRUE(TTV_SUCCEEDED(ec));
      ASSERT_EQ(result.size(), 2);
      ASSERT_EQ(result[0].commentId, "commentreplyid1");
      ASSERT_EQ(result[0].channelId, 12826);
      ASSERT_EQ(result[0].contentId, "26282362096");
      ASSERT_EQ(result[0].timestampMilliseconds, 5000);
      ASSERT_EQ(result[0].commentSource, ChatCommentSource::Comment);
      ASSERT_EQ(result[0].publishedState, ChatCommentPublishedState::Published);
      ASSERT_EQ(result[0].parentCommentId, "commentid");
      ASSERT_EQ(result[0].messageInfo.userName, "twitch");
      ASSERT_EQ(result[0].messageInfo.displayName, "Twitch");
      ASSERT_EQ(result[0].messageInfo.userId, 12826);
      ASSERT_EQ(result[0].messageInfo.tokens.size(), 1);
      ASSERT_EQ(result[0].messageInfo.nameColorARGB, 4284760484);
      ASSERT_EQ(result[0].messageInfo.badges.size(), 3);

      ASSERT_EQ(result[1].commentId, "commentreplyid2");
      ASSERT_EQ(result[1].channelId, 12826);
      ASSERT_EQ(result[1].contentId, "26282362096");
      ASSERT_EQ(result[1].timestampMilliseconds, 5000);
      ASSERT_EQ(result[1].commentSource, ChatCommentSource::Comment);
      ASSERT_EQ(result[1].publishedState, ChatCommentPublishedState::Published);
      ASSERT_EQ(result[1].parentCommentId, "commentid");
      ASSERT_EQ(result[1].messageInfo.userName, "twitch");
      ASSERT_EQ(result[1].messageInfo.displayName, "Twitch");
      ASSERT_EQ(result[1].messageInfo.userId, 12826);
      ASSERT_EQ(result[1].messageInfo.tokens.size(), 1);
      ASSERT_EQ(result[1].messageInfo.nameColorARGB, 4284760484);
      ASSERT_EQ(result[1].messageInfo.badges.size(), 3);
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(callbackCalled);
}

TEST_F(ChatApiTest, ChatComment_PostChatCommentReply) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("core/twitch_vod.json")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "id"}, kVodId);

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();

  mHttpRequest->AddResponse("https://api.twitch.tv/kraken/videos/comments/commentid/replies")
    .SetType(HTTP_POST_REQUEST)
    .SetResponseBodyFromFile("chat/chatcomment_postreplytask.json")
    .Done();

  TokenizationOptions options;
  options.emoticons = true;
  TestChatApiInitializationCallback(options);

  auto result = mChatApi->CreateChatCommentManager(0, kVodId, std::make_shared<TestChatCommentListener>());
  ASSERT_FALSE(result.IsError());

  std::shared_ptr<ttv::chat::IChatCommentManager> chatCommentManager = result.GetResult();

  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };

  TTV_ErrorCode ec = chatCommentManager->PostCommentReply(
    "commentid", "this is a reply", [&callbackCalled](TTV_ErrorCode ec, ChatComment&& result) {
      callbackCalled = true;
      ASSERT_TRUE(TTV_SUCCEEDED(ec));
      ASSERT_EQ(result.commentId, "commentreplyid");
      ASSERT_EQ(result.channelId, 12826);
      ASSERT_EQ(result.contentId, "175578515");
      ASSERT_EQ(result.parentCommentId, "commentid");
      ASSERT_EQ(result.timestampMilliseconds, 5000);
      ASSERT_EQ(result.commentSource, ChatCommentSource::Comment);
      ASSERT_EQ(result.publishedState, ChatCommentPublishedState::Published);
      ASSERT_EQ(result.messageInfo.userName, "twitch");
      ASSERT_EQ(result.messageInfo.displayName, "Twitch");
      ASSERT_EQ(result.messageInfo.userId, 12826);
      ASSERT_EQ(result.messageInfo.tokens.size(), 1);
      ASSERT_EQ(result.messageInfo.nameColorARGB, 4284760484);
      ASSERT_EQ(result.messageInfo.badges.size(), 3);
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(callbackCalled);
}
