/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "chattestmanager.h"
#include "fixtures/chatapitest.h"
#include "twitchsdk/core/stringutilities.h"

#include "gtest/gtest.h"

#if WIN32
#include <Windows.h>

#include <Psapi.h>
#endif

using namespace ttv;
using namespace ttv::chat;
using namespace ttv::chat::test;

// TODO: We probably don't need these tests since we don't allocate
// raw memory anymore
#if 0

namespace
{
    const int32_t kNumIterations = 10000;
    const int32_t kLeakAllowanceBytes = 1 * 1024; // Kilobytes

    int32_t GetProcessMemoryUsage()
    {
#if WIN32
        _PROCESS_MEMORY_COUNTERS procMemCounters;
        GetProcessMemoryInfo(GetCurrentProcess(), &procMemCounters, sizeof(procMemCounters));
        return static_cast<int32_t>(procMemCounters.WorkingSetSize);
#else
        return 0;
#endif
    }

    bool IsMemoryWithinAllowance(int32_t initialMemoryUsage, int32_t leakTolerance)
    {
        int32_t currentMemoryUsage = GetProcessMemoryUsage();

        if (currentMemoryUsage <= initialMemoryUsage)
        {
            return true;
        }

        return( (currentMemoryUsage - initialMemoryUsage) <= leakTolerance );
    }

    std::vector<std::unique_ptr<MessageToken>> AllocateTokens(uint32_t count)
    {
        std::vector<std::unique_ptr<MessageToken>> tokens;
        tokens.resize(count);

        for (uint32_t i = 0; i < count; ++i)
        {
            switch (i % 5)
            {
            case 0:
                tokens[i] = std::make_unique<TextToken>("text");
                break;
            case 1:
                tokens[i] = std::make_unique<EmoticonToken>("emoteText", 42);
                break;
            case 2:
                tokens[i] = std::make_unique<MentionToken>("mention", 1234);
                break;
            case 3:
                tokens[i] = std::make_unique<UrlToken>("url", false);
                break;
            case 4:
                tokens[i] = std::make_unique<BitsToken>("cheer", 1000);
                break;
            }
        }

        return tokens;
    }

    void AllocateMessage(Message& msg)
    {
        msg.threadId = "threadId";
        msg.userName = "userName";
        msg.tokens = AllocateTokens(5);
    }

    std::vector<Message> AllocateMessageArray(uint32_t count)
    {
        std::vector<Message> messages;
        messages.resize(count);
        for (uint32_t i = 0; i < count; ++i)
        {
            AllocateMessage(messages[i]);
        }

        return messages;
    }

    void AllocateMessageVector(std::vector<Message>& vector, uint32_t count)
    {
        for (uint32_t i = 0; i < count; ++i)
        {
            vector.emplace_back();
            auto& msg = vector.back();
            msg.threadId = "threadId";
            msg.userName = "username";
            msg.tokens = AllocateTokens(5);
        }
    }

    void AllocateThreadData(ThreadData& thread)
    {
        thread.lastMessage = std::make_unique<Message>();
        AllocateMessage(*thread.lastMessage);

        thread.participants.resize(4);

        thread.threadId = "threadId";
    }

    std::vector<ThreadData> AllocateThreadDataVector(uint32_t count)
    {
        std::vector<ThreadData> threadData;
        threadData.resize(count);
        for (uint32_t i = 0; i < count; ++i)
        {
            AllocateThreadData(threadData[i]);
        }
        return threadData;
    }

    void AllocateBadgeSets(BadgeSets& badgeSets)
    {
        badgeSets.language = "EN";

        for (auto& kvp : badgeSets.badgeSets)
        {
            auto set = kvp
            set.name = "badge_name";

            for (auto version : set.versions)
            {
                version.clickAction = BadgeVersion::Action::VisitUrl;
                version.clickUrl = "http://twitch.tv/really_awesome_url";
                version.description = "Descriptions";
                version.title = "Some Test Title";
                version.name = "Some Test Name";
                version.images.resize(10);

                for (auto& image : version.images)
                {
                    image.scale = 3.0f;
                    image.url = "http://twitch.tv/really_awesome_url";
                }
            }
        }
    }
} // Anonymyous namespace

#if (WIN32 && NDEBUG)
TEST_F(ChatBaseTest, BitsConfigurationMemory)
#else
TEST_F(ChatBaseTest, DISABLED_BitsConfigurationMemory)
#endif
{
    int32_t initialMemoryUsage = GetProcessMemoryUsage();

    for (int32_t i = 0; i < kNumIterations; i++)
    {
        std::vector<BitsConfiguration::ImageTier> imageTiers;
        std::vector<float> imageScales;
        std::vector<std::string> imageBackgrounds;
        std::vector<std::string> imageAnimationStates;

        imageTiers.resize(5);
        imageScales.resize(5);
        imageBackgrounds.resize(5);
        imageAnimationStates.resize(5);

        for (int32_t i = 0; i < 5; i++)
        {
            BitsConfiguration::ImageTier imageTier;
            std::string imageBackground;
            std::string imageAnimationState;

            imageTier.minBits = 10;
            imageTier.color = 1111;

            imageTiers[i] = imageTier;
            imageScales[i] = 1.0f;
            imageBackgrounds[i] = "imageBackground";
            imageAnimationStates[i] = "imageAnimationStates";
        }

        BitsConfiguration::Action action;
        action.prefix = "cheer";
        action.tiers = imageTiers;

        BitsConfiguration bitConfig;
        bitConfig.actions = {action};
    }

    ASSERT_TRUE(IsMemoryWithinAllowance(initialMemoryUsage, kLeakAllowanceBytes));
}

#if (WIN32 && NDEBUG)
TEST_F(ChatBaseTest, EmoticonSetMemory)
#else
TEST_F(ChatBaseTest, DISABLED_EmoticonSetMemory)
#endif
{
    int32_t initialMemoryUsage = GetProcessMemoryUsage();

    for (int32_t i = 0; i < kNumIterations; i++)
    {
        {
            std::string setId;
            std::string id;
            std::string str;
            std::string regexString;
            std::regex regex;

            setId = 1111;
            id = 1111;
            str = "string";
            regexString = "regex";
            regex = regexString;

            std::vector<StringEmoticon> stringEmoticons;
            std::vector<RegexEmoticon> regexEmoticons;

            stringEmoticons.resize(5);
            regexEmoticons.resize(5);

            for (int32_t i = 0; i < 5; i++)
            {
                StringEmoticon& strImgEntry = stringEmoticons[i];
                strImgEntry.emoticonId = id;
                strImgEntry.match = str;

                RegexEmoticon& regexImgEntry = regexEmoticons[i];
                regexImgEntry.emoticonId = id;
                regexImgEntry.regex = regex;
                regexImgEntry.regexString = regexString;
            }

            EmoticonSet emoticonSet;
            emoticonSet.emoticonSetId = setId;
            emoticonSet.stringEmoticons = stringEmoticons;
            emoticonSet.regexEmoticons = regexEmoticons;
        }

        {
            std::string setId;
            setId = 1111;

            EmoticonSet emoticonSet;
            emoticonSet.emoticonSetId = setId;
        }
    }

    ASSERT_TRUE(IsMemoryWithinAllowance(initialMemoryUsage, kLeakAllowanceBytes));
}

#if (WIN32 && NDEBUG)
TEST_F(ChatBaseTest, GlobalBadgesMemory)
#else
TEST_F(ChatBaseTest, DISABLED_GlobalBadgesMemory)
#endif
{
    int32_t initialMemoryUsage = GetProcessMemoryUsage();

    for (int32_t i = 0; i < kNumIterations; ++i)
    {
        BadgeSets src;
        AllocateBadgeSets(src);
    }

    ASSERT_TRUE(IsMemoryWithinAllowance(initialMemoryUsage, kLeakAllowanceBytes));
}

#if (WIN32 && NDEBUG)
TEST_F(ChatBaseTest, ChannelBadgesMemory)
#else
TEST_F(ChatBaseTest, DISABLED_ChannelBadgesMemory)
#endif
{
    int32_t initialMemoryUsage = GetProcessMemoryUsage();

    for (int32_t i = 0; i < kNumIterations; ++i)
    {
        BadgeSets src;
        AllocateBadgeSets(src);
    }

    ASSERT_TRUE(IsMemoryWithinAllowance(initialMemoryUsage, kLeakAllowanceBytes));
}

#if (WIN32 && NDEBUG)
TEST_F(ChatBaseTest, ThreadMessageListMemory)
#else
TEST_F(ChatBaseTest, DISABLED_ThreadMessageListMemory)
#endif
{
    int32_t initialMemoryUsage = GetProcessMemoryUsage();

    for (int32_t i = 0; i < kNumIterations; ++i)
    {
        {
            std::vector<Message> messages;
            AllocateMessageVector(messages, 4);
        }
        {
            std::vector<Message> messages;
            AllocateMessageVector(messages, 4);
        }
        {
            Message msg;
            AllocateMessage(msg);
        }
        {
            std::vector<std::unique_ptr<MessageToken>> tokenList;
            tokenList.resize(4);

            tokenList[0] = std::make_unique<TextToken>("text");
            tokenList[1] = std::make_unique<EmoticonToken>("emoteText", 42);
            tokenList[2] = std::make_unique<MentionToken>("mention", 1234);
            tokenList[3] = std::make_unique<UrlToken>("url", false);

            Message msg;
            msg.threadId = "threadId";
            msg.userName = "username";
            msg.tokens = std::move(tokenList);

            std::vector<Message> messages;
            messages.push_back(msg);
            messages.push_back(msg);
            messages.push_back(msg);
            messages.push_back(msg);
        }
    }

    ASSERT_TRUE(IsMemoryWithinAllowance(initialMemoryUsage, kLeakAllowanceBytes));
}

#if (WIN32 && NDEBUG)
TEST_F(ChatBaseTest, StandardMessageMemory)
#else
TEST_F(ChatBaseTest, DISABLED_StandardMessageMemory)
#endif
{
    int32_t initialMemoryUsage = GetProcessMemoryUsage();

    for (int32_t i = 0; i < kNumIterations; ++i)
    {
        {
            Message msg;
            AllocateMessage(msg);
        }
        {
            std::vector<std::unique_ptr<MessageToken>> tokenList;
            tokenList.resize(4);
            tokenList[0] = std::make_unique<TextToken>("text");
            tokenList[1] = std::make_unique<EmoticonToken>("emotetext", 42);
            tokenList[2] = std::make_unique<MentionToken>("mention", 1234);
            tokenList[3] = std::make_unique<UrlToken>("url", false);

            Message msg;
            msg.threadId = "threadId";
            msg.userName = "username";
            msg.tokens = std::move(tokenList);
        }
    }

    ASSERT_TRUE(IsMemoryWithinAllowance(initialMemoryUsage, kLeakAllowanceBytes));
}

#if (WIN32 && NDEBUG)
TEST_F(ChatBaseTest, ThreadMemory)
#else
TEST_F(ChatBaseTest, DISABLED_ThreadMemory)
#endif
{
    int32_t initialMemoryUsage = GetProcessMemoryUsage();

    for (int32_t i = 0; i < kNumIterations; ++i)
    {
        ThreadData data;
        AllocateThreadData(data);
    }

    ASSERT_TRUE(IsMemoryWithinAllowance(initialMemoryUsage, kLeakAllowanceBytes));
}

#endif
