/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/chatapitest.h"
#include "twitchsdk/chat/chatmessagehandler.h"

using namespace ttv::chat;
using namespace ttv::chat::test;

namespace {
class TestChatHandlerCallbacks : public ChatMessageHandler::ICallbacks {
 public:
  virtual bool PassThrough(const std::string& message) override {
    mPassThroughCalled = true;
    mResultMessage = message;
    return true;
  }

  virtual bool BlockUser(const std::string& blockUserName) override {
    mPassThroughCalled = false;
    mResultUserName = blockUserName;
    return true;
  }

  virtual bool UnblockUser(const std::string& unblockUserName) override {
    mPassThroughCalled = false;
    mResultUserName = unblockUserName;
    return true;
  }

  virtual bool WhisperUser(const std::string& whisperUserName, const std::string& message) override {
    mPassThroughCalled = false;
    mResultUserName = whisperUserName;
    mResultMessage = message;
    return true;
  }

  virtual bool CreateRaid(const std::string& raidTargetUser) override {
    mPassThroughCalled = false;
    mResultUserName = raidTargetUser;
    return true;
  }

  virtual bool CancelRaid() override {
    mPassThroughCalled = false;
    mResultMessage = "cancelled raid";
    return true;
  }

  virtual bool GrantVIP(const std::string& vipUserName) override {
    mPassThroughCalled = false;
    mResultUserName = vipUserName;
    return true;
  }

  virtual bool RevokeVIP(const std::string& unvipUserName) override {
    mPassThroughCalled = false;
    mResultUserName = unvipUserName;
    return true;
  }

  virtual bool ListVIPs() override {
    mPassThroughCalled = false;
    mResultMessage = "list vips";
    return true;
  }

  std::string GetResultMessage() const { return mResultMessage; }

  std::string GetResultUserName() const { return mResultUserName; }

  bool GetPassThroughCalled() const { return mPassThroughCalled; }

 private:
  std::string mResultMessage;
  std::string mResultUserName;
  bool mPassThroughCalled;
};
}  // namespace

TEST_F(ChatBaseTest, ChatHandlerBlockUnblockTest) {
  std::shared_ptr<TestChatHandlerCallbacks> callbacks = std::make_shared<TestChatHandlerCallbacks>();
  ChatMessageHandler handler;
  handler.SetCallbacks(static_cast<std::shared_ptr<ChatMessageHandler::ICallbacks>>(callbacks));

  std::vector<std::string> commandStrings = {
    "/ignore ", "/IgNoRe ", " /ignore ", "/block ", "/BLOCK ", "/unignore ", "/UnIgNoRe ", " /unignore "};
  std::vector<std::string> validParsing = {"user_name", " user_name ", "user_name extra strings should be ignored"};
  std::vector<std::string> invalidParsing = {"", " "};

  for (const std::string& commandString : commandStrings) {
    for (const std::string& messageString : validParsing) {
      ASSERT_TRUE(handler.HandleMessage(commandString + messageString));
      ASSERT_FALSE(callbacks->GetPassThroughCalled());
      ASSERT_EQ(callbacks->GetResultUserName(), "user_name");
    }

    for (const std::string& messageString : invalidParsing) {
      ASSERT_TRUE(handler.HandleMessage(commandString + messageString));
      ASSERT_TRUE(callbacks->GetPassThroughCalled());
      ASSERT_EQ(callbacks->GetResultMessage(), commandString + messageString);
    }
  }
}

TEST_F(ChatBaseTest, ChatHandlerWhisperTest) {
  std::shared_ptr<TestChatHandlerCallbacks> callbacks = std::make_shared<TestChatHandlerCallbacks>();
  ChatMessageHandler handler;
  handler.SetCallbacks(static_cast<std::shared_ptr<ChatMessageHandler::ICallbacks>>(callbacks));

  std::string whisperString = "/w user_name ";

  ASSERT_TRUE(handler.HandleMessage(whisperString + "message to user"));
  ASSERT_FALSE(callbacks->GetPassThroughCalled());
  ASSERT_EQ(callbacks->GetResultUserName(), "user_name");
  ASSERT_EQ(callbacks->GetResultMessage(), "message to user");

  ASSERT_TRUE(handler.HandleMessage(whisperString + "   "));
  ASSERT_TRUE(callbacks->GetPassThroughCalled());
  ASSERT_EQ(callbacks->GetResultMessage(), whisperString + "   ");

  ASSERT_TRUE(handler.HandleMessage(whisperString));
  ASSERT_TRUE(callbacks->GetPassThroughCalled());
  ASSERT_EQ(callbacks->GetResultMessage(), whisperString);
}

TEST_F(ChatBaseTest, ChatHandlerRaidsTest) {
  std::shared_ptr<TestChatHandlerCallbacks> callbacks = std::make_shared<TestChatHandlerCallbacks>();
  ChatMessageHandler handler;
  handler.SetCallbacks(static_cast<std::shared_ptr<ChatMessageHandler::ICallbacks>>(callbacks));

  std::string raidString = "/raid user_name ";

  ASSERT_TRUE(handler.HandleMessage(raidString));
  ASSERT_FALSE(callbacks->GetPassThroughCalled());
  ASSERT_EQ(callbacks->GetResultUserName(), "user_name");

  std::string unraidString = "/unraid";

  ASSERT_TRUE(handler.HandleMessage(unraidString));
  ASSERT_FALSE(callbacks->GetPassThroughCalled());
  ASSERT_EQ(callbacks->GetResultMessage(), "cancelled raid");
}

TEST_F(ChatBaseTest, ChatHandlerVipTest) {
  std::shared_ptr<TestChatHandlerCallbacks> callbacks = std::make_shared<TestChatHandlerCallbacks>();
  ChatMessageHandler handler;
  handler.SetCallbacks(static_cast<std::shared_ptr<ChatMessageHandler::ICallbacks>>(callbacks));

  std::string vipString = "/vip user_name";

  ASSERT_TRUE(handler.HandleMessage(vipString));
  ASSERT_FALSE(callbacks->GetPassThroughCalled());
  ASSERT_EQ(callbacks->GetResultUserName(), "user_name");

  vipString = "/unvip user_name";

  ASSERT_TRUE(handler.HandleMessage(vipString));
  ASSERT_FALSE(callbacks->GetPassThroughCalled());
  ASSERT_EQ(callbacks->GetResultUserName(), "user_name");

  vipString = "/vips";

  ASSERT_TRUE(handler.HandleMessage(vipString));
  ASSERT_FALSE(callbacks->GetPassThroughCalled());
  ASSERT_EQ(callbacks->GetResultMessage(), "list vips");
}

TEST_F(ChatBaseTest, ChatHandlerPassThroughTest) {
  std::shared_ptr<TestChatHandlerCallbacks> callbacks = std::make_shared<TestChatHandlerCallbacks>();
  ChatMessageHandler handler;
  handler.SetCallbacks(static_cast<std::shared_ptr<ChatMessageHandler::ICallbacks>>(callbacks));

  std::vector<std::string> messageStrings = {
    "normal message", "/notarealcommand test", "", "  ", "   message   ", "/ignor command too short"};

  for (const std::string& messageString : messageStrings) {
    ASSERT_TRUE(handler.HandleMessage(messageString));
    ASSERT_TRUE(callbacks->GetPassThroughCalled());
    ASSERT_EQ(callbacks->GetResultMessage(), messageString);
  }
}
