/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/chatmessagetest.h"
#include "testchatobjectfactory.h"
#include "twitchsdk/chat/chatapi.h"
#include "twitchsdk/chat/ichatobjectfactory.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/thread.h"

#include <sstream>

#include "gtest/gtest.h"

using namespace ttv;
using namespace ttv::chat;
using namespace ttv::chat::test;
using namespace ttv::test;

TEST_F(ChatMessageTest, SubscriptionNotice) {
  // Generate the message
  std::string message =
    "@badges=staff/1,broadcaster/1,turbo/1;color=#008000;display-name=Test;emotes=;mod=0;msg-id=resub;msg-param-cumulative-months=10;msg-param-months=0;msg-param-should-share-streak=1;msg-param-streak-months=2;msg-param-sub-plan=Prime;msg-param-sub-plan-name=Prime;room-id=1001;subscriber=1;system-msg=test\\ssubscribed\\swith\\sTwitch\\sPrime.\\sThey’ve\\ssubscribed\\sfor\\s10\\smonths,\\scurrently\\son\\sa\\s2\\smonth\\sstreak!;login=test;turbo=1;user-id=9001;user-type=staff :tmi.twitch.tv USERNOTICE #test :Great stream -- keep it up!\r\n";

  // Set up callback
  bool receivedCallback = false;
  auto subscriptionNoticeFunc = [&receivedCallback](
                                  UserId userId, ChannelId channelId, const SubscriptionNotice& notice) {
    ASSERT_EQ(userId, 9001);
    ASSERT_EQ(channelId, 1001);
    ASSERT_EQ(notice.userMessage->userName, "test");
    ASSERT_EQ(notice.userMessage->displayName, "Test");
    ASSERT_EQ(notice.userMessage->tokens.size(), 1);
    ASSERT_EQ(notice.userMessage->tokens[0]->GetType(), MessageToken::Type::Text);
    TextToken* textToken = static_cast<TextToken*>(notice.userMessage->tokens[0].get());
    ASSERT_EQ(textToken->text, "Great stream -- keep it up!");
    ASSERT_EQ(notice.userMessage->badges.size(), 3);
    ASSERT_EQ(notice.userMessage->badges[0].name, "staff");
    ASSERT_EQ(notice.userMessage->badges[0].version, "1");
    ASSERT_EQ(notice.userMessage->badges[1].name, "broadcaster");
    ASSERT_EQ(notice.userMessage->badges[1].version, "1");
    ASSERT_EQ(notice.userMessage->badges[2].name, "turbo");
    ASSERT_EQ(notice.userMessage->badges[2].version, "1");

    ASSERT_EQ(notice.userMessage->userId, 9001);
    ASSERT_EQ(notice.systemMessage,
      "test subscribed with Twitch Prime. They’ve subscribed for 10 months, currently on a 2 month streak!");
    ASSERT_EQ(notice.planDisplayName, "Prime");
    ASSERT_EQ(notice.subCumulativeMonthCount, 10);
    ASSERT_EQ(notice.subStreakMonthCount, 2);
    ASSERT_TRUE(notice.shouldShowSubStreak);

    ASSERT_EQ(notice.type, SubscriptionNotice::Type::Resub);
    ASSERT_EQ(notice.plan, SubscriptionNotice::Plan::Prime);

    ASSERT_EQ(notice.recipient.userName, "");
    ASSERT_EQ(notice.recipient.displayName, "");
    ASSERT_EQ(notice.recipient.userId, 0);

    receivedCallback = true;
  };

  mChatChannelListenerProxy->chatChannelSubscriptionNoticeReceivedCallback = subscriptionNoticeFunc;

  SendMessage(message);

  ASSERT_TRUE(WaitUntilResult(1000, [&receivedCallback]() { return receivedCallback; }));
}

TEST_F(ChatMessageTest, SubGiftNotice) {
  // Generate the message
  std::string message =
    "@badges=staff/1,bits/1;color=#8A2CE2;display-name=Test;emotes=;id=d5f235d0-1942-4d0f-a3cd-f5ec1f933c83;login=test;mod=0;msg-id=subgift;msg-param-months=1;msg-param-recipient-display-name=Recipient;msg-param-recipient-id=1002;msg-param-recipient-user-name=recipient;msg-param-sender-count=1;msg-param-sub-plan-name=Channel\\sSubscription\\s(test);msg-param-sub-plan=1000;room-id=1001;subscriber=0;system-msg=test\\sgifted\\sa\\s$4.99\\ssub\\sto\\srecipient!;tmi-sent-ts=1518116164376;turbo=0;user-id=9001;user-type=staff :tmi.twitch.tv USERNOTICE #test\r\n";
  // Set up callback
  bool receivedCallback = false;
  auto subscriptionNoticeFunc = [&receivedCallback](
                                  UserId userId, ChannelId channelId, const SubscriptionNotice& notice) {
    ASSERT_EQ(userId, 9001);
    ASSERT_EQ(channelId, 1001);
    ASSERT_EQ(notice.userMessage->userName, "test");
    ASSERT_EQ(notice.userMessage->displayName, "Test");
    ASSERT_EQ(notice.userMessage->tokens.size(), 0);
    ASSERT_EQ(notice.userMessage->badges.size(), 2);
    ASSERT_EQ(notice.userMessage->badges[0].name, "staff");
    ASSERT_EQ(notice.userMessage->badges[0].version, "1");
    ASSERT_EQ(notice.userMessage->badges[1].name, "bits");
    ASSERT_EQ(notice.userMessage->badges[1].version, "1");

    ASSERT_EQ(notice.userMessage->userId, 9001);
    ASSERT_EQ(notice.systemMessage, "test gifted a $4.99 sub to recipient!");
    ASSERT_EQ(notice.planDisplayName, "Channel Subscription (test)");
    ASSERT_EQ(notice.messageId, "d5f235d0-1942-4d0f-a3cd-f5ec1f933c83");

    ASSERT_EQ(notice.type, SubscriptionNotice::Type::SubGift);
    ASSERT_EQ(notice.plan, SubscriptionNotice::Plan::Sub1000);

    ASSERT_EQ(notice.recipient.userName, "recipient");
    ASSERT_EQ(notice.recipient.displayName, "Recipient");
    ASSERT_EQ(notice.recipient.userId, 1002);

    ASSERT_EQ(notice.senderCount, 1);

    receivedCallback = true;
  };

  mChatChannelListenerProxy->chatChannelSubscriptionNoticeReceivedCallback = subscriptionNoticeFunc;

  SendMessage(message);

  ASSERT_TRUE(WaitUntilResult(1000, [&receivedCallback]() { return receivedCallback; }));
}

TEST_F(ChatMessageTest, ExtendSubNotice) {
  // Generate the message
  std::string message =
    "@badges=staff/1,bits/1;color=#8A2CE2;display-name=Test;emotes=;id=d5f235d0-1942-4d0f-a3cd-f5ec1f933c83;login=test;mod=0;"
    "msg-id=extendsub;msg-param-sub-plan=2000;msg-param-cumulative-months=3;msg-param-sub-benefit-end-month=1;"
    "subscriber=0;system-msg=test\\sgifted\\sa\\s$4.99\\ssub\\sto\\srecipient!;tmi-sent-ts=1518116164376;turbo=0;user-id=9001;user-type=staff :tmi.twitch.tv USERNOTICE #test\r\n";

  // Set up callback
  bool receivedCallback = false;
  auto subscriptionNoticeFunc = [&receivedCallback](
                                  UserId userId, ChannelId channelId, const SubscriptionNotice& notice) {
    ASSERT_EQ(userId, 9001);
    ASSERT_EQ(channelId, 1001);
    ASSERT_EQ(notice.userMessage->userName, "test");
    ASSERT_EQ(notice.userMessage->displayName, "Test");
    ASSERT_EQ(notice.userMessage->tokens.size(), 0);
    ASSERT_EQ(notice.userMessage->badges.size(), 2);
    ASSERT_EQ(notice.userMessage->badges[0].name, "staff");
    ASSERT_EQ(notice.userMessage->badges[0].version, "1");
    ASSERT_EQ(notice.userMessage->badges[1].name, "bits");
    ASSERT_EQ(notice.userMessage->badges[1].version, "1");

    ASSERT_EQ(notice.userMessage->userId, 9001);
    ASSERT_EQ(notice.systemMessage, "test gifted a $4.99 sub to recipient!");
    ASSERT_EQ(notice.messageId, "d5f235d0-1942-4d0f-a3cd-f5ec1f933c83");

    ASSERT_EQ(notice.type, SubscriptionNotice::Type::ExtendSub);
    ASSERT_EQ(notice.benefitEndMonth, 1);
    ASSERT_EQ(notice.plan, SubscriptionNotice::Plan::Sub2000);
    ASSERT_EQ(notice.subCumulativeMonthCount, 3);

    receivedCallback = true;
  };

  mChatChannelListenerProxy->chatChannelSubscriptionNoticeReceivedCallback = subscriptionNoticeFunc;

  SendMessage(message);

  ASSERT_TRUE(WaitUntilResult(1000, [&receivedCallback]() { return receivedCallback; }));
}

TEST_F(ChatMessageTest, FirstChatterNotice) {
  // Generate the message
  std::string message =
    "@badges=;color=;display-name=Test;emotes=30259:0-6;id=37feed0f-b9c7-4c3a-b475-21c6c6d21c3d;login=test;mod=0;msg-id=ritual;msg-param-ritual-name=new_chatter;room-id=1001;subscriber=0;system-msg=Somebody\\sis\\snew\\shere!;tmi-sent-ts=1508363903;turbo=0;user-id=9001;user-type= :tmi.twitch.tv USERNOTICE #test :HeyGuys\r\n";

  // Set up callback
  bool receivedCallback = false;
  auto firstTimeChatterFunc = [&receivedCallback](
                                UserId userId, ChannelId channelId, const FirstTimeChatterNotice& notice) {
    ASSERT_EQ(userId, 9001);
    ASSERT_EQ(channelId, 1001);
    ASSERT_EQ(notice.userMessage.userName, "test");
    ASSERT_EQ(notice.userMessage.displayName, "Test");
    ASSERT_EQ(notice.userMessage.tokens.size(), 1);
    ASSERT_EQ(notice.userMessage.tokens[0]->GetType(), MessageToken::Type::Emoticon);
    EmoticonToken* emoticonToken = static_cast<EmoticonToken*>(notice.userMessage.tokens[0].get());
    ASSERT_EQ(emoticonToken->emoticonText, "HeyGuys");
    ASSERT_EQ(emoticonToken->emoticonId, "30259");
    ASSERT_EQ(notice.userMessage.badges.size(), 0);
    ASSERT_EQ(notice.systemMessage, "Somebody is new here!");
    ASSERT_EQ(notice.messageId, "37feed0f-b9c7-4c3a-b475-21c6c6d21c3d");

    receivedCallback = true;
  };

  mChatChannelListenerProxy->chatChannelFirstTimeChatterNoticeReceivedCallback = firstTimeChatterFunc;

  SendMessage(message);

  ASSERT_TRUE(WaitUntilResult(1000, [&receivedCallback]() { return receivedCallback; }));
}

TEST_F(ChatMessageTest, FirstChatterNoticeModifiedEmote) {
  // Generate the message
  std::string message =
    "@badges=;color=;display-name=Test;emotes=30259_BW:0-8;id=37feed0f-b9c7-4c3a-b475-21c6c6d21c3d;login=test;mod=0;msg-id=ritual;msg-param-ritual-name=new_chatter;room-id=1001;subscriber=0;system-msg=Somebody\\sis\\snew\\shere!;tmi-sent-ts=1508363903;turbo=0;user-id=9001;user-type= :tmi.twitch.tv USERNOTICE #test :HeyGuysBW\r\n";

  // Set up callback
  bool receivedCallback = false;
  auto firstTimeChatterFunc = [&receivedCallback](
                                UserId userId, ChannelId channelId, const FirstTimeChatterNotice& notice) {
    ASSERT_EQ(userId, 9001);
    ASSERT_EQ(channelId, 1001);
    ASSERT_EQ(notice.userMessage.userName, "test");
    ASSERT_EQ(notice.userMessage.displayName, "Test");
    ASSERT_EQ(notice.userMessage.tokens.size(), 1);
    ASSERT_EQ(notice.userMessage.tokens[0]->GetType(), MessageToken::Type::Emoticon);
    EmoticonToken* emoticonToken = static_cast<EmoticonToken*>(notice.userMessage.tokens[0].get());
    ASSERT_EQ(emoticonToken->emoticonText, "HeyGuysBW");
    ASSERT_EQ(emoticonToken->emoticonId, "30259_BW");
    ASSERT_EQ(notice.userMessage.badges.size(), 0);
    ASSERT_EQ(notice.systemMessage, "Somebody is new here!");
    ASSERT_EQ(notice.messageId, "37feed0f-b9c7-4c3a-b475-21c6c6d21c3d");

    receivedCallback = true;
  };

  mChatChannelListenerProxy->chatChannelFirstTimeChatterNoticeReceivedCallback = firstTimeChatterFunc;

  SendMessage(message);

  ASSERT_TRUE(WaitUntilResult(1000, [&receivedCallback]() { return receivedCallback; }));
}

TEST_F(ChatMessageTest, RaidNotice) {
  // Generate the message
  std::string message =
    "@badges=staff/1;color=;display-name=Test;emotes=;id=d293de02-fdf7-4f1c-80f3-ef4ff443b43e;login=test;mod=0;msg-id=raid;msg-param-displayName=Test;msg-param-login=test;msg-param-viewerCount=1;room-id=1001;subscriber=0;system-msg=1\\sraiders\\sfrom\\stest\\shave\\sjoined!;tmi-sent-ts=1505174301026;turbo=0;user-id=9001;user-type= :tmi.twitch.tv USERNOTICE #test\r\n";

  // Set up callback
  bool receivedCallback = false;
  auto raidNoticeFunc = [&receivedCallback](UserId userId, ChannelId channelId, const RaidNotice& notice) {
    ASSERT_EQ(userId, 9001);
    ASSERT_EQ(channelId, 1001);
    ASSERT_EQ(notice.systemMessage, "1 raiders from test have joined!");
    ASSERT_EQ(notice.raidingUserInfo.userName, "test");
    ASSERT_EQ(notice.raidingUserInfo.displayName, "Test");
    ASSERT_EQ(notice.raidingUserInfo.userId, 9001);
    ASSERT_EQ(notice.raidingUserInfo.userName, "test");
    ASSERT_EQ(notice.viewerCount, 1);

    receivedCallback = true;
  };

  mChatChannelListenerProxy->chatChannelRaidNoticeReceivedCallback = raidNoticeFunc;

  SendMessage(message);

  ASSERT_TRUE(WaitUntilResult(1000, [&receivedCallback]() { return receivedCallback; }));
}

TEST_F(ChatMessageTest, GenericMessageNotice) {
  // Generate the message
  std::string message =
    "@badges=;color=;display-name=Test;emotes=30259:0-6/30259_BW:7-15;id=37feed0f-b9c7-4c3a-b475-21c6c6d21c3d;login=test;mod=0;msg-id=GENERIC;msg-param-ritual-name=new_chatter;room-id=1001;subscriber=0;system-msg=Somebody\\sis\\snew\\shere!;tmi-sent-ts=1508363903;turbo=0;user-id=9001;user-type= :tmi.twitch.tv USERNOTICE #test :HeyGuysHeyGuysBW\r\n";

  // Set up callback
  bool receivedCallback = false;
  auto genericMessageFunc = [&receivedCallback](
                              UserId userId, ChannelId channelId, const GenericMessageNotice& notice) {
    ASSERT_EQ(userId, 9001);
    ASSERT_EQ(channelId, 1001);
    ASSERT_EQ(notice.messageInfo.userName, "test");
    ASSERT_EQ(notice.messageInfo.displayName, "Test");
    ASSERT_EQ(notice.messageInfo.tokens.size(), 2);
    ASSERT_EQ(notice.messageInfo.tokens[0]->GetType(), MessageToken::Type::Emoticon);
    EmoticonToken* emoticonToken = static_cast<EmoticonToken*>(notice.messageInfo.tokens[0].get());
    ASSERT_EQ(emoticonToken->emoticonText, "HeyGuys");
    ASSERT_EQ(emoticonToken->emoticonId, "30259");
    ASSERT_EQ(notice.messageInfo.tokens[1]->GetType(), MessageToken::Type::Emoticon);
    emoticonToken = static_cast<EmoticonToken*>(notice.messageInfo.tokens[1].get());
    ASSERT_EQ(emoticonToken->emoticonText, "HeyGuysBW");
    ASSERT_EQ(emoticonToken->emoticonId, "30259_BW");
    ASSERT_EQ(notice.messageInfo.badges.size(), 0);
    ASSERT_EQ(notice.messageId, "37feed0f-b9c7-4c3a-b475-21c6c6d21c3d");
    ASSERT_EQ(notice.messageInfo.messageTags.size(), 16);
    ASSERT_EQ(notice.messageInfo.messageType, "GENERIC");

    receivedCallback = true;
  };

  mChatChannelListenerProxy->chatChannelGenericNoticeReceivedCallback = genericMessageFunc;

  SendMessage(message);

  ASSERT_TRUE(WaitUntilResult(1000, [&receivedCallback]() { return receivedCallback; }));
}
