#include "chattestmanager.h"
#include "fixtures/chatapitest.h"
#include "testsystemclock.h"
#include "testutilities.h"
#include "twitchsdk/chat/chatapi.h"
#include "twitchsdk/chat/ichatraid.h"
#include "twitchsdk/chat/ichatraidlistener.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/socket.h"
#include "twitchsdk/core/stringutilities.h"

#include "gtest/gtest.h"

using namespace ttv;
using namespace ttv::chat;
using namespace ttv::chat::test;
using namespace ttv::test;

namespace {
const UserId kTwitchUserId = 12826;            // The userid for the company account
const UserId kSourceBroadcasterUserId = 2222;  // The userid of a broadcaster
const UserId kSomeViewerUserId = 1111;         // The userid for some random viewer without mod access
const UserId kTargetChannelId = 1234;
const std::string kTargetUserLogin = "login";
const std::string kTargetUserDisplayName = "display";
const std::string kTargetUserProfileImage = "www.profileimage.com";
const std::string kDefaultRaidId = "aabbccdd";

/**
 * A helper IChatRaidListener implementation that proxies events into optional lambda calls.
 */
class TestChatRaidListener : public ttv::chat::IChatRaidListener {
 public:
  using RaidStartedFunc = std::function<void(const RaidStatus& status)>;
  using RaidUpdatedFunc = std::function<void(const RaidStatus& status)>;
  using RaidFiredFunc = std::function<void(const RaidStatus& status)>;
  using RaidCancelledFunc = std::function<void(const RaidStatus& status)>;

 public:
  TestChatRaidListener() : startedCalled(false), updatedCalled(false), firedCalled(false), cancelledCalled(false) {}

  virtual void RaidStarted(const RaidStatus& status) override {
    startedCalled = true;
    startedStatus = status;

    if (raidStartedFunc != nullptr) {
      raidStartedFunc(status);
    }
  }

  virtual void RaidUpdated(const RaidStatus& status) override {
    updatedCalled = true;
    updatedStatus = status;

    if (raidUpdatedFunc != nullptr) {
      raidUpdatedFunc(status);
    }
  }

  virtual void RaidFired(const RaidStatus& status) override {
    firedCalled = true;
    firedStatus = status;

    if (raidFiredFunc != nullptr) {
      raidFiredFunc(status);
    }
  }

  virtual void RaidCancelled(const RaidStatus& status) override {
    cancelledCalled = true;
    cancelledStatus = status;

    if (raidCancelledFunc != nullptr) {
      raidCancelledFunc(status);
    }
  }

  void ClearCallbackData() {
    startedCalled = false;
    updatedCalled = false;
    firedCalled = false;
    cancelledCalled = false;

    startedStatus = {};
    updatedStatus = {};
    firedStatus = {};
    cancelledStatus = {};
  }

  bool startedCalled;
  bool updatedCalled;
  bool firedCalled;
  bool cancelledCalled;

  RaidStatus startedStatus;
  RaidStatus updatedStatus;
  RaidStatus firedStatus;
  RaidStatus cancelledStatus;

  RaidStartedFunc raidStartedFunc;
  RaidUpdatedFunc raidUpdatedFunc;
  RaidFiredFunc raidFiredFunc;
  RaidCancelledFunc raidCancelledFunc;
};

class ChatRaidTest : public ChatApiTest {
 public:
  ChatRaidTest() : mJoined(true) {}

 protected:
  void BuildRaidStatusPubSubMessage(const std::string& msgType, const RaidStatus& status, ttv::json::Value& result) {
    result["type"] = msgType;
    result["raid"] = json::Value(json::objectValue);
    result["raid"]["id"] = status.raidId;
    result["raid"]["creator_id"] = std::to_string(status.creatorUserId);
    result["raid"]["source_id"] = std::to_string(status.sourceChannelId);
    result["raid"]["target_id"] = std::to_string(status.targetChannelId);
    result["raid"]["target_login"] = status.targetUserLogin;
    result["raid"]["target_display_name"] = status.targetUserDisplayName;
    result["raid"]["target_profile_image"] = status.targetUserProfileImageUrl;
    result["raid"]["viewer_count"] = status.numUsersInRaid;
    result["raid"]["transition_jitter_seconds"] = status.transitionJitterSeconds;
    result["raid"]["force_raid_now_seconds"] = status.forceRaidNowSeconds;
  }

  void LogIn(UserId userId) {
    TestParams testParams;
    testParams.userInfo.userId = userId;
    testParams.userInfo.userName = "valid_login";
    testParams.userInfo.displayName = "Display Name";
    testParams.oauth = "valid_oauth";

    TTV_ErrorCode ec = ChatApiTest::LogIn(testParams);
    EXPECT_EQ(TTV_EC_SUCCESS, ec);
  }

  void CreateRaid(UserId userId) {
    mListener = std::make_shared<TestChatRaidListener>();
    ASSERT_TRUE(TTV_SUCCEEDED(mChatApi->CreateChatRaid(userId, kSourceBroadcasterUserId, mListener, mRaid)));
  }

  void TestStarted(RaidStatus& expectedStatus) {
    auto waitForStartedCallbackFunc = [listener = mListener]() { return listener->startedCalled; };

    json::Value msg;
    BuildRaidStatusPubSubMessage("raid_update_v2", expectedStatus, msg);
    mPubSubTestUtility.PushPubSubMessage("raid." + std::to_string(kSourceBroadcasterUserId), msg);

    // Wait for started callback
    ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForStartedCallbackFunc, GetDefaultUpdateFunc()));
    ASSERT_TRUE(!mListener->updatedCalled);
    ASSERT_TRUE(!mListener->firedCalled);
    ASSERT_TRUE(!mListener->cancelledCalled);
    ASSERT_EQ(mListener->startedStatus, expectedStatus);
  }

  void TestUpdates(RaidStatus& expectedStatus, uint32_t numUpdates) {
    auto waitForUpdatedCallbackFunc = [listener = mListener]() { return listener->updatedCalled; };

    mListener->ClearCallbackData();

    expectedStatus.numUsersInRaid += 1;
    expectedStatus.joined = mJoined;

    json::Value msg;
    BuildRaidStatusPubSubMessage("raid_update_v2", expectedStatus, msg);
    mPubSubTestUtility.PushPubSubMessage("raid." + std::to_string(kSourceBroadcasterUserId), msg);

    // Wait for updated callback
    ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForUpdatedCallbackFunc, GetDefaultUpdateFunc()));
    ASSERT_TRUE(!mListener->startedCalled);
    ASSERT_TRUE(!mListener->firedCalled);
    ASSERT_TRUE(!mListener->cancelledCalled);
    ASSERT_EQ(mListener->updatedStatus, expectedStatus);
  }

  void TestCompleted(RaidStatus& expectedStatus) {
    auto waitForCompletedCallbackFunc = [listener = mListener]() { return listener->firedCalled; };

    mListener->ClearCallbackData();

    json::Value msg;
    BuildRaidStatusPubSubMessage("raid_go_v2", expectedStatus, msg);
    mPubSubTestUtility.PushPubSubMessage("raid." + std::to_string(kSourceBroadcasterUserId), msg);

    // Wait for completed callback
    ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCompletedCallbackFunc, GetDefaultUpdateFunc()));
    ASSERT_TRUE(!mListener->startedCalled);
    ASSERT_TRUE(!mListener->updatedCalled);
    ASSERT_TRUE(!mListener->cancelledCalled);
    ASSERT_EQ(mListener->firedStatus, expectedStatus);
  }

  void TestCancelled(RaidStatus& expectedStatus) {
    auto waitForCancelledCallbackFunc = [listener = mListener]() { return listener->cancelledCalled; };

    mListener->ClearCallbackData();

    json::Value msg;
    BuildRaidStatusPubSubMessage("raid_cancel_v2", expectedStatus, msg);
    mPubSubTestUtility.PushPubSubMessage("raid." + std::to_string(kSourceBroadcasterUserId), msg);

    // Wait for cancelled callback
    ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCancelledCallbackFunc, GetDefaultUpdateFunc()));
    ASSERT_TRUE(!mListener->startedCalled);
    ASSERT_TRUE(!mListener->updatedCalled);
    ASSERT_TRUE(!mListener->firedCalled);
    ASSERT_EQ(mListener->cancelledStatus, expectedStatus);
  }

 protected:
  std::shared_ptr<ttv::chat::IChatRaid> mRaid;
  std::shared_ptr<TestChatRaidListener> mListener;
  bool mJoined;
};
}  // namespace

TEST_F(ChatRaidTest, SuccessfulRaid) {
  TestChatApiInitializationCallback(TokenizationOptions::All());

  RaidStatus status;
  status.raidId = kDefaultRaidId;
  status.creatorUserId = kTwitchUserId;
  status.sourceChannelId = kSourceBroadcasterUserId;
  status.targetChannelId = kTargetChannelId;
  status.targetUserLogin = kTargetUserLogin;
  status.targetUserDisplayName = kTargetUserDisplayName;
  status.targetUserProfileImageUrl = kTargetUserProfileImage;
  status.transitionJitterSeconds = 1;
  status.numUsersInRaid = 1;
  status.forceRaidNowSeconds = 90;
  status.joined = mJoined;

  CreateRaid(0);
  TestStarted(status);
  TestUpdates(status, 5);
  TestCompleted(status);
}

TEST_F(ChatRaidTest, CancelledRaid) {
  TestChatApiInitializationCallback(TokenizationOptions::All());

  RaidStatus status;
  status.raidId = kDefaultRaidId;
  status.creatorUserId = kTwitchUserId;
  status.sourceChannelId = kSourceBroadcasterUserId;
  status.targetChannelId = kTargetChannelId;
  status.targetUserLogin = kTargetUserLogin;
  status.targetUserDisplayName = kTargetUserDisplayName;
  status.targetUserProfileImageUrl = kTargetUserProfileImage;
  status.transitionJitterSeconds = 1;
  status.numUsersInRaid = 1;
  status.forceRaidNowSeconds = 90;
  status.joined = mJoined;

  CreateRaid(0);
  TestStarted(status);
  TestUpdates(status, 3);
  TestCancelled(status);
}

// Test trying to use ChatRaid as an anonymous user
TEST_F(ChatRaidTest, AnonymousViewerMutations) {
  TestChatApiInitializationCallback(TokenizationOptions::All());

  // Calls to modify state should fail
  CreateRaid(0);
  ASSERT_TRUE(mRaid->Join(kDefaultRaidId, nullptr) == TTV_EC_NEED_TO_LOGIN);
  ASSERT_TRUE(mRaid->Leave(kDefaultRaidId, nullptr) == TTV_EC_NEED_TO_LOGIN);
  ASSERT_TRUE(mRaid->Start(kTargetChannelId, nullptr) == TTV_EC_NEED_TO_LOGIN);
  ASSERT_TRUE(mRaid->Cancel(nullptr) == TTV_EC_NEED_TO_LOGIN);
}

// Test trying to use ChatRaid as a logged in viewer
TEST_F(ChatRaidTest, LoggedInViewerMutations) {
  TestChatApiInitializationCallback(TokenizationOptions::All());

  mHttpRequest->AddResponse("https://api.twitch.tv/kraken/raids/" + kDefaultRaidId + "/join")
    .SetType(HTTP_POST_REQUEST)
    .SetStatusCode(204)
    .Done();
  mHttpRequest->AddResponse("https://api.twitch.tv/kraken/raids/" + kDefaultRaidId + "/leave")
    .SetType(HTTP_POST_REQUEST)
    .SetStatusCode(204)
    .Done();
  mHttpRequest
    ->AddResponse("https://api.twitch.tv/kraken/raids?source_id=" + std::to_string(kSourceBroadcasterUserId) +
                  "&target_id=" + std::to_string(kTargetChannelId))
    .SetType(HTTP_POST_REQUEST)
    .SetStatusCode(401)
    .Done();
  mHttpRequest->AddResponse("https://api.twitch.tv/kraken/raids?source_id=" + std::to_string(kSourceBroadcasterUserId))
    .SetType(HTTP_DELETE_REQUEST)
    .SetStatusCode(401)
    .Done();

  // Log a viewer in
  LogIn(kSomeViewerUserId);

  CreateRaid(kSomeViewerUserId);

  // Calls to Create/Cancel should fail
  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };
  TTV_ErrorCode ec = mRaid->Start(kTargetChannelId, [&callbackCalled](TTV_ErrorCode ec) {
    callbackCalled = true;
    ASSERT_TRUE(TTV_FAILED(ec));
  });
  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(waitForCallbackFunc());

  callbackCalled = false;
  ec = mRaid->Cancel([&callbackCalled](TTV_ErrorCode ec) {
    callbackCalled = true;
    ASSERT_TRUE(TTV_FAILED(ec));
  });
  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(waitForCallbackFunc());

  // Test a raid starting and the client joining and leaving
  RaidStatus status;
  status.raidId = kDefaultRaidId;
  status.creatorUserId = kTwitchUserId;
  status.sourceChannelId = kSourceBroadcasterUserId;
  status.targetChannelId = kTargetChannelId;
  status.targetUserLogin = kTargetUserLogin;
  status.targetUserDisplayName = kTargetUserDisplayName;
  status.targetUserProfileImageUrl = kTargetUserProfileImage;
  status.transitionJitterSeconds = 1;
  status.numUsersInRaid = 1;
  status.forceRaidNowSeconds = 90;
  status.joined = mJoined;

  TestStarted(status);
  TestUpdates(status, 3);

  // Calls to Join/Leave should succeed
  callbackCalled = false;
  ec = mRaid->Join(kDefaultRaidId, [this, &callbackCalled](TTV_ErrorCode ec) {
    callbackCalled = true;
    mJoined = TTV_SUCCEEDED(ec);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
  });
  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(waitForCallbackFunc());
  ASSERT_TRUE(mListener->updatedCalled);
  ASSERT_TRUE(mListener->updatedStatus.joined);
  mListener->ClearCallbackData();

  TestUpdates(status, 3);

  callbackCalled = false;
  ec = mRaid->Leave(kDefaultRaidId, [this, &callbackCalled](TTV_ErrorCode ec) {
    callbackCalled = true;
    mJoined = !TTV_SUCCEEDED(ec);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
  });
  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(waitForCallbackFunc());
  ASSERT_TRUE(mListener->updatedCalled);
  ASSERT_TRUE(!mListener->updatedStatus.joined);
  mListener->ClearCallbackData();

  TestUpdates(status, 3);
}

// Test trying to use ChatRaid as a logged in broadcaster
TEST_F(ChatRaidTest, LoggedInBroadcasterMutations) {
  TestChatApiInitializationCallback(TokenizationOptions::All());

  mHttpRequest->AddResponse("https://api.twitch.tv/kraken/raids/" + kDefaultRaidId + "/join")
    .SetType(HTTP_POST_REQUEST)
    .SetStatusCode(204)
    .Done();
  mHttpRequest->AddResponse("https://api.twitch.tv/kraken/raids/" + kDefaultRaidId + "/leave")
    .SetType(HTTP_POST_REQUEST)
    .SetStatusCode(204)
    .Done();
  mHttpRequest
    ->AddResponse("https://api.twitch.tv/kraken/raids?source_id=" + std::to_string(kSourceBroadcasterUserId) +
                  "&target_id=" + std::to_string(kTargetChannelId))
    .SetType(HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatraid_create_success.json")
    .Done();
  mHttpRequest->AddResponse("https://api.twitch.tv/kraken/raids?source_id=" + std::to_string(kSourceBroadcasterUserId))
    .SetType(HTTP_DELETE_REQUEST)
    .SetStatusCode(204)
    .Done();

  // Log a viewer in
  LogIn(kTwitchUserId);

  CreateRaid(kTwitchUserId);

  RaidStatus status;
  status.raidId = kDefaultRaidId;
  status.creatorUserId = kTwitchUserId;
  status.sourceChannelId = kSourceBroadcasterUserId;
  status.targetChannelId = kTargetChannelId;
  status.targetUserLogin = kTargetUserLogin;
  status.targetUserDisplayName = kTargetUserDisplayName;
  status.targetUserProfileImageUrl = kTargetUserProfileImage;
  status.transitionJitterSeconds = 1;
  status.numUsersInRaid = 1;
  status.forceRaidNowSeconds = 90;
  status.joined = true;

  // Calls to Start/Cancel should succeed
  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };
  TTV_ErrorCode ec = mRaid->Start(kTargetChannelId, [&callbackCalled](TTV_ErrorCode ec) {
    callbackCalled = true;
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
  });
  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(waitForCallbackFunc());

  callbackCalled = false;
  ec = mRaid->Cancel([&callbackCalled](TTV_ErrorCode ec) {
    callbackCalled = true;
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
  });
  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(waitForCallbackFunc());
}
