/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/chatapitest.h"
#include "twitchsdk/chat/chatroommessagehandler.h"

using namespace ttv::chat;
using namespace ttv::chat::test;

namespace {
class TestChatRoomMessageHandlerCallbacks : public ChatRoomMessageHandler::ICallbacks {
 public:
  enum class Functions {
    PassThrough,
    BanUser,
    UnbanUser,
    TimeoutUser,
    UntimeoutUser,
    SetUserColor,
    Help,
    ListRooms,
    ModUser,
    UnmodUser,
    BlockUser,
    UnblockUser,
    ListModerators,
    SetTopic,
    SlowMode,
    R9kMode,
    EmotesOnlyMode,
    MalformedCommand
  };

  virtual bool PassThrough(const std::string& message) override {
    mFunctionCalled = Functions::PassThrough;
    mStringParam = message;
    return true;
  }

  virtual bool BanUser(const std::string& userName) override {
    mFunctionCalled = Functions::BanUser;
    mStringParam = userName;
    return true;
  }

  virtual bool UnbanUser(const std::string& userName) override {
    mFunctionCalled = Functions::UnbanUser;
    mStringParam = userName;
    return true;
  }

  virtual bool TimeoutUser(const std::string& userName, uint32_t duration) override {
    mFunctionCalled = Functions::TimeoutUser;
    mStringParam = userName;
    mIntParam = duration;
    return true;
  }

  virtual bool UntimeoutUser(const std::string& userName) override {
    mFunctionCalled = Functions::UntimeoutUser;
    mStringParam = userName;
    return true;
  }

  virtual bool SetUserColor(const std::string& color) override {
    mFunctionCalled = Functions::SetUserColor;
    mStringParam = color;
    return true;
  }

  virtual bool Help() override {
    mFunctionCalled = Functions::Help;
    return true;
  }

  virtual bool ListRooms() override {
    mFunctionCalled = Functions::ListRooms;
    return true;
  }

  virtual bool ModUser(const std::string& userName) override {
    mFunctionCalled = Functions::ModUser;
    mStringParam = userName;
    return true;
  }

  virtual bool UnmodUser(const std::string& userName) override {
    mFunctionCalled = Functions::UnmodUser;
    mStringParam = userName;
    return true;
  }

  virtual bool BlockUser(const std::string& userName) override {
    mFunctionCalled = Functions::BlockUser;
    mStringParam = userName;
    return true;
  }

  virtual bool UnblockUser(const std::string& userName) override {
    mFunctionCalled = Functions::UnblockUser;
    mStringParam = userName;
    return true;
  }

  virtual bool ListModerators() override {
    mFunctionCalled = Functions::ListModerators;
    return true;
  }

  virtual bool SetTopic(const std::string& topic) override {
    mFunctionCalled = Functions::SetTopic;
    mStringParam = topic;
    return true;
  }

  virtual bool SlowMode(bool turnOn, uint32_t durationSeconds) override {
    mFunctionCalled = Functions::SlowMode;
    mBoolParam = turnOn;
    mIntParam = durationSeconds;
    return true;
  }

  virtual bool R9kMode(bool turnOn) override {
    mFunctionCalled = Functions::R9kMode;
    mBoolParam = turnOn;
    return true;
  }

  virtual bool EmotesOnlyMode(bool turnOn) override {
    mFunctionCalled = Functions::EmotesOnlyMode;
    mBoolParam = turnOn;
    return true;
  }

  virtual bool MalformedCommand(ChatRoomMessageHandler::CommandError command, const std::string& word) override {
    mFunctionCalled = Functions::MalformedCommand;
    mErrorParam = command;
    return true;
  }

  Functions GetFunctionCalled() const { return mFunctionCalled; }

  std::string GetStringParam() const { return mStringParam; }

  uint32_t GetIntParam() const { return mIntParam; }

  ChatRoomMessageHandler::CommandError GetErrorParam() const { return mErrorParam; }

  bool GetBoolParam() const { return mBoolParam; }

 private:
  Functions mFunctionCalled;
  std::string mStringParam;
  uint32_t mIntParam;
  ChatRoomMessageHandler::CommandError mErrorParam;
  bool mBoolParam;
};
}  // namespace

TEST_F(ChatBaseTest, ChatRoomMessageHandler) {
  std::shared_ptr<TestChatRoomMessageHandlerCallbacks> callbacks =
    std::make_shared<TestChatRoomMessageHandlerCallbacks>();
  ChatRoomMessageHandler handler;
  handler.SetCallbacks(callbacks);

  ASSERT_TRUE(handler.HandleMessage("/ban username"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::BanUser);
  ASSERT_EQ(callbacks->GetStringParam(), "username");

  ASSERT_TRUE(handler.HandleMessage("/unban username2"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::UnbanUser);
  ASSERT_EQ(callbacks->GetStringParam(), "username2");

  ASSERT_TRUE(handler.HandleMessage("/timeout username3"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::TimeoutUser);
  ASSERT_EQ(callbacks->GetStringParam(), "username3");
  ASSERT_EQ(callbacks->GetIntParam(), 600);

  ASSERT_TRUE(handler.HandleMessage("/timeout username4 400"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::TimeoutUser);
  ASSERT_EQ(callbacks->GetStringParam(), "username4");
  ASSERT_EQ(callbacks->GetIntParam(), 400);

  ASSERT_TRUE(handler.HandleMessage("/untimeout username5"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::UntimeoutUser);
  ASSERT_EQ(callbacks->GetStringParam(), "username5");

  ASSERT_TRUE(handler.HandleMessage("/color #000000"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::SetUserColor);
  ASSERT_EQ(callbacks->GetStringParam(), "#000000");

  ASSERT_TRUE(handler.HandleMessage("/color purple"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::SetUserColor);
  ASSERT_EQ(callbacks->GetStringParam(), "purple");

  ASSERT_TRUE(handler.HandleMessage("/help"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::Help);

  ASSERT_TRUE(handler.HandleMessage("/listrooms"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::ListRooms);

  ASSERT_TRUE(handler.HandleMessage("/mod username6"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::ModUser);
  ASSERT_EQ(callbacks->GetStringParam(), "username6");

  ASSERT_TRUE(handler.HandleMessage("/unmod username7"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::UnmodUser);
  ASSERT_EQ(callbacks->GetStringParam(), "username7");

  ASSERT_TRUE(handler.HandleMessage("/ignore username8"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::BlockUser);
  ASSERT_EQ(callbacks->GetStringParam(), "username8");

  ASSERT_TRUE(handler.HandleMessage("/unignore username9"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::UnblockUser);
  ASSERT_EQ(callbacks->GetStringParam(), "username9");

  ASSERT_TRUE(handler.HandleMessage("/mods"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::ListModerators);

  ASSERT_TRUE(handler.HandleMessage("/topic this is the new topic of the room"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::SetTopic);
  ASSERT_EQ(callbacks->GetStringParam(), "this is the new topic of the room");

  ASSERT_TRUE(handler.HandleMessage("/slow"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::SlowMode);
  ASSERT_TRUE(callbacks->GetBoolParam());
  ASSERT_EQ(callbacks->GetIntParam(), 120);

  ASSERT_TRUE(handler.HandleMessage("/slow 60"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::SlowMode);
  ASSERT_TRUE(callbacks->GetBoolParam());
  ASSERT_EQ(callbacks->GetIntParam(), 60);

  ASSERT_TRUE(handler.HandleMessage("/slowoff"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::SlowMode);
  ASSERT_FALSE(callbacks->GetBoolParam());

  ASSERT_TRUE(handler.HandleMessage("/r9kbeta"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::R9kMode);
  ASSERT_TRUE(callbacks->GetBoolParam());

  ASSERT_TRUE(handler.HandleMessage("/r9kbetaoff"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::R9kMode);
  ASSERT_FALSE(callbacks->GetBoolParam());

  ASSERT_TRUE(handler.HandleMessage("/emoteonly"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::EmotesOnlyMode);
  ASSERT_TRUE(callbacks->GetBoolParam());

  ASSERT_TRUE(handler.HandleMessage("/emoteonlyoff"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::EmotesOnlyMode);
  ASSERT_FALSE(callbacks->GetBoolParam());
}

TEST_F(ChatBaseTest, ChatRoomMessageHandlerPassThroughTest) {
  std::shared_ptr<TestChatRoomMessageHandlerCallbacks> callbacks =
    std::make_shared<TestChatRoomMessageHandlerCallbacks>();
  ChatRoomMessageHandler handler;
  handler.SetCallbacks(callbacks);

  std::vector<std::string> messageStrings = {"normal message", "", "  ", "   message   ", "/me is feeling good"};

  for (const std::string& messageString : messageStrings) {
    ASSERT_TRUE(handler.HandleMessage(messageString));
    ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::PassThrough);
    ASSERT_EQ(callbacks->GetStringParam(), messageString);
  }
}

TEST_F(ChatBaseTest, ChatRoomMessageHandlerErrorTest) {
  std::shared_ptr<TestChatRoomMessageHandlerCallbacks> callbacks =
    std::make_shared<TestChatRoomMessageHandlerCallbacks>();
  ChatRoomMessageHandler handler;
  handler.SetCallbacks(static_cast<std::shared_ptr<ChatRoomMessageHandler::ICallbacks>>(callbacks));

  std::vector<std::string> messageStrings = {
    "normal message", "/notarealcommand test", "", "  ", "   message   ", "/emoteonl command too short"};

  ASSERT_TRUE(handler.HandleMessage("/ban"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::MalformedCommand);
  ASSERT_EQ(callbacks->GetErrorParam(), ChatRoomMessageHandler::CommandError::Ban);

  ASSERT_TRUE(handler.HandleMessage("/unban"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::MalformedCommand);
  ASSERT_EQ(callbacks->GetErrorParam(), ChatRoomMessageHandler::CommandError::Unban);

  ASSERT_TRUE(handler.HandleMessage("/timeout"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::MalformedCommand);
  ASSERT_EQ(callbacks->GetErrorParam(), ChatRoomMessageHandler::CommandError::Timeout);

  ASSERT_TRUE(handler.HandleMessage("/untimeout"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::MalformedCommand);
  ASSERT_EQ(callbacks->GetErrorParam(), ChatRoomMessageHandler::CommandError::Untimeout);

  ASSERT_TRUE(handler.HandleMessage("/mod"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::MalformedCommand);
  ASSERT_EQ(callbacks->GetErrorParam(), ChatRoomMessageHandler::CommandError::Mod);

  ASSERT_TRUE(handler.HandleMessage("/unmod"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::MalformedCommand);
  ASSERT_EQ(callbacks->GetErrorParam(), ChatRoomMessageHandler::CommandError::Unmod);

  ASSERT_TRUE(handler.HandleMessage("/color"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::MalformedCommand);
  ASSERT_EQ(callbacks->GetErrorParam(), ChatRoomMessageHandler::CommandError::Color);

  ASSERT_TRUE(handler.HandleMessage("/topic"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::MalformedCommand);
  ASSERT_EQ(callbacks->GetErrorParam(), ChatRoomMessageHandler::CommandError::Topic);

  ASSERT_TRUE(handler.HandleMessage("/slow asdf"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::MalformedCommand);
  ASSERT_EQ(callbacks->GetErrorParam(), ChatRoomMessageHandler::CommandError::Slow);

  ASSERT_TRUE(handler.HandleMessage("/me"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::MalformedCommand);
  ASSERT_EQ(callbacks->GetErrorParam(), ChatRoomMessageHandler::CommandError::Me);

  ASSERT_TRUE(handler.HandleMessage("/me          "));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::MalformedCommand);
  ASSERT_EQ(callbacks->GetErrorParam(), ChatRoomMessageHandler::CommandError::Me);

  ASSERT_TRUE(handler.HandleMessage("/notarealcommand test"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::MalformedCommand);
  ASSERT_EQ(callbacks->GetErrorParam(), ChatRoomMessageHandler::CommandError::Unknown);

  ASSERT_TRUE(handler.HandleMessage("/emoteonl command too short"));
  ASSERT_EQ(callbacks->GetFunctionCalled(), TestChatRoomMessageHandlerCallbacks::Functions::MalformedCommand);
  ASSERT_EQ(callbacks->GetErrorParam(), ChatRoomMessageHandler::CommandError::Unknown);
}
