#include "chattestmanager.h"
#include "fixtures/chatapitest.h"
#include "testsystemclock.h"
#include "testutilities.h"
#include "twitchsdk/chat/chatapi.h"
#include "twitchsdk/chat/ichatroom.h"
#include "twitchsdk/chat/ichatroomlistener.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/socket.h"
#include "twitchsdk/core/stringutilities.h"

#include "gtest/gtest.h"

using namespace ttv;
using namespace ttv::chat;
using namespace ttv::chat::test;
using namespace ttv::test;

namespace {
const std::string kDefaultRoomId = "aabbccdd";
const UserId kDefaultUserId = 12826;

class TestChatRoomListener : public IChatRoomListener {
 public:
  virtual void MessageReceived(const std::string& /*roomId*/, ChatRoomMessage&& /*message*/) override {}

  virtual void MessageEdited(const std::string& /*roomId*/, ChatRoomMessage&& /*message*/) override {}

  virtual void MessageDeleted(const std::string& /*roomId*/, ChatRoomMessage&& /*message*/) override {}

  virtual void RoomUpdated(ChatRoomInfo&& /*roomInfo*/) override {}
};

class ChatRoomTest : public ChatApiTest {
 public:
 protected:
  void LogIn(UserId userId) {
    TestParams testParams;
    testParams.userInfo.userId = userId;
    testParams.userInfo.userName = "valid_login";
    testParams.userInfo.displayName = "Display Name";
    testParams.oauth = "valid_oauth";

    TTV_ErrorCode ec = ChatApiTest::LogIn(testParams);
    EXPECT_EQ(TTV_EC_SUCCESS, ec);
  }

  void CreateRoom(UserId userId) {
    mListener = std::make_shared<TestChatRoomListener>();
    ASSERT_TRUE(TTV_SUCCEEDED(mChatApi->CreateChatRoom(userId, kDefaultRoomId, userId, mListener, mRoom)));
  }

 protected:
  std::shared_ptr<ttv::chat::IChatRoom> mRoom;
  std::shared_ptr<TestChatRoomListener> mListener;
};
}  // namespace

TEST_F(ChatRoomTest, FetchLastMessages) {
  TestChatApiInitializationCallback(TokenizationOptions::All());

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatroom_fetchmessages.json")
    .Done();

  LogIn(kDefaultUserId);
  CreateRoom(kDefaultUserId);

  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };
  TTV_ErrorCode ec = mRoom->FetchMessagesBeforeCursor("", 100,
    [&callbackCalled](
      TTV_ErrorCode ec, std::vector<ChatRoomMessage>&& messages, std::string&& nextCursor, bool moreMessages) {
      callbackCalled = true;
      ASSERT_TRUE(TTV_SUCCEEDED(ec));

      ASSERT_EQ(messages.size(), 5);
      ASSERT_EQ(nextCursor, "1509429100997201692|c15fa1a5-d09f-4ef6-9797-3923b8178c92");
      ASSERT_TRUE(moreMessages);

      ChatRoomMessage message = std::move(messages[0]);
      ASSERT_EQ(message.roomId, "2cf8d794-46bd-4c29-8e6e-c602e1becaa9");
      ASSERT_EQ(message.roomMessageId, "528c8d7e-bef4-4e33-af11-80e62a009ea0");
      ASSERT_EQ(message.messageInfo.userId, 12826);
      ASSERT_EQ(message.messageInfo.displayName, "Twitch");
      ASSERT_EQ(message.messageInfo.userName, "Twitch");
      ASSERT_EQ(message.messageInfo.nameColorARGB, 4287245282);
      ASSERT_EQ(message.messageInfo.badges.size(), 2);
      ASSERT_EQ(message.messageInfo.timestamp, 1509431274);
      ASSERT_TRUE(message.messageInfo.flags.deleted);

      ASSERT_EQ(message.messageInfo.tokens.size(), 6);
      ASSERT_EQ(message.messageInfo.tokens[0]->GetType(), MessageToken::Type::Text);
      TextToken* textToken = static_cast<TextToken*>(message.messageInfo.tokens[0].get());
      ASSERT_EQ(textToken->text, "@fake");
      ASSERT_EQ(message.messageInfo.tokens[1]->GetType(), MessageToken::Type::Text);
      textToken = static_cast<TextToken*>(message.messageInfo.tokens[1].get());
      ASSERT_EQ(textToken->text, " ");
      ASSERT_EQ(message.messageInfo.tokens[2]->GetType(), MessageToken::Type::Mention);
      MentionToken* mentionToken = static_cast<MentionToken*>(message.messageInfo.tokens[2].get());
      ASSERT_EQ(mentionToken->text, "@twitch");
      ASSERT_EQ(mentionToken->userName, "Twitch");
      ASSERT_EQ(message.messageInfo.tokens[3]->GetType(), MessageToken::Type::Text);
      textToken = static_cast<TextToken*>(message.messageInfo.tokens[3].get());
      ASSERT_EQ(textToken->text, " test ");
      ASSERT_EQ(message.messageInfo.tokens[4]->GetType(), MessageToken::Type::Emoticon);
      EmoticonToken* emoticonToken = static_cast<EmoticonToken*>(message.messageInfo.tokens[4].get());
      ASSERT_EQ(emoticonToken->emoticonText, "PogChamp");
      ASSERT_EQ(emoticonToken->emoticonId, "88");
      ASSERT_EQ(message.messageInfo.tokens[5]->GetType(), MessageToken::Type::Text);
      textToken = static_cast<TextToken*>(message.messageInfo.tokens[5].get());
      ASSERT_EQ(textToken->text, " ay");
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(waitForCallbackFunc());
}

TEST_F(ChatRoomTest, FetchBlockedMessage) {
  const uint32_t blockedUserId = 45168716;

  TestChatApiInitializationCallback(TokenizationOptions::All());

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .AddJsonValue({"variables", "channelId"}, "12826")
    .SetResponseBodyFromFile("chat/chatroom_fetchmessages.json")
    .Done();

  std::shared_ptr<MockResponse> mock = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                                         .SetType(ttv::HTTP_POST_REQUEST)
                                         .SetStatusCode(200)
                                         .SetResponseBodyFromFile("chat/pikachu_blocklist_1.json")
                                         .AddJsonValue({"variables", "id"}, "12826")
                                         .Done();

  LogIn(kDefaultUserId);
  CreateRoom(kDefaultUserId);

  std::function<bool()> checkBlockedFunc = [this, blockedUserId = blockedUserId]() {
    bool blocked;
    mChatApi->GetUserBlocked(kDefaultUserId, blockedUserId, blocked);
    return blocked;
  };

  // Wait until we have fetched the list of blocked users before fetching messages
  ASSERT_TRUE(WaitUntilResultWithPollTask(250, checkBlockedFunc, GetDefaultUpdateFunc()));

  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };
  TTV_ErrorCode ec = mRoom->FetchMessagesBeforeCursor("", 100,
    [&callbackCalled, blockedUserId = blockedUserId](
      TTV_ErrorCode ec, std::vector<ChatRoomMessage>&& messages, std::string&& /*nextCursor*/, bool /*moreMessages*/) {
      callbackCalled = true;
      ASSERT_TRUE(TTV_SUCCEEDED(ec));

      ASSERT_EQ(messages.size(), 5);

      ChatRoomMessage message = std::move(messages[4]);
      ASSERT_EQ(message.messageInfo.userId, blockedUserId);
      ASSERT_TRUE(message.messageInfo.flags.ignored);
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(waitForCallbackFunc());
}

TEST_F(ChatRoomTest, FetchRoomInfo) {
  TestChatApiInitializationCallback(TokenizationOptions::All());

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatroom_fetchinfo.json")
    .Done();

  LogIn(kDefaultUserId);
  CreateRoom(kDefaultUserId);

  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };
  TTV_ErrorCode ec = mRoom->FetchRoomInfo([&callbackCalled](TTV_ErrorCode ec, ChatRoomInfo&& info) {
    callbackCalled = true;
    ASSERT_TRUE(TTV_SUCCEEDED(ec));

    ASSERT_EQ(info.id, "2cf8d794-46bd-4c29-8e6e-c602e1becaa9");
    ASSERT_EQ(info.name, "newroom");
    ASSERT_EQ(info.topic, "this is a topic");
    ASSERT_EQ(info.rolePermissions.read, RoomRole::Everyone);
    ASSERT_EQ(info.rolePermissions.send, RoomRole::Moderator);
    ASSERT_TRUE(info.view.isMuted);
    ASSERT_TRUE(info.view.isUnread);
    ASSERT_FALSE(info.view.isArchived);
    ASSERT_TRUE(info.view.permissions.readMessages);
    ASSERT_TRUE(info.view.permissions.sendMessages);
    ASSERT_TRUE(info.view.permissions.moderate);
    ASSERT_EQ(info.view.unreadMentionCount, 1);
    ASSERT_EQ(info.owner.userName, "twitch");
    ASSERT_EQ(info.owner.displayName, "Twitch");
    ASSERT_EQ(info.owner.bio, "The company.");
    ASSERT_EQ(info.owner.logoImageUrl,
      "https://static-cdn.jtvnw.net/user-default-pictures/27103734-3cda-44d6-a384-f2ab71e4bb85-profile_image-300x300.jpg");
    ASSERT_EQ(info.owner.userId, 12826);
    ASSERT_EQ(info.modes.slowModeDurationSeconds, 60);
    ASSERT_TRUE(info.modes.r9kMode);
    ASSERT_FALSE(info.modes.emotesOnlyMode);
  });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(waitForCallbackFunc());
}

TEST_F(ChatRoomTest, SendMessage) {
  TestChatApiInitializationCallback(TokenizationOptions::All());

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatroom_sendmessage.json")
    .Done();

  LogIn(kDefaultUserId);
  CreateRoom(kDefaultUserId);

  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };

  ChatRoomMessage placeholder;
  TTV_ErrorCode ec = mRoom->SendMessage("@twitch2 test message", placeholder,
    [&callbackCalled](TTV_ErrorCode ec, SendRoomMessageError&& error, ChatRoomMessage&& message) {
      callbackCalled = true;
      ASSERT_TRUE(TTV_SUCCEEDED(ec));
      ASSERT_EQ(error.code, GraphQLErrorCode::SUCCESS);

      ASSERT_EQ(message.roomId, "2cf8d794-46bd-4c29-8e6e-c602e1becaa9");
      ASSERT_EQ(message.roomMessageId, "385c06ca-342f-4a96-a99a-e54a31d663ff");
      ASSERT_EQ(message.messageInfo.userId, 12826);
      ASSERT_EQ(message.messageInfo.displayName, "Twitch");
      ASSERT_EQ(message.messageInfo.userName, "Twitch");
      ASSERT_EQ(message.messageInfo.nameColorARGB, 0xFF8A2BE2);
      ASSERT_EQ(message.messageInfo.badges.size(), 3);
      ASSERT_EQ(message.messageInfo.tokens.size(), 2);
      ASSERT_EQ(message.messageInfo.timestamp, 1509578110);
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_EQ(placeholder.roomId, "aabbccdd");
  ASSERT_EQ(placeholder.roomMessageId, "");
  ASSERT_EQ(placeholder.messageInfo.userId, 12826);
  ASSERT_EQ(placeholder.messageInfo.displayName, "Display Name");
  ASSERT_EQ(placeholder.messageInfo.userName, "valid_login");
  ASSERT_EQ(placeholder.messageInfo.nameColorARGB, 0xFFC0C0C0);
  ASSERT_EQ(placeholder.messageInfo.badges.size(), 0);
  ASSERT_EQ(placeholder.messageInfo.tokens.size(), 2);

  ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(waitForCallbackFunc());

  callbackCalled = false;
  ec = mRoom->SendMessage("test message 2", placeholder,
    [&callbackCalled](TTV_ErrorCode ec, SendRoomMessageError&& error, ChatRoomMessage&& /*message*/) {
      callbackCalled = true;
      ASSERT_TRUE(TTV_SUCCEEDED(ec));
      ASSERT_EQ(error.code, GraphQLErrorCode::SUCCESS);
    });

  // These values for the placeholder are now cached from the previous result.
  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_EQ(placeholder.messageInfo.nameColorARGB, 0xFF8A2BE2);
  ASSERT_EQ(placeholder.messageInfo.badges.size(), 3);

  ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(waitForCallbackFunc());
}

TEST_F(ChatRoomTest, SendMessageMeCommand) {
  TestChatApiInitializationCallback(TokenizationOptions::All());

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatroom_sendmessage_me.json")
    .Done();

  LogIn(kDefaultUserId);
  CreateRoom(kDefaultUserId);

  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };

  ChatRoomMessage placeholder;
  TTV_ErrorCode ec = mRoom->SendMessage("/me is using the /me command", placeholder,
    [&callbackCalled](TTV_ErrorCode ec, SendRoomMessageError&& error, ChatRoomMessage&& message) {
      callbackCalled = true;
      ASSERT_TRUE(TTV_SUCCEEDED(ec));
      ASSERT_EQ(error.code, GraphQLErrorCode::SUCCESS);

      ASSERT_EQ(message.roomId, "2cf8d794-46bd-4c29-8e6e-c602e1becaa9");
      ASSERT_EQ(message.roomMessageId, "385c06ca-342f-4a96-a99a-e54a31d663ff");
      ASSERT_EQ(message.messageInfo.userId, 12826);
      ASSERT_EQ(message.messageInfo.displayName, "Twitch");
      ASSERT_EQ(message.messageInfo.userName, "Twitch");
      ASSERT_EQ(message.messageInfo.nameColorARGB, 0xFF8A2BE2);
      ASSERT_EQ(message.messageInfo.badges.size(), 3);
      ASSERT_EQ(message.messageInfo.tokens.size(), 1);
      ASSERT_EQ(message.messageInfo.timestamp, 1509578110);

      TextToken* textToken = static_cast<TextToken*>(message.messageInfo.tokens[0].get());
      ASSERT_EQ(textToken->text, "is using the /me command");
      ASSERT_TRUE(message.messageInfo.flags.action);
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_EQ(placeholder.roomId, "aabbccdd");
  ASSERT_EQ(placeholder.roomMessageId, "");
  ASSERT_EQ(placeholder.messageInfo.userId, 12826);
  ASSERT_EQ(placeholder.messageInfo.displayName, "Display Name");
  ASSERT_EQ(placeholder.messageInfo.userName, "valid_login");
  ASSERT_EQ(placeholder.messageInfo.nameColorARGB, 0xFFC0C0C0);
  ASSERT_EQ(placeholder.messageInfo.badges.size(), 0);
  ASSERT_EQ(placeholder.messageInfo.tokens.size(), 1);

  TextToken* textToken = static_cast<TextToken*>(placeholder.messageInfo.tokens[0].get());
  ASSERT_EQ(textToken->text, "is using the /me command");
  ASSERT_TRUE(placeholder.messageInfo.flags.action);

  ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(waitForCallbackFunc());
}

TEST_F(ChatRoomTest, SendMessageError) {
  TestChatApiInitializationCallback(TokenizationOptions::All());

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatroom_sendmessage_error.json")
    .Done();

  LogIn(kDefaultUserId);
  CreateRoom(kDefaultUserId);

  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };

  ChatRoomMessage placeholder;
  TTV_ErrorCode ec = mRoom->SendMessage("test message", placeholder,
    [&callbackCalled](TTV_ErrorCode ec, SendRoomMessageError&& error, ChatRoomMessage&& /*message*/) {
      callbackCalled = true;
      ASSERT_FALSE(TTV_SUCCEEDED(ec));
      ASSERT_EQ(error.code, GraphQLErrorCode::SLOW_MODE_ENFORCEMENT_FAILED);
      ASSERT_EQ(error.slowModeDurationSeconds, 60);
      ASSERT_EQ(error.remainingDurationSeconds, 58);
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_EQ(placeholder.roomId, "aabbccdd");
  ASSERT_EQ(placeholder.roomMessageId, "");
  ASSERT_EQ(placeholder.messageInfo.userId, 12826);
  ASSERT_EQ(placeholder.messageInfo.displayName, "Display Name");
  ASSERT_EQ(placeholder.messageInfo.userName, "valid_login");
  ASSERT_EQ(placeholder.messageInfo.nameColorARGB, 0xFFC0C0C0);
  ASSERT_EQ(placeholder.messageInfo.badges.size(), 0);
  ASSERT_EQ(placeholder.messageInfo.tokens.size(), 1);

  ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(waitForCallbackFunc());
}

TEST_F(ChatRoomTest, EditMessage) {
  TestChatApiInitializationCallback(TokenizationOptions::All());

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatroom_editmessage.json")
    .Done();

  LogIn(kDefaultUserId);
  CreateRoom(kDefaultUserId);

  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };

  ChatRoomMessage placeholder;
  TTV_ErrorCode ec = mRoom->EditMessage("6e726405-e937-4a79-9f7c-34c5df66b151", "new message content", placeholder,
    [&callbackCalled](TTV_ErrorCode ec, ChatRoomMessage&& message) {
      callbackCalled = true;
      ASSERT_TRUE(TTV_SUCCEEDED(ec));

      ASSERT_EQ(message.roomId, "2cf8d794-46bd-4c29-8e6e-c602e1becaa9");
      ASSERT_EQ(message.roomMessageId, "6e726405-e937-4a79-9f7c-34c5df66b151");
      ASSERT_EQ(message.messageInfo.userId, 12826);
      ASSERT_EQ(message.messageInfo.displayName, "Twitch");
      ASSERT_EQ(message.messageInfo.userName, "Twitch");
      ASSERT_EQ(message.messageInfo.nameColorARGB, 0xFF8A2BE2);
      ASSERT_EQ(message.messageInfo.badges.size(), 3);
      ASSERT_EQ(message.messageInfo.tokens.size(), 1);
      ASSERT_EQ(message.messageInfo.timestamp, 1509578110);
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_EQ(placeholder.roomId, "aabbccdd");
  ASSERT_EQ(placeholder.roomMessageId, "");
  ASSERT_EQ(placeholder.messageInfo.userId, 12826);
  ASSERT_EQ(placeholder.messageInfo.displayName, "Display Name");
  ASSERT_EQ(placeholder.messageInfo.userName, "valid_login");
  ASSERT_EQ(placeholder.messageInfo.nameColorARGB, 0xFFC0C0C0);
  ASSERT_EQ(placeholder.messageInfo.badges.size(), 0);
  ASSERT_EQ(placeholder.messageInfo.tokens.size(), 1);

  ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(waitForCallbackFunc());
}

TEST_F(ChatRoomTest, DeleteRoom) {
  TestChatApiInitializationCallback(TokenizationOptions::All());

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatroom_deleteroom.json")
    .Done();

  LogIn(kDefaultUserId);
  CreateRoom(kDefaultUserId);

  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };

  ChatRoomMessage placeholder;
  TTV_ErrorCode ec = mRoom->DeleteRoom([&callbackCalled](TTV_ErrorCode ec) {
    callbackCalled = true;
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
  });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(waitForCallbackFunc());
}

TEST_F(ChatRoomTest, UpdateRoomInfo) {
  TestChatApiInitializationCallback(TokenizationOptions::All());

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatroom_updateinfo.json")
    .Done();

  LogIn(kDefaultUserId);
  CreateRoom(kDefaultUserId);

  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };

  ChatRoomMessage placeholder;
  TTV_ErrorCode ec = mRoom->SetRoomName("room-name", [&callbackCalled](
                                                       TTV_ErrorCode ec, UpdateRoomError&& error, ChatRoomInfo&& info) {
    callbackCalled = true;
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    ASSERT_EQ(error.code, GraphQLErrorCode::SUCCESS);

    ASSERT_EQ(info.id, "2cf8d794-46bd-4c29-8e6e-c602e1becaa9");
    ASSERT_EQ(info.name, "newroom");
    ASSERT_EQ(info.topic, "this is a topic");
    ASSERT_EQ(info.rolePermissions.read, RoomRole::Everyone);
    ASSERT_EQ(info.rolePermissions.send, RoomRole::Moderator);
    ASSERT_TRUE(info.view.isMuted);
    ASSERT_TRUE(info.view.isUnread);
    ASSERT_FALSE(info.view.isArchived);
    ASSERT_TRUE(info.view.permissions.readMessages);
    ASSERT_TRUE(info.view.permissions.sendMessages);
    ASSERT_TRUE(info.view.permissions.moderate);
    ASSERT_EQ(info.view.unreadMentionCount, 1);
    ASSERT_EQ(info.owner.userName, "twitch");
    ASSERT_EQ(info.owner.displayName, "Twitch");
    ASSERT_EQ(info.owner.bio, "The company.");
    ASSERT_EQ(info.owner.logoImageUrl,
      "https://static-cdn.jtvnw.net/user-default-pictures/27103734-3cda-44d6-a384-f2ab71e4bb85-profile_image-300x300.jpg");
    ASSERT_EQ(info.owner.userId, 12826);
    ASSERT_EQ(info.modes.slowModeDurationSeconds, 60);
    ASSERT_TRUE(info.modes.r9kMode);
    ASSERT_FALSE(info.modes.emotesOnlyMode);
  });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(waitForCallbackFunc());
}

TEST_F(ChatRoomTest, UpdateRoomInfo_Error) {
  TestChatApiInitializationCallback(TokenizationOptions::All());

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatroom_updateinfo_error.json")
    .Done();

  LogIn(kDefaultUserId);
  CreateRoom(kDefaultUserId);

  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };

  ChatRoomMessage placeholder;
  TTV_ErrorCode ec =
    mRoom->SetRoomName("aa", [&callbackCalled](TTV_ErrorCode ec, UpdateRoomError&& error, ChatRoomInfo&& /*info*/) {
      callbackCalled = true;
      ASSERT_FALSE(TTV_SUCCEEDED(ec));
      ASSERT_EQ(error.code, GraphQLErrorCode::NAME_LENGTH_INVALID);
      ASSERT_EQ(error.minLength, 3);
      ASSERT_EQ(error.maxLength, 25);
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(waitForCallbackFunc());
}

TEST_F(ChatRoomTest, UpdateRoomModes) {
  TestChatApiInitializationCallback(TokenizationOptions::All());

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatroom_updatemodes.json")
    .Done();

  LogIn(kDefaultUserId);
  CreateRoom(kDefaultUserId);

  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };

  ChatRoomMessage placeholder;
  TTV_ErrorCode ec = mRoom->EnableSlowMode(60, [&callbackCalled](
                                                 TTV_ErrorCode ec, UpdateRoomModesError&& error, ChatRoomInfo&& info) {
    callbackCalled = true;
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    ASSERT_EQ(error.code, GraphQLErrorCode::SUCCESS);

    ASSERT_EQ(info.id, "2cf8d794-46bd-4c29-8e6e-c602e1becaa9");
    ASSERT_EQ(info.name, "newroom");
    ASSERT_EQ(info.topic, "this is a topic");
    ASSERT_EQ(info.rolePermissions.read, RoomRole::Everyone);
    ASSERT_EQ(info.rolePermissions.send, RoomRole::Moderator);
    ASSERT_TRUE(info.view.isMuted);
    ASSERT_TRUE(info.view.isUnread);
    ASSERT_FALSE(info.view.isArchived);
    ASSERT_TRUE(info.view.permissions.readMessages);
    ASSERT_TRUE(info.view.permissions.sendMessages);
    ASSERT_TRUE(info.view.permissions.moderate);
    ASSERT_EQ(info.view.unreadMentionCount, 1);
    ASSERT_EQ(info.owner.userName, "twitch");
    ASSERT_EQ(info.owner.displayName, "Twitch");
    ASSERT_EQ(info.owner.bio, "The company.");
    ASSERT_EQ(info.owner.logoImageUrl,
      "https://static-cdn.jtvnw.net/user-default-pictures/27103734-3cda-44d6-a384-f2ab71e4bb85-profile_image-300x300.jpg");
    ASSERT_EQ(info.owner.userId, 12826);
    ASSERT_EQ(info.modes.slowModeDurationSeconds, 60);
    ASSERT_TRUE(info.modes.r9kMode);
    ASSERT_FALSE(info.modes.emotesOnlyMode);
  });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(waitForCallbackFunc());
}

TEST_F(ChatRoomTest, UpdateRoomModes_Error) {
  TestChatApiInitializationCallback(TokenizationOptions::All());

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatroom_updatemodes_error.json")
    .Done();

  LogIn(kDefaultUserId);
  CreateRoom(kDefaultUserId);

  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };

  ChatRoomMessage placeholder;
  TTV_ErrorCode ec = mRoom->EnableSlowMode(
    100000, [&callbackCalled](TTV_ErrorCode ec, UpdateRoomModesError&& error, ChatRoomInfo&& /*info*/) {
      callbackCalled = true;
      ASSERT_FALSE(TTV_SUCCEEDED(ec));
      ASSERT_EQ(error.code, GraphQLErrorCode::SLOW_MODE_DURATION_INVALID);
      ASSERT_EQ(error.minimumSlowModeDurationSeconds, 1);
      ASSERT_EQ(error.maximumSlowModeDurationSeconds, 86400);
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(waitForCallbackFunc());
}

TEST_F(ChatRoomTest, UpdateRoomView) {
  TestChatApiInitializationCallback(TokenizationOptions::All());

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatroom_updateview.json")
    .Done();

  LogIn(kDefaultUserId);
  CreateRoom(kDefaultUserId);

  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };

  ChatRoomMessage placeholder;
  TTV_ErrorCode ec = mRoom->SetMuted(true, [&callbackCalled](TTV_ErrorCode ec, ChatRoomInfo&& info) {
    callbackCalled = true;
    ASSERT_TRUE(TTV_SUCCEEDED(ec));

    ASSERT_EQ(info.id, "2cf8d794-46bd-4c29-8e6e-c602e1becaa9");
    ASSERT_EQ(info.name, "newroom");
    ASSERT_EQ(info.topic, "this is a topic");
    ASSERT_EQ(info.rolePermissions.read, RoomRole::Everyone);
    ASSERT_EQ(info.rolePermissions.send, RoomRole::Moderator);
    ASSERT_TRUE(info.view.isMuted);
    ASSERT_TRUE(info.view.isUnread);
    ASSERT_FALSE(info.view.isArchived);
    ASSERT_TRUE(info.view.permissions.readMessages);
    ASSERT_TRUE(info.view.permissions.sendMessages);
    ASSERT_TRUE(info.view.permissions.moderate);
    ASSERT_EQ(info.view.unreadMentionCount, 1);
    ASSERT_EQ(info.owner.userName, "twitch");
    ASSERT_EQ(info.owner.displayName, "Twitch");
    ASSERT_EQ(info.owner.bio, "The company.");
    ASSERT_EQ(info.owner.logoImageUrl,
      "https://static-cdn.jtvnw.net/user-default-pictures/27103734-3cda-44d6-a384-f2ab71e4bb85-profile_image-300x300.jpg");
    ASSERT_EQ(info.owner.userId, 12826);
    ASSERT_EQ(info.modes.slowModeDurationSeconds, 60);
    ASSERT_TRUE(info.modes.r9kMode);
    ASSERT_FALSE(info.modes.emotesOnlyMode);
  });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(waitForCallbackFunc());
}
