/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/chathelpers.h"
#include "twitchsdk/chat/internal/ircstring.h"

#include "gtest/gtest.h"

using namespace ttv::chat;

namespace {
bool TestRegexAgainstUrlParse(const std::regex& expression, const std::string& word) {
  return (std::regex_match(word, expression) == IsTwitchChatUrl(word));
}
}  // namespace

TEST(MessageTagUnescaping, NoSpecialCharacters) {
  std::string escapedString = "StringWithoutAnySpecialCharacters";
  std::string unescapedString = UnescapeMessageTag(escapedString);
  ASSERT_EQ(unescapedString, escapedString);
}

TEST(MessageTagUnescaping, EmptyString) {
  std::string escapedString = "";
  std::string unescapedString = UnescapeMessageTag(escapedString);
  ASSERT_EQ(unescapedString, escapedString);
}

TEST(MessageTagUnescaping, SpecialCharacters) {
  std::string escapedString = "Space\\sBackslash\\\\Semicolon\\:CarriageReturn\\rLineFeed\\n";
  std::string unescapedString = UnescapeMessageTag(escapedString);
  ASSERT_EQ(unescapedString, "Space Backslash\\Semicolon;CarriageReturn\rLineFeed\n");
}

TEST(MessageTagUnescpaing, IllegalBackslash) {
  std::string escapedString = "Backslash\\Alone";
  std::string unescapedString = UnescapeMessageTag(escapedString);
  ASSERT_EQ(unescapedString, "Backslash\\Alone");
}

TEST(MessageTagUnescaping, TrailingBackslash) {
  std::string escapedString = "TrailingBackslash\\";
  std::string unescapedString = UnescapeMessageTag(escapedString);
  ASSERT_EQ(unescapedString, "TrailingBackslash\\");
}

TEST(UrlParsing, TestUrls) {
  std::regex expression(
    "^(?:(https?://)?((?:[\\w#%\\-+=:~]+\\.)+[a-z]{2,8}(?:\\/[\\w.\\/#%&()\\-+=:?~]*)?))$", std::regex::icase);

  EXPECT_TRUE(TestRegexAgainstUrlParse(expression, ""));
  EXPECT_TRUE(TestRegexAgainstUrlParse(expression, "Kappa"));

  EXPECT_TRUE(TestRegexAgainstUrlParse(expression, "http://www.twitch.tv"));
  EXPECT_TRUE(TestRegexAgainstUrlParse(expression, "https://www.twitch.tv"));
  EXPECT_TRUE(TestRegexAgainstUrlParse(expression, "HTTP://www.twitch.tv"));
  EXPECT_TRUE(TestRegexAgainstUrlParse(expression, "HTTPS://www.twitch.tv"));

  EXPECT_TRUE(TestRegexAgainstUrlParse(expression, "twitch.tv"));
  EXPECT_TRUE(TestRegexAgainstUrlParse(expression, "twitch.t"));
  EXPECT_TRUE(TestRegexAgainstUrlParse(expression, "twitch.television"));
  EXPECT_TRUE(TestRegexAgainstUrlParse(expression, "twitch..tv"));

  EXPECT_TRUE(TestRegexAgainstUrlParse(expression, "http://www.twitch.tv/lirik"));
  EXPECT_TRUE(TestRegexAgainstUrlParse(expression, "http://www.twitch.tv?param=1&param=2"));
  EXPECT_TRUE(TestRegexAgainstUrlParse(expression, "http://www.twitch.tv#anchortag"));
  EXPECT_TRUE(TestRegexAgainstUrlParse(expression, "http://www.twitch.tv/lirik#anchortag"));

  EXPECT_TRUE(TestRegexAgainstUrlParse(expression, "http://www.twi_tch.tv"));
  EXPECT_TRUE(TestRegexAgainstUrlParse(expression, "http://www.twi_tch.t_v"));
  EXPECT_TRUE(TestRegexAgainstUrlParse(expression, "http://www.twi_tch.tv/li_rik"));

  EXPECT_TRUE(TestRegexAgainstUrlParse(expression, "http://www.twitch.tv/percent%20encoded%20crap"));
  EXPECT_TRUE(TestRegexAgainstUrlParse(expression, "http://user:pass@www.twitch.tv"));
  EXPECT_TRUE(TestRegexAgainstUrlParse(expression, "user@twitch.tv"));

  EXPECT_TRUE(TestRegexAgainstUrlParse(expression, ".twitch.tv"));
  EXPECT_TRUE(TestRegexAgainstUrlParse(expression, "http://.twitch.tv"));
  EXPECT_TRUE(TestRegexAgainstUrlParse(expression, "http://www.twitch.tv./test"));

  EXPECT_TRUE(TestRegexAgainstUrlParse(
    expression, "http://www.twitch.tv/something?guid=6D570905-95EB-41F5-B161-F19D2E249C87_(test)"));

  char extendedCharacterString[] = {-13, 0};
  EXPECT_TRUE(TestRegexAgainstUrlParse(expression, extendedCharacterString));

#if !TTV_PLATFORM_WIN32
  // Don't run this test on windows, as it will cause the regex library to choke.
  auto longString = std::string(2048, 'A');
  EXPECT_TRUE(TestRegexAgainstUrlParse(expression, longString));
#endif
}

TEST(EmoticonParsing, Unescaping) {
  std::vector<std::string> inputs = {"\\&gt\\;\\(", "\\&lt\\;3", "&amp\\;", "&apos\\;", "&quot\\;"};
  std::vector<std::string> outputs = {"\\>\\(", "\\<3", "&", "'", "\""};

  for (size_t i = 0; i < inputs.size(); ++i) {
    UnescapeEmoticonToken(inputs[i]);
    ASSERT_EQ(inputs[i], outputs[i]);
  }
}

TEST(IrcString, Compare) {
  // Empty string
  EXPECT_TRUE(ttv::chat::Compare("", "", false));
  EXPECT_TRUE(ttv::chat::Compare("", "", true));

  EXPECT_FALSE(ttv::chat::Compare("a", "", false));
  EXPECT_FALSE(ttv::chat::Compare("a", "", true));

  EXPECT_FALSE(ttv::chat::Compare("", "a", false));
  EXPECT_FALSE(ttv::chat::Compare("", "a", true));

  // length
  EXPECT_FALSE(ttv::chat::Compare("a", "abc", false));
  EXPECT_FALSE(ttv::chat::Compare("abc", "a", false));
  EXPECT_FALSE(ttv::chat::Compare("a", "abc", true));
  EXPECT_FALSE(ttv::chat::Compare("abc", "a", true));

  // case sensitivity
  EXPECT_TRUE(ttv::chat::Compare("abc", "abc", false));
  EXPECT_TRUE(ttv::chat::Compare("ABC", "abc", false));
  EXPECT_TRUE(ttv::chat::Compare("abc", "ABC", false));
  EXPECT_FALSE(ttv::chat::Compare("ABC", "abc", true));
  EXPECT_FALSE(ttv::chat::Compare("abc", "ABC", true));
}
