/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/tasktest.h"
#include "testutilities.h"
#include "twitchsdk/chat/internal/task/chatdeletecommenttask.h"
#include "twitchsdk/chat/internal/task/chatgetbadgestask.h"
#include "twitchsdk/chat/internal/task/chatgetbitsconfigtask.h"
#include "twitchsdk/chat/internal/task/chatgetvodcommentstask.h"
#include "twitchsdk/chat/internal/task/chatpropertiestask.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/task/getvodtask.h"

#include <algorithm>
#include <iostream>
#include <numeric>

using namespace ttv;
using namespace ttv::test;
using namespace ttv::chat;

namespace {
const char* kOAuthToken = "***********";
const char* kUserName = "twitch";
const char* kThreadId = "111_222";
}  // namespace

TEST_F(TaskTest, DISABLED_ChatChangeUserBlockTask) {}

TEST_F(TaskTest, DISABLED_ChatChannelUsersTask) {}

TEST_F(TaskTest, ChatGetBadgesTask_Global) {
  auto response = mHttpRequest->AddResponse("https://badges.twitch.tv/v1/badges/global/display")
                    .AddRequestParam("language", "EN")
                    .SetResponseBodyFromFile("chat/chatgetbadgestask_global.json")
                    .Done();

  bool callbackCalled = false;
  std::shared_ptr<ChatGetBadgesTask> task;

  ChatGetBadgesTask::Callback callback = [&callbackCalled, &task](
                                           ChatGetBadgesTask* source, TTV_ErrorCode ec, BadgeSet&& badgeSet) {
    callbackCalled = true;

    ASSERT_EQ(task.get(), source);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));

    ASSERT_EQ(badgeSet.language, "EN");
    ASSERT_EQ(badgeSet.badges.size(), 37);

    std::unordered_map<std::string, int> nameToVersionsSize{
      {"bits", 18}, {"subscriber", 2}, {"power-rangers", 7}, {"warcraft", 2}};

    for (const auto& kvp : badgeSet.badges) {
      auto versionSizeIter = nameToVersionsSize.find(kvp.second.name);
      if (versionSizeIter != nameToVersionsSize.end()) {
        ASSERT_EQ(kvp.second.versions.size(), versionSizeIter->second);
      } else {
        ASSERT_EQ(kvp.second.versions.size(), 1);
      }

      if (kvp.second.name == "warcraft") {
        ASSERT_TRUE(kvp.second.versions.find("alliance") != kvp.second.versions.end());
      } else {
        ASSERT_TRUE(kvp.second.versions.find("1") != kvp.second.versions.end());
      }
    }
  };

  auto checkFunction = [&callbackCalled]() -> bool { return callbackCalled; };

  task = std::make_shared<ChatGetBadgesTask>(0, std::move(callback));
  task->SetLanguage("EN");
  mTaskRunner->AddTask(task);

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, checkFunction, GetDefaultUpdateFunc()));
  response->AssertRequestsMade();
}

TEST_F(TaskTest, ChatGetBadgesTask_Channel) {
  auto response = mHttpRequest->AddResponse("https://badges.twitch.tv/v1/badges/channels/1001/display")
                    .AddRequestParam("language", "JP")
                    .SetResponseBodyFromFile("chat/chatgetbadgestask_channel.json")
                    .Done();

  bool callbackCalled = false;
  std::shared_ptr<ChatGetBadgesTask> task;

  ChatGetBadgesTask::Callback callback = [&callbackCalled, &task](
                                           ChatGetBadgesTask* source, TTV_ErrorCode ec, BadgeSet&& badgeSet) {
    callbackCalled = true;

    ASSERT_EQ(task.get(), source);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));

    ASSERT_EQ(badgeSet.language, "JP");
    ASSERT_EQ(badgeSet.badges.size(), 2);

    auto& badge = badgeSet.badges["subscriber"];
    ASSERT_EQ(badge.name, "subscriber");
    ASSERT_EQ(badge.versions.size(), 5);
    ASSERT_TRUE(badge.versions.find("0") != badge.versions.end());

    badge = badgeSet.badges["bits"];
    ASSERT_EQ(badge.name, "bits");
    ASSERT_EQ(badge.versions.size(), 16);
    ASSERT_TRUE(badge.versions.find("1000") != badge.versions.end());
  };

  auto checkFunction = [&callbackCalled]() -> bool { return callbackCalled; };

  task = std::make_shared<ChatGetBadgesTask>(1001, std::move(callback));
  task->SetLanguage("JP");
  mTaskRunner->AddTask(task);

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, checkFunction, GetDefaultUpdateFunc()));
  response->AssertRequestsMade();
}

// TEST_F(TaskTest, ChatGetBitsConfigTask) {
//  auto response = mHttpRequest->AddResponse("https://api.twitch.tv/v5/bits/actions?channel_id=1001")
//                    .SetResponseBodyFromFile("chat/chatgetbitsconfigtask_global.json")
//                    .Done();
//
//  bool callbackCalled = false;
//
//  auto callback = [&callbackCalled](ChatGetBitsConfigTask* /*source*/, TTV_ErrorCode ec,
//                    std::shared_ptr<ChatGetBitsConfigTask::Result> result) {
//    callbackCalled = true;
//
//    ASSERT_EQ(ec, TTV_EC_SUCCESS);
//
//    std::vector<BitsConfiguration::Cheermote>& cheermotes = result->cheermotes;
//    ASSERT_EQ(cheermotes.size(), 28);
//    ASSERT_EQ(cheermotes[0].prefix, "Cheer");
//    ASSERT_EQ(cheermotes[1].prefix, "DoodleCheer");
//    ASSERT_EQ(cheermotes[2].prefix, "PogChamp");
//    ASSERT_EQ(cheermotes[3].tiers.size(), 5);
//    ASSERT_EQ(cheermotes[3].tiers[2].bits, 1000);
//    ASSERT_EQ(cheermotes[3].tiers[2].images.size(), 20);
//    ASSERT_EQ(cheermotes[3].tiers[2].images[1].url,
//      "https://d3aqoihi2n8ty8.cloudfront.net/actions/pride/dark/animated/1000/1.5.gif");
//    ASSERT_EQ(cheermotes[3].tiers[2].images[12].theme, BitsConfiguration::CheermoteImage::Theme::Light);
//    ASSERT_FALSE(cheermotes[8].tiers[3].images[15].isAnimated);
//    ASSERT_FLOAT_EQ(cheermotes[8].tiers[3].images[15].dpiScale, 1.0);
//    ASSERT_FLOAT_EQ(cheermotes[8].tiers[3].images[16].dpiScale, 1.5);
//    ASSERT_FLOAT_EQ(cheermotes[8].tiers[3].images[17].dpiScale, 2.0);
//    ASSERT_FLOAT_EQ(cheermotes[8].tiers[3].images[18].dpiScale, 3.0);
//    ASSERT_FLOAT_EQ(cheermotes[8].tiers[3].images[19].dpiScale, 4.0);
//
//    ASSERT_EQ(cheermotes[0].type, BitsConfiguration::Cheermote::Type::FirstParty);
//    ASSERT_EQ(cheermotes[1].type, BitsConfiguration::Cheermote::Type::ThirdParty);
//    ASSERT_EQ(cheermotes[2].type, BitsConfiguration::Cheermote::Type::FirstParty);
//  };
//
//  auto checkFunction = [&callbackCalled]() -> bool { return callbackCalled; };
//
//  std::shared_ptr<ChatGetBitsConfigTask> task = std::make_shared<ChatGetBitsConfigTask>(1001, callback);
//  mTaskRunner->AddTask(task);
//
//  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, checkFunction, GetDefaultUpdateFunc()));
//  response->AssertRequestsMade();
//}

TEST_F(TaskTest, ChatPropertiesTask) {
  auto response = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                    .SetResponseBodyFromFile("chat/pokemonchatproperties.json")
                    .SetType(HTTP_POST_REQUEST)
                    .Done();

  bool callbackCalled = false;
  auto callback = [&callbackCalled](ChatPropertiesTask* /*source*/, TTV_ErrorCode ec,
                    std::shared_ptr<ChatPropertiesTask::Result> result) {
    callbackCalled = true;

    ASSERT_EQ(ec, TTV_EC_SUCCESS);
    ASSERT_EQ(result->chat_delay_duration, 2);
    ASSERT_EQ(result->chat_rules.size(), 2);
    ASSERT_EQ(result->chat_rules[0],
      "One Rule, don't be an idiot. pretty simple. Have some etiquette. If you don't, it is really easy to get banned.");
    ASSERT_EQ(result->chat_rules[1], "PIKAAAAAAA");
    ASSERT_EQ(result->hide_chat_links, false);
  };

  auto checkFunction = [&callbackCalled]() -> bool { return callbackCalled; };

  // Kick off the request for chat properties
  std::shared_ptr<ChatPropertiesTask> task = std::make_shared<ChatPropertiesTask>(1001, callback);
  mTaskRunner->AddTask(task);

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, checkFunction, GetDefaultUpdateFunc()));
  response->AssertRequestsMade();
}

TEST_F(TaskTest, DISABLED_ChatUpdateUserThreadTask) {}

TEST_F(TaskTest, GetVodTask) {
  MockResponse& response = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                             .SetResponseBodyFromFile("core/twitch_vod.json")
                             .SetType(ttv::HTTP_POST_REQUEST)
                             .AddJsonValue({"variables", "id"}, "123");

  bool callbackCalled = false;
  std::shared_ptr<GetVodTask> task;
  GetVodTask::Callback fetchCallback = [&callbackCalled, &task](GetVodTask* source, TTV_ErrorCode ec,
                                         std::shared_ptr<GetVodTask::Result> result) {
    callbackCalled = true;

    ASSERT_EQ(task.get(), source);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    ASSERT_TRUE(result != nullptr);

    ASSERT_STREQ(result->vodId.c_str(), "123");
    ASSERT_STREQ(result->title.c_str(), "Twitch Vod");
    ASSERT_EQ(result->channelId, 1001);
    ASSERT_EQ(result->vodType, VodType::Archive);
    ASSERT_EQ(result->status, VodStatus::Recorded);
    ASSERT_EQ(result->durationSeconds, 8372);
    ASSERT_EQ(result->recordedAtSeconds, 1469310979);
  };

  auto checkFunction = [&callbackCalled]() -> bool { return callbackCalled; };

  auto pollFunction = [this]() { Update(); };

  task = std::make_shared<GetVodTask>("123", fetchCallback);
  mTaskRunner->AddTask(task);

  EXPECT_TRUE(WaitUntilResultWithPollTask(1000, checkFunction, pollFunction));
  ASSERT_TRUE(callbackCalled);
  response.AssertRequestsMade();
}

TEST_F(TaskTest, ChatGetVodCommentsTask) {
  std::string vodId = "123";
  uint32_t offsetSeconds = 0;

  MockResponse& response = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                             .SetType(ttv::HTTP_POST_REQUEST)
                             .SetResponseBodyFromFile("chat/chatcomment_messages_cursor_next.json");

  bool callbackCalled = false;
  std::shared_ptr<ChatGetVodCommentsTask> task;
  ChatGetVodCommentsTask::Callback fetchCallback = [&callbackCalled, &task](ChatGetVodCommentsTask* source,
                                                     TTV_ErrorCode ec, ChatGetVodCommentsTask::Result&& result) {
    callbackCalled = true;
    ASSERT_EQ(task.get(), source);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));

    ASSERT_EQ(result.commentsListBatches[0].baseTimestampMilliseconds, 0);
    ChatComment commentMessage = std::move(result.commentsListBatches[0].comments[4]);
    ASSERT_EQ(commentMessage.commentId, "aH8Uy4XCvxXcpQ");
    ASSERT_EQ(commentMessage.timestampMilliseconds, 0);
    ASSERT_EQ(commentMessage.contentId, "123");
    ASSERT_EQ(commentMessage.replies.size(), 0);
    ASSERT_FALSE(commentMessage.moreReplies);
    ASSERT_EQ(commentMessage.messageInfo.userName, "freaknbigpanda");
    Color color;
    ttv::ParseColor("#008000", color);
    ASSERT_EQ(commentMessage.messageInfo.nameColorARGB, color);
    ASSERT_EQ(commentMessage.messageInfo.userId, 192829011);
    ASSERT_EQ(commentMessage.messageInfo.badges.size(), 1);
    ASSERT_EQ(commentMessage.messageInfo.tokens.size(), 1);
    ASSERT_EQ(result.nextCursorUrl, "cursorUrl");

    // Testing that emojies are properly parsed from the json response
    ChatComment multipleEmojiMessage = result.commentsListBatches[0].comments[1];
    const auto& tokens = multipleEmojiMessage.messageInfo.tokens;
    const auto& wutFaceOne = static_cast<EmoticonToken*>(tokens[1].get());
    ASSERT_EQ(wutFaceOne->emoticonText, "WutFace");
    const auto& wutFaceTwo = static_cast<EmoticonToken*>(tokens[3].get());
    ASSERT_EQ(wutFaceTwo->emoticonText, "WutFace");
    const auto& kreyGasm = static_cast<EmoticonToken*>(tokens[5].get());
    ASSERT_EQ(kreyGasm->emoticonText, "Kreygasm");
    const auto& pogChamp = static_cast<EmoticonToken*>(tokens[7].get());
    ASSERT_EQ(pogChamp->emoticonText, "PogChamp");

    ChatComment noEmojiesMessage = result.commentsListBatches[0].comments[0];
    const auto& noEmojiTokens = noEmojiesMessage.messageInfo.tokens;
    for (const auto& token : noEmojiTokens) {
      ASSERT_NE(token->GetType(), MessageToken::Type::Emoticon);
    }

    ChatComment singleEmojiMessage = result.commentsListBatches[0].comments[2];
    const auto& singleEmojiTokens = singleEmojiMessage.messageInfo.tokens;
    const auto& pogChampOne = static_cast<EmoticonToken*>(singleEmojiTokens[1].get());
    ASSERT_EQ(pogChampOne->emoticonText, "PogChamp");
  };

  auto checkFunction = [&callbackCalled]() -> bool { return callbackCalled; };

  auto pollFunction = [this]() { Update(); };

  task = std::make_shared<ChatGetVodCommentsTask>(
    vodId, offsetSeconds * 1000, TokenizationOptions::All(), nullptr, std::move(fetchCallback));
  mTaskRunner->AddTask(task);

  EXPECT_TRUE(WaitUntilResultWithPollTask(1000, checkFunction, pollFunction));
  ASSERT_TRUE(callbackCalled);
  response.AssertRequestsMade();
}

TEST_F(TaskTest, ChatGetVodCommentsTask_EmptyMessages) {
  std::string vodId = "123";

  uint32_t offsetSeconds = 0;

  MockResponse& response = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                             .SetType(ttv::HTTP_POST_REQUEST)
                             .SetResponseBodyFromFile("chat/chatcomment_messages_empty.json");

  bool callbackCalled = false;
  std::shared_ptr<ChatGetVodCommentsTask> task;
  ChatGetVodCommentsTask::Callback fetchCallback = [&callbackCalled, &task](ChatGetVodCommentsTask* source,
                                                     TTV_ErrorCode ec, ChatGetVodCommentsTask::Result&& result) {
    callbackCalled = true;

    ASSERT_EQ(task.get(), source);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    ASSERT_TRUE(result.commentsListBatches.empty());
  };

  auto checkFunction = [&callbackCalled]() -> bool { return callbackCalled; };

  auto pollFunction = [this]() { Update(); };

  task = std::make_shared<ChatGetVodCommentsTask>(
    vodId, offsetSeconds, TokenizationOptions::All(), nullptr, std::move(fetchCallback));
  mTaskRunner->AddTask(task);

  EXPECT_TRUE(WaitUntilResultWithPollTask(1000, checkFunction, pollFunction));
  ASSERT_TRUE(callbackCalled);
  response.AssertRequestsMade();
}

TEST_F(TaskTest, ChatGetVodCommentsTask_BufferOverflow) {
  std::string vodId = "123";
  uint32_t offsetSeconds = 0;

  MockResponse& response = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                             .SetType(ttv::HTTP_POST_REQUEST)
                             .SetResponseBodyFromFile("chat/chatcomment_messages.json");

  bool callbackCalled = false;
  std::shared_ptr<ChatGetVodCommentsTask> task;
  ChatGetVodCommentsTask::Callback fetchCallback = [&callbackCalled, &task](ChatGetVodCommentsTask* source,
                                                     TTV_ErrorCode ec, ChatGetVodCommentsTask::Result&& result) {
    callbackCalled = true;

    ASSERT_EQ(task.get(), source);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    ASSERT_TRUE(result.commentsListBatches.size() > 1);
    int count = std::accumulate(result.commentsListBatches.begin(), result.commentsListBatches.end(), 0,
      [](int currentSum, const auto& batch) { return currentSum + static_cast<int>(batch.comments.size()); });
    ASSERT_EQ(count, 56);
  };

  auto checkFunction = [&callbackCalled]() -> bool { return callbackCalled; };

  auto pollFunction = [this]() { Update(); };

  task = std::make_shared<ChatGetVodCommentsTask>(
    vodId, offsetSeconds, TokenizationOptions::All(), nullptr, std::move(fetchCallback));
  mTaskRunner->AddTask(task);

  EXPECT_TRUE(WaitUntilResultWithPollTask(5000, checkFunction, pollFunction));
  ASSERT_TRUE(callbackCalled);
  response.AssertRequestsMade();
}

TEST_F(TaskTest, ChatGetVodCommentsTask_InvalidJson) {
  std::string vodId = "123";
  uint32_t offsetSeconds = 0;

  MockResponse& response = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                             .SetType(ttv::HTTP_POST_REQUEST)
                             .SetResponseBodyFromFile("chat/chatcomment_messages_invalid_json.json");

  bool callbackCalled = false;
  std::shared_ptr<ChatGetVodCommentsTask> task;
  ChatGetVodCommentsTask::Callback fetchCallback = [&callbackCalled, &task](ChatGetVodCommentsTask* source,
                                                     TTV_ErrorCode ec, ChatGetVodCommentsTask::Result&& result) {
    callbackCalled = true;

    ASSERT_EQ(task.get(), source);
    ASSERT_TRUE(TTV_FAILED(ec));
    ASSERT_TRUE(result.commentsListBatches.empty());
  };

  auto checkFunction = [&callbackCalled]() -> bool { return callbackCalled; };

  auto pollFunction = [this]() { Update(); };

  task = std::make_shared<ChatGetVodCommentsTask>(
    vodId, offsetSeconds, TokenizationOptions::All(), nullptr, std::move(fetchCallback));
  mTaskRunner->AddTask(task);

  EXPECT_TRUE(WaitUntilResultWithPollTask(1000, checkFunction, pollFunction));
  ASSERT_TRUE(callbackCalled);
  response.AssertRequestsMade();
}

TEST_F(TaskTest, ChatDeleteCommentTask_Success) {
  MockResponse& response = mHttpRequest->AddResponse("https://api.twitch.tv/kraken/videos/comments/testCommentId")
                             .SetType(HTTP_DELETE_REQUEST)
                             .SetStatusCode(200);

  bool callbackCalled = false;
  std::shared_ptr<ChatDeleteCommentTask> task;
  ChatDeleteCommentTask::Callback callback = [&callbackCalled, &task](ChatDeleteCommentTask* source, TTV_ErrorCode ec) {
    callbackCalled = true;
    ASSERT_EQ(task.get(), source);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
  };

  auto checkFunction = [&callbackCalled]() -> bool { return callbackCalled; };

  auto pollFunction = [this]() { Update(); };

  task = std::make_shared<ChatDeleteCommentTask>("testCommentId", "testAuthToken", std::move(callback));
  mTaskRunner->AddTask(task);

  EXPECT_TRUE(WaitUntilResultWithPollTask(1000, checkFunction, pollFunction));
  response.AssertRequestsMade();
}
