
#include "chattestmanager.h"

#include "twitchsdk/core/systemclock.h"

#include <fstream>
#include <iostream>
#include <sstream>
#include <streambuf>

using namespace ttv;
using namespace ttv::chat;

ChatTestManager::ChatTestManager() {
  mChatTestApiState = std::make_shared<ChatTestApiState>();
}

ChatTestManager::~ChatTestManager() {}

std::shared_ptr<ChatTestApiState> ChatTestManager::GetChatApiState() {
  return mChatTestApiState;
}

bool ChatTestManager::ChannelExists(UserId userId, ChannelId channelId) {
  auto iter = mChatChannelStateMap.find(userId);
  if (iter == mChatChannelStateMap.end()) {
    return false;
  }

  return iter->second.find(channelId) != iter->second.end();
}

ChatChannelInfo ChatTestManager::GetChatChannelInfoFor(UserId userId, ChannelId channelId) {
  auto iter = mChatChannelInfoMap.find(userId);

  TTV_ASSERT(iter != mChatChannelInfoMap.end());
  TTV_ASSERT(iter->second.find(channelId) != iter->second.end());

  return iter->second[channelId];
}

ChatChannelRestrictions ChatTestManager::GetChatChannelRestrictionsFor(UserId userId, ChannelId channelId) {
  auto iter = mChatChannelRestrictionsMap.find(userId);

  if ((iter != mChatChannelRestrictionsMap.end()) && (iter->second.find(channelId) != iter->second.end())) {
    return iter->second[channelId];
  } else {
    return ChatChannelRestrictions();
  }
}

ChatChannelState ChatTestManager::GetChatChannelStateForChannel(UserId userId, ChannelId channelId) {
  auto iter = mChatChannelStateMap.find(userId);
  if (iter == mChatChannelStateMap.end()) {
    return ChatChannelState::Disconnected;
  }

  auto innerIter = iter->second.find(channelId);
  if (innerIter == iter->second.end()) {
    return ChatChannelState::Disconnected;
  }

  return innerIter->second;
}

const std::vector<EmoticonSet>& ChatTestManager::GetChatUserEmoticonSets(ttv::UserId userId) {
  return mChatUserEmoticonSetsMap[userId];
}

bool ChatTestManager::HasMessageList(
  const std::vector<MessageInfo>& expectedMessageList, UserId userId, ChannelId channelId) {
  auto iter = mChatChannelMessages.find(userId);
  if (iter == mChatChannelMessages.end()) {
    return false;
  }

  auto innerIter = iter->second.find(channelId);
  if (innerIter == iter->second.end()) {
    return false;
  }

  auto messageList = innerIter->second;

  if (messageList.size() != expectedMessageList.size()) {
    return false;
  }

  for (uint32_t index = 0; index < messageList.size(); index++) {
    if (!AssertMessage(messageList[index].messageInfo, expectedMessageList[index])) {
      return false;
    }
  }

  return true;
}

bool ChatTestManager::AssertMessage(const MessageInfo& actual, const MessageInfo& expected) {
  bool metadataSame =
    (actual.flags == expected.flags) &&
    //(actual.nameColorARGB == expected.nameColorARGB) && // Don't compare colors since they can be random
    (actual.userMode == expected.userMode) && (actual.userName == expected.userName);

  if (!metadataSame) {
    return false;
  }

  if (actual.tokens.size() != expected.tokens.size()) {
    return false;
  }

  bool equalTokens = std::equal(actual.tokens.begin(), actual.tokens.end(), expected.tokens.begin(),
    expected.tokens.end(), [](const std::unique_ptr<MessageToken>& lhs, const std::unique_ptr<MessageToken>& rhs) {
      return TokensEqual(lhs, rhs);
    });

  if (!equalTokens) {
    return false;
  }

  for (uint32_t idx = 0; idx < actual.badges.size(); idx++) {
    if (actual.badges[idx].name != expected.badges[idx].name)
      return false;
    if (actual.badges[idx].version != expected.badges[idx].version)
      return false;
  }

  return true;
}

void ChatTestManager::FreeAllMessages() {
  mChatChannelMessages.clear();
}

void ChatTestManager::ModuleStateChanged(IModule* /*source*/, IModule::State state, TTV_ErrorCode ec) {
  mChatTestApiState->state = state;
  mChatTestApiState->stateResult = ec;
}

/*
 * IChatChannelListener implementation
 */
void ChatTestManager::ChatChannelStateChanged(
  UserId userId, ChannelId channelId, ChatChannelState state, TTV_ErrorCode /*ec*/) {
  mChatChannelStateMap[userId][channelId] = state;
}

void ChatTestManager::ChatChannelInfoChanged(UserId userId, ChannelId channelId, const ChatChannelInfo& channelInfo) {
  // This is disabled now, because channel info is being received after connecting, and before connection.
  // TTV_ASSERT(mChatChannelStateMap[channelName] == TTV_CHAT_CHANNEL_STATE_CONNECTED);

  mChatChannelInfoMap[userId][channelId] = channelInfo;
}

void ChatTestManager::ChatChannelRestrictionsChanged(
  UserId userId, ChannelId channelId, const ttv::chat::ChatChannelRestrictions& restrictions) {
  mChatChannelRestrictionsMap[userId][channelId] = restrictions;
}

void ChatTestManager::ChatChannelLocalUserChanged(
  UserId /*userId*/, ChannelId /*channelId*/, const ChatUserInfo& /*userInfo*/) {}

void ChatTestManager::ChatChannelMessagesReceived(
  UserId userId, ChannelId channelId, const std::vector<LiveChatMessage>& messageList) {
  mChatChannelMessages[userId][channelId] = messageList;
}

void ChatTestManager::ChatChannelSubscriptionNoticeReceived(
  UserId /*userId*/, ChannelId /*channelId*/, const ttv::chat::SubscriptionNotice& /*notice*/) {}

void ChatTestManager::ChatChannelFirstTimeChatterNoticeReceived(
  UserId /*userId*/, ChannelId /*channelId*/, const ttv::chat::FirstTimeChatterNotice& /*notice*/) {}

void ChatTestManager::ChatChannelRaidNoticeReceived(
  UserId /*userId*/, ChannelId /*channelId*/, const RaidNotice& /*notice*/) {}

void ChatTestManager::ChatChannelUnraidNoticeReceived(
  UserId /*userId*/, ChannelId /*channelId*/, const UnraidNotice& /*notice*/) {}

void ChatTestManager::ChatChannelGenericNoticeReceived(
  UserId /*userId*/, ChannelId /*channelId*/, const GenericMessageNotice& /*notice*/) {}

void ChatTestManager::ChatChannelMessagesCleared(UserId /*userId*/, ChannelId /*channelId*/) {}

void ChatTestManager::ChatChannelUserMessagesCleared(
  UserId /*userId*/, ChannelId /*channelId*/, UserId /*clearUserId*/) {}

void ChatTestManager::ChatChannelMessageDeleted(ttv::UserId /*userId*/, ttv::ChannelId /*channelId*/,
  std::string&& /*messageId*/, std::string&& /*senderLoginName*/, std::string&& /*messageContent*/) {}

void ChatTestManager::ChatChannelModNoticeUserTimedOut(UserId /*userId*/, ChannelId /*channelId*/,
  ModerationActionInfo&& /*modActionInfo*/, uint32_t /*timeoutDurationSeconds*/, std::string&& /*reason*/) {}

void ChatTestManager::ChatChannelModNoticeUserBanned(
  UserId /*userId*/, ChannelId /*channelId*/, ModerationActionInfo&& /*modActionInfo*/, std::string&& /*reason*/) {}

void ChatTestManager::ChatChannelModNoticeUserUntimedOut(
  UserId /*userId*/, ChannelId /*channelId*/, ModerationActionInfo&& /*modActionInfo*/) {}

void ChatTestManager::ChatChannelModNoticeUserUnbanned(
  UserId /*userId*/, ChannelId /*channelId*/, ModerationActionInfo&& /*modActionInfo*/) {}

void ChatTestManager::ChatChannelModNoticeMessageDeleted(ttv::UserId /*userId*/, ttv::ChannelId /*channelId*/,
  ttv::chat::ModerationActionInfo&& /*modActionInfo*/, std::string&& /*messageId*/, std::string&& /*message*/) {}

void ChatTestManager::ChatChannelModNoticeClearChat(
  ttv::UserId /*userId*/, ttv::ChannelId /*channelId*/, ttv::UserId /*modId*/, std::string&& /*modName*/) {}

void ChatTestManager::ChatChannelModNoticeEmoteOnly(
  ttv::UserId /*userId*/, ttv::ChannelId /*channelId*/, ttv::UserId /*modId*/, std::string&& /*modName*/) {}

void ChatTestManager::ChatChannelModNoticeEmoteOnlyOff(
  ttv::UserId /*userId*/, ttv::ChannelId /*channelId*/, ttv::UserId /*modId*/, std::string&& /*modName*/) {}

void ChatTestManager::ChatChannelModNoticeFollowersOnly(ttv::UserId /*userId*/, ttv::ChannelId /*channelId*/,
  ttv::UserId /*modId*/, std::string&& /*modName*/, uint32_t /*minimumFollowingDurationMinutes*/) {}

void ChatTestManager::ChatChannelModNoticeFollowersOnlyOff(
  ttv::UserId /*userId*/, ttv::ChannelId /*channelId*/, ttv::UserId /*modId*/, std::string&& /*modName*/) {}

void ChatTestManager::ChatChannelModNoticeR9K(
  ttv::UserId /*userId*/, ttv::ChannelId /*channelId*/, ttv::UserId /*modId*/, std::string&& /*modName*/) {}

void ChatTestManager::ChatChannelModNoticeR9KOff(
  ttv::UserId /*userId*/, ttv::ChannelId /*channelId*/, ttv::UserId /*modId*/, std::string&& /*modName*/) {}

void ChatTestManager::ChatChannelModNoticeSlow(ttv::UserId /*userId*/, ttv::ChannelId /*channelId*/,
  ttv::UserId /*modId*/, std::string&& /*modName*/, uint32_t /*slowModeDurationSeconds*/) {}

void ChatTestManager::ChatChannelModNoticeSlowOff(
  ttv::UserId /*userId*/, ttv::ChannelId /*channelId*/, ttv::UserId /*modId*/, std::string&& /*modName*/) {}

void ChatTestManager::ChatChannelModNoticeSubsOnly(
  ttv::UserId /*userId*/, ttv::ChannelId /*channelId*/, ttv::UserId /*modId*/, std::string&& /*modName*/) {}

void ChatTestManager::ChatChannelModNoticeSubsOnlyOff(
  ttv::UserId /*userId*/, ttv::ChannelId /*channelId*/, ttv::UserId /*modId*/, std::string&& /*modName*/) {}

void ChatTestManager::ChatChannelHostTargetChanged(
  UserId /*userId*/, ChannelId /*channelId*/, const std::string& /*targetChannel*/, uint32_t /*numViewers*/) {}

void ChatTestManager::ChatChannelNoticeReceived(UserId /*userId*/, ChannelId /*channelId*/, const std::string& /*id*/,
  const std::map<std::string, std::string>& /*params*/) {}

void ChatTestManager::AutoModCaughtSentMessage(ttv::UserId /*userId*/, ttv::ChannelId /*channelId*/) {}

void ChatTestManager::AutoModDeniedSentMessage(ttv::UserId /*userId*/, ttv::ChannelId /*channelId*/) {}

void ChatTestManager::AutoModApprovedSentMessage(ttv::UserId /*userId*/, ttv::ChannelId /*channelId*/) {}

void ChatTestManager::AutoModCaughtMessageForMods(ttv::UserId /*userId*/, ttv::ChannelId /*channelId*/,
  std::string&& /*messageId*/, std::string&& /*message*/, ttv::UserId /*senderId*/, std::string&& /*senderName*/,
  std::string&& /*reason*/) {}

void ChatTestManager::AutoModMessageApprovedByMod(ttv::UserId /*userId*/, ttv::ChannelId /*channelId*/,
  std::string&& /*messageId*/, ttv::UserId /*moderatorId*/, std::string&& /*moderatorName*/) {}

void ChatTestManager::AutoModMessageDeniedByMod(UserId /*userId*/, ChannelId /*channelId*/, std::string&& /*messageId*/,
  ttv::UserId /*moderatorId*/, std::string&& /*moderatorName*/) {}

void ChatTestManager::AutoModDeniedSentCheer(ttv::UserId /*userId*/, ttv::ChannelId /*channelId*/) {}

void ChatTestManager::AutoModTimedOutSentCheer(ttv::UserId /*userId*/, ttv::ChannelId /*channelId*/) {}

void ChatTestManager::AutoModCaughtCheerForMods(ttv::UserId /*userId*/, ttv::ChannelId /*channelId*/,
  std::string&& /*messageId*/, std::string&& /*message*/, ttv::UserId /*senderId*/, std::string&& /*senderName*/,
  std::string&& /*reason*/) {}

/*
 * ChatChannel Api Listener implementation
 */
void ChatTestManager::ChatUserEmoticonSetsChanged(UserId userId, const std::vector<EmoticonSet>& sets) {
  mChatUserEmoticonSetsMap[userId] = sets;
}
