/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/chattokenizationtest.h"
#include "testutilities.h"
#include "twitchsdk/chat/internal/chatmessageparsing.h"
#include "twitchsdk/core/systemclock.h"

#include <algorithm>
#include <iostream>
#include <numeric>
/*
 * Tests local tokenization with a default emoticon set.
 * See defaultemoticonimages.json for response details.
 * See chattokenizationtest to get user list
 */

using namespace ttv;
using namespace ttv::chat;
using namespace ttv::chat::test;

TEST_F(ChatTokenizationTest, LocalTokenizationSimpleRegexEmoticon) {
  TestTokenizeLocalMessage("dragonball z :)", "1:13-14");
}

TEST_F(ChatTokenizationTest, LocalTokenizationSimpleStringEmoticon) {
  TestTokenizeLocalMessage("dragonball z Volcania", "166:13-20");
}

TEST_F(ChatTokenizationTest, LocalTokenizationSimpleStringEmoticonModified) {
  // Test Emotes with modifiers
  TestTokenizeLocalMessage("WinWaker", "167:0-7");
  TestTokenizeLocalMessage("WinWaker_BW", "167_BW:0-10");

  // Test Emotes without modifiers
  TestTokenizeLocalMessage("Volcania", "166:0-7");
  TestTokenizeLocalMessage("Volcania_BW", "");
}

TEST_F(ChatTokenizationTest, LocalTokenizationUtf8StringEmoticon) {
  // utf8 [0-1]
  // emoticon [0-7]

  // length = 2 utf8 chars
  std::string utf8Message({(char)0xe5, (char)0xb1, (char)0x8c, (char)0xe5, (char)0xb1, (char)0x8c});

  TestTokenizeLocalMessage(utf8Message + " Volcania", "166:3-10");
  TestTokenizeLocalMessage(utf8Message + " Volcania " + utf8Message + " Volcania", "166:3-10,15-22");
  TestTokenizeLocalMessage(utf8Message + " Volcania " + utf8Message + " Volcania" + utf8Message, "166:3-10,15-22");
}

TEST_F(ChatTokenizationTest, LocalTokenizationInterspersedStringAndRegexEmoticons) {
  TestTokenizeLocalMessage("dragon :( ball StoneLightning z :P", "2:7-8/12:32-33/17:15-28");
  TestTokenizeLocalMessage("StoneLightning :( dragon ball z", "2:15-16/17:0-13");
}

TEST_F(ChatTokenizationTest, LocalTokenizationRangeSorting) {
  TestTokenizeLocalMessage(":( :P :(", "2:0-1,6-7/12:3-4");
}

TEST_F(ChatTokenizationTest, RegexWordsOnly) {
  TestTokenizeLocalMessage(":) :) :) :)", "1:0-1,3-4,6-7,9-10");
  TestTokenizeLocalMessage(":):) :):)", "");
  TestTokenizeLocalMessage(":) :):) :)", "1:0-1,8-9");
}

TEST_F(ChatTokenizationTest, ServerTokenizationSimpleStringEmoticon) {
  std::vector<std::unique_ptr<MessageToken>> expectedList;

  expectedList.push_back(std::make_unique<TextToken>("dragonballz "));
  expectedList.push_back(std::make_unique<EmoticonToken>("Volcania", "166"));

  TestTokenizeServerMessage("dragonballz Volcania", "166:12-19", "vegeta", std::move(expectedList));
}

TEST_F(ChatTokenizationTest, ServerTokenizationUrlCapitalization) {
  std::vector<std::unique_ptr<MessageToken>> expectedTokens;
  expectedTokens.push_back(std::make_unique<UrlToken>("https://www.zombo.com", false));
  TestTokenizeServerMessage("https://www.zombo.com", "", "vegeta", std::move(expectedTokens));

  expectedTokens.clear();
  expectedTokens.push_back(std::make_unique<UrlToken>("HTTPS://www.zombo.com", false));
  TestTokenizeServerMessage("HTTPS://www.zombo.com", "", "vegeta", std::move(expectedTokens));

  expectedTokens.clear();
  expectedTokens.push_back(std::make_unique<UrlToken>("HTTPS://WWW.ZOMBO.COM", false));
  TestTokenizeServerMessage("HTTPS://WWW.ZOMBO.COM", "", "vegeta", std::move(expectedTokens));
}

TEST_F(ChatTokenizationTest, ServerTokenizationUrlSymbols) {
  std::vector<std::unique_ptr<MessageToken>> expectedTokens;
  expectedTokens.push_back(std::make_unique<UrlToken>("https://www.zombo.com/@test", false));
  TestTokenizeServerMessage("https://www.zombo.com/@test", "", "vegeta", std::move(expectedTokens));

  expectedTokens.clear();
  expectedTokens.push_back(std::make_unique<UrlToken>("www.zombo.com/@test", false));
  TestTokenizeServerMessage("www.zombo.com/@test", "", "vegeta", std::move(expectedTokens));

  expectedTokens.clear();
  expectedTokens.push_back(std::make_unique<UrlToken>("zombo.com/@test", false));
  TestTokenizeServerMessage("zombo.com/@test", "", "vegeta", std::move(expectedTokens));

  expectedTokens.clear();
  expectedTokens.push_back(std::make_unique<TextToken>("test@zombo.com"));
  TestTokenizeServerMessage("test@zombo.com", "", "vegeta", std::move(expectedTokens));
}

TEST_F(ChatTokenizationTest, ServerTokenizationUrlVsNumber) {
  std::vector<std::unique_ptr<MessageToken>> expectedTokens;

  expectedTokens.clear();
  expectedTokens.push_back(std::make_unique<TextToken>("999.9"));
  TestTokenizeServerMessage("999.9", "", "vegeta", std::move(expectedTokens));

  expectedTokens.clear();
  expectedTokens.push_back(std::make_unique<TextToken>("999.99"));
  TestTokenizeServerMessage("999.99", "", "vegeta", std::move(expectedTokens));

  expectedTokens.clear();
  expectedTokens.push_back(std::make_unique<TextToken>("999.9com"));
  TestTokenizeServerMessage("999.9com", "", "vegeta", std::move(expectedTokens));

  expectedTokens.clear();
  expectedTokens.push_back(std::make_unique<TextToken>("abc.9"));
  TestTokenizeServerMessage("abc.9", "", "vegeta", std::move(expectedTokens));

  expectedTokens.clear();
  expectedTokens.push_back(std::make_unique<TextToken>("abc.99"));
  TestTokenizeServerMessage("abc.99", "", "vegeta", std::move(expectedTokens));

  expectedTokens.clear();
  expectedTokens.push_back(std::make_unique<TextToken>("abc.9com"));
  TestTokenizeServerMessage("abc.9com", "", "vegeta", std::move(expectedTokens));

  expectedTokens.clear();
  expectedTokens.push_back(std::make_unique<UrlToken>("abc.abc123", false));
  TestTokenizeServerMessage("abc.abc123", "", "vegeta", std::move(expectedTokens));

  expectedTokens.clear();
  expectedTokens.push_back(std::make_unique<UrlToken>("123.abc123", false));
  TestTokenizeServerMessage("123.abc123", "", "vegeta", std::move(expectedTokens));
}

TEST_F(ChatTokenizationTest, ServerTokenizationPermutationOfAllTokenTypes) {
  /*
   * stringTokens is a vector of tokens to permute, the url token includes
   * a string emoticon(www.Kappa.org). Website displays this as www.Kappa.com
   * and does not insert emoticon in place of Kappa. This url token in this
   * test, checks that our sdk follows the website format of display.
   */
  std::vector<std::string> stringTokens;
  stringTokens.push_back("dragonballz");
  stringTokens.push_back("@goku");
  stringTokens.push_back("Volcania");
  stringTokens.push_back("www.Kappa.org");

  /*
   * tokenMap is used to construct expectedTokenList.
   * text tokens have additional entries to take into account
   * spaces before or after being part of the same text token
   */

  std::map<std::string, std::unique_ptr<MessageToken>> tokenMap;
  tokenMap["dragonballz"] = std::make_unique<TextToken>("dragonballz");
  tokenMap["dragonballz "] = std::make_unique<TextToken>("dragonballz ");
  tokenMap[" dragonballz"] = std::make_unique<TextToken>(" dragonballz");
  tokenMap[" dragonballz "] = std::make_unique<TextToken>(" dragonballz ");
  tokenMap["@goku"] = std::make_unique<MentionToken>("goku", "@goku", false);
  tokenMap["Volcania"] = std::make_unique<EmoticonToken>("Volcania", "166");
  tokenMap["www.Kappa.org"] = std::make_unique<UrlToken>("www.Kappa.org", false);
  tokenMap[" "] = std::make_unique<TextToken>(" ");

  /*
   * Test all permutations. Insert space character between tokens.
   */

  do {
    std::string emotesMessageTag;
    std::string badgesMessageTag;
    std::string channelName = "";
    std::string message = std::accumulate(
      stringTokens.begin(), stringTokens.end(), std::string(""), [](const std::string& init, const std::string& curr) {
        if (init == "")
          return init + curr;
        return init + " " + curr;
      });

    // TokenizeLocalMessage returns the emotesMessageTag used in the test.
    std::shared_ptr<User> user = mUserRepository->GetUser(mTestUserId);
    TokenizeLocalMessage(user, 1001, message, emotesMessageTag, badgesMessageTag);

    std::vector<std::unique_ptr<MessageToken>> expectedTokens;

    for (uint idx = 0; idx < stringTokens.size(); idx++) {
      /*
       * For a TTV_CHAT_MSGTOKEN_TEXT, insert a space after
       * if appears at idx 0, insert a space before, if appears
       * at last position, insert a space before and after for
       * other positions. If Test is modified, to include interspersed
       * text tokens, logic needs to be modified, or tests will fail.
       */

      if (tokenMap.at(stringTokens[idx])->GetType() == MessageToken::Type::Text) {
        if (idx == 0) {
          expectedTokens.push_back(tokenMap.at(stringTokens[idx] + " ")->Clone());
        } else if (idx == stringTokens.size() - 1) {
          expectedTokens.push_back(tokenMap.at(" " + stringTokens[idx])->Clone());
        } else {
          expectedTokens.push_back(tokenMap.at(" " + stringTokens[idx] + " ")->Clone());
        }
      } else {
        expectedTokens.push_back(tokenMap.at(stringTokens[idx])->Clone());
        /*
         * For expectedTokens, push a text token for spaces for non text tokens
         */
        if (idx != stringTokens.size() - 1 &&
            tokenMap.at(stringTokens[idx + 1])->GetType() != MessageToken::Type::Text) {
          expectedTokens.push_back(tokenMap.at(" ")->Clone());
        }
      }
    }

    TestTokenizeServerMessage(message, emotesMessageTag, "vegeta", std::move(expectedTokens));
  } while (std::next_permutation(stringTokens.begin(), stringTokens.end()));
}

TEST_F(ChatTokenizationTest, BitsParsing) {
  std::vector<ttv::chat::BitsConfiguration::Cheermote> cheermotes;
  ttv::chat::BitsConfiguration::Cheermote uppercaseAction;
  uppercaseAction.prefix = "MVPBlack";
  cheermotes.push_back(uppercaseAction);

  ttv::chat::BitsConfiguration::Cheermote cheerAction;
  cheerAction.prefix = "cheer";
  cheermotes.push_back(cheerAction);

  ttv::chat::BitsConfiguration::Cheermote swiftrageAction;
  swiftrageAction.prefix = "swiftrage";
  cheermotes.push_back(swiftrageAction);

  auto bitsConfig = std::make_shared<ttv::chat::BitsConfiguration>(std::move(cheermotes), 0, 0);

  std::vector<std::unique_ptr<MessageToken>> expectedList;

  expectedList.push_back(std::make_unique<BitsToken>("MVPBlack", 18));
  expectedList.push_back(std::make_unique<TextToken>(" MVPBlack MVPBlack20d MVPBlacke20 mvPBlack0 cheer-32 "));
  expectedList.push_back(std::make_unique<BitsToken>("mVpBlaCk", 20));
  expectedList.push_back(std::make_unique<TextToken>(" hey "));
  expectedList.push_back(std::make_unique<BitsToken>("cheer", 32));
  expectedList.push_back(std::make_unique<TextToken>(" dude "));
  expectedList.push_back(std::make_unique<BitsToken>("swiftrage", 24));
  expectedList.push_back(std::make_unique<TextToken>(" bye"));

  TestTokenizeServerMessage(
    "MVPBlack18 MVPBlack MVPBlack20d MVPBlacke20 mvPBlack0 cheer-32 mVpBlaCk20 hey cheer32 dude swiftrage24 bye", "",
    bitsConfig, 56, "vegeta", std::move(expectedList));
}

TEST_F(ChatTokenizationTest, UserMentionsWithoutAtSymbol) {
  TokenizationOptions options;
  options.mentions = true;
  MessageInfo tokenizedMessage;
  std::vector<std::string> localUserNames = {"twitch", "TwitchDisplayName"};

  ttv::chat::TokenizeServerMessage("This is a mention for Twitch and TwitchDisplayName PogChamp", options, "", nullptr,
    localUserNames, tokenizedMessage);
  ASSERT_EQ(5, tokenizedMessage.tokens.size());

  auto& token = tokenizedMessage.tokens[1];
  ASSERT_EQ(MessageToken::Type::Mention, token->GetType());

  MentionToken* mentionToken = static_cast<MentionToken*>(token.get());
  ASSERT_EQ("Twitch", mentionToken->userName);
  ASSERT_EQ("Twitch", mentionToken->text);
  ASSERT_TRUE(mentionToken->isLocalUser);

  auto& token2 = tokenizedMessage.tokens[3];
  ASSERT_EQ(MessageToken::Type::Mention, token2->GetType());

  mentionToken = static_cast<MentionToken*>(token2.get());
  ASSERT_EQ("TwitchDisplayName", mentionToken->userName);
  ASSERT_EQ("TwitchDisplayName", mentionToken->text);
  ASSERT_TRUE(mentionToken->isLocalUser);
}

TEST_F(ChatTokenizationTest, UserMentionsUTF8WithoutAtSymbol) {
  TokenizationOptions options;
  options.mentions = true;
  MessageInfo tokenizedMessage;
  std::vector<std::string> localUserNames = {
    "normal_name", "\x61\xE0\xA4\xAE\xE0\xA4\xB9\xE0\xA4\xB8\xE0\xA5\x81\xE0\xA4\xB8\x61"};

  ttv::chat::TokenizeServerMessage(
    "This is a mention for normal_name and \x61\xE0\xA4\xAE\xE0\xA4\xB9\xE0\xA4\xB8\xE0\xA5\x81\xE0\xA4\xB8\x61 PogChamp",
    options, "", nullptr, localUserNames, tokenizedMessage);
  ASSERT_EQ(5, tokenizedMessage.tokens.size());

  auto& token = tokenizedMessage.tokens[1];
  ASSERT_EQ(MessageToken::Type::Mention, token->GetType());

  MentionToken* mentionToken = static_cast<MentionToken*>(token.get());
  ASSERT_EQ("normal_name", mentionToken->userName);
  ASSERT_EQ("normal_name", mentionToken->text);
  ASSERT_TRUE(mentionToken->isLocalUser);

  auto& token2 = tokenizedMessage.tokens[3];
  ASSERT_EQ(MessageToken::Type::Mention, token2->GetType());

  mentionToken = static_cast<MentionToken*>(token2.get());
  ASSERT_EQ("\x61\xE0\xA4\xAE\xE0\xA4\xB9\xE0\xA4\xB8\xE0\xA5\x81\xE0\xA4\xB8\x61", mentionToken->userName);
  ASSERT_EQ("\x61\xE0\xA4\xAE\xE0\xA4\xB9\xE0\xA4\xB8\xE0\xA5\x81\xE0\xA4\xB8\x61", mentionToken->text);
  ASSERT_TRUE(mentionToken->isLocalUser);
}

TEST_F(ChatTokenizationTest, UserMentionsComplicated) {
  TokenizationOptions options;
  options.mentions = true;
  MessageInfo tokenizedMessage;
  std::vector<std::string> localUserNames = {"twitch", "TwitchDisplayName"};

  ttv::chat::TokenizeServerMessage(
    "twitchdisplayname and \"'\\{<*@twitch@twitch! should both work but not prefix@twitch or twitch, or *twitch or @twitchdisplayname_n",
    options, "", nullptr, localUserNames, tokenizedMessage);
  ASSERT_EQ(5,
    tokenizedMessage.tokens.size());  // "twitchdisplayname" " and \"'\\{<*" "@twitch" "@twitch! should both work but
                                      // not prefix@twitch or twitch, or *twitch or " "@twitchdisplayname_n"

  auto& token = tokenizedMessage.tokens[0];
  ASSERT_EQ(MessageToken::Type::Mention, token->GetType());
  MentionToken* mentionToken = static_cast<MentionToken*>(token.get());
  ASSERT_EQ("twitchdisplayname", mentionToken->userName);
  ASSERT_EQ("twitchdisplayname", mentionToken->text);
  ASSERT_TRUE(mentionToken->isLocalUser);

  auto& token2 = tokenizedMessage.tokens[2];
  ASSERT_EQ(MessageToken::Type::Mention, token2->GetType());
  mentionToken = static_cast<MentionToken*>(token2.get());
  ASSERT_EQ("twitch", mentionToken->userName);
  ASSERT_EQ("@twitch", mentionToken->text);
  ASSERT_TRUE(mentionToken->isLocalUser);

  auto& token3 = tokenizedMessage.tokens[4];
  ASSERT_EQ(MessageToken::Type::Mention, token3->GetType());
  mentionToken = static_cast<MentionToken*>(token3.get());
  ASSERT_EQ("twitchdisplayname_n", mentionToken->userName);
  ASSERT_EQ("@twitchdisplayname_n", mentionToken->text);
  ASSERT_FALSE(mentionToken->isLocalUser);
}

TEST_F(ChatTokenizationTest, UserMentionsAndUrl) {
  TokenizationOptions options;
  options.mentions = true;
  options.urls = true;
  MessageInfo tokenizedMessage;
  std::vector<std::string> localUserNames = {"twitch"};

  ttv::chat::TokenizeServerMessage("Hey @twitch! Check out user@twitch.tv and https://twitch.tv", options, "", nullptr,
    localUserNames, tokenizedMessage);
  ASSERT_EQ(
    4, tokenizedMessage.tokens.size());  // "Hey " "@twitch" "! Check out user@twitch.tv and " "https://twitch.tv"

  auto& token = tokenizedMessage.tokens[1];
  ASSERT_EQ(MessageToken::Type::Mention, token->GetType());
  MentionToken* mentionToken = static_cast<MentionToken*>(token.get());
  ASSERT_EQ("twitch", mentionToken->userName);
  ASSERT_EQ("@twitch", mentionToken->text);
  ASSERT_TRUE(mentionToken->isLocalUser);

  auto& token2 = tokenizedMessage.tokens[3];
  ASSERT_EQ(MessageToken::Type::Url, token2->GetType());
  UrlToken* urlToken = static_cast<UrlToken*>(token2.get());
  ASSERT_EQ("https://twitch.tv", urlToken->url);
}

TEST_F(ChatTokenizationTest, AutoModFlags) {
  TokenizationOptions options;
  options.mentions = true;
  options.urls = true;
  MessageInfo tokenizedMessage;

  ttv::chat::TokenizeServerMessage(
    "hey badword1 and also badword2", options, "", "0-1:,4-11:S.4/A.6/P.7,22-29:I.1", nullptr, {}, tokenizedMessage);
  ASSERT_EQ(4, tokenizedMessage.tokens.size());  // "hey " "badword1" " and also " "badword2"

  auto& token = tokenizedMessage.tokens[1];
  ASSERT_EQ(MessageToken::Type::Text, token->GetType());
  TextToken* textToken = static_cast<TextToken*>(token.get());
  ASSERT_EQ("badword1", textToken->text);
  ASSERT_EQ(4, textToken->autoModFlags.sexualLevel);
  ASSERT_EQ(6, textToken->autoModFlags.aggressiveLevel);
  ASSERT_EQ(7, textToken->autoModFlags.profanityLevel);
  ASSERT_EQ(0, textToken->autoModFlags.identityLevel);

  auto& token2 = tokenizedMessage.tokens[2];
  ASSERT_EQ(MessageToken::Type::Text, token2->GetType());
  textToken = static_cast<TextToken*>(token2.get());
  ASSERT_EQ(" and also ", textToken->text);
  ASSERT_EQ(0, textToken->autoModFlags.sexualLevel);
  ASSERT_EQ(0, textToken->autoModFlags.aggressiveLevel);
  ASSERT_EQ(0, textToken->autoModFlags.profanityLevel);
  ASSERT_EQ(0, textToken->autoModFlags.identityLevel);

  auto& token3 = tokenizedMessage.tokens[3];
  ASSERT_EQ(MessageToken::Type::Text, token3->GetType());
  textToken = static_cast<TextToken*>(token3.get());
  ASSERT_EQ("badword2", textToken->text);
  ASSERT_EQ(0, textToken->autoModFlags.sexualLevel);
  ASSERT_EQ(0, textToken->autoModFlags.aggressiveLevel);
  ASSERT_EQ(0, textToken->autoModFlags.profanityLevel);
  ASSERT_EQ(1, textToken->autoModFlags.identityLevel);
}

TEST_F(ChatTokenizationTest, AutoModFlagsUnicode) {
  TokenizationOptions options;
  options.mentions = true;
  options.urls = true;
  MessageInfo tokenizedMessage;
  // "abc[utf8]de"
  ttv::chat::TokenizeServerMessage(
    {'a', 'b', 'c', (char)0xe5, (char)0xb1, (char)0x8c, (char)0xe5, (char)0xb1, (char)0x8c, 'd', 'e'}, options, "",
    "3-4:S.4/A.6/P.7", nullptr, {}, tokenizedMessage);

  ASSERT_EQ(3, tokenizedMessage.tokens.size());  // "abc" "[utf8]" "de"

  auto& token0 = tokenizedMessage.tokens[0];
  ASSERT_EQ(MessageToken::Type::Text, token0->GetType());
  TextToken* textToken = static_cast<TextToken*>(token0.get());
  ASSERT_EQ("abc", textToken->text);
  ASSERT_EQ(0, textToken->autoModFlags.sexualLevel);
  ASSERT_EQ(0, textToken->autoModFlags.aggressiveLevel);
  ASSERT_EQ(0, textToken->autoModFlags.profanityLevel);
  ASSERT_EQ(0, textToken->autoModFlags.identityLevel);

  auto& token1 = tokenizedMessage.tokens[1];
  ASSERT_EQ(MessageToken::Type::Text, token1->GetType());
  textToken = static_cast<TextToken*>(token1.get());
  ASSERT_EQ(std::string({(char)0xe5, (char)0xb1, (char)0x8c, (char)0xe5, (char)0xb1, (char)0x8c}), textToken->text);
  ASSERT_EQ(4, textToken->autoModFlags.sexualLevel);
  ASSERT_EQ(6, textToken->autoModFlags.aggressiveLevel);
  ASSERT_EQ(7, textToken->autoModFlags.profanityLevel);
  ASSERT_EQ(0, textToken->autoModFlags.identityLevel);

  auto& token2 = tokenizedMessage.tokens[2];
  ASSERT_EQ(MessageToken::Type::Text, token2->GetType());
  textToken = static_cast<TextToken*>(token2.get());
  ASSERT_EQ("de", textToken->text);
  ASSERT_EQ(0, textToken->autoModFlags.sexualLevel);
  ASSERT_EQ(0, textToken->autoModFlags.aggressiveLevel);
  ASSERT_EQ(0, textToken->autoModFlags.profanityLevel);
  ASSERT_EQ(0, textToken->autoModFlags.identityLevel);
}
