/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/chatapitest.h"

#include "testtransport.h"
#include "testutilities.h"
#include "twitchsdk/chat/internal/ircstring.h"

using namespace ttv;
using namespace ttv::chat;

ttv::chat::test::ChatApiTest::ChatApiTest()
    : mChatTestManager(std::make_shared<ChatTestManager>()),
      mCoreApi(std::make_shared<CoreAPI>()),
      mChatApi(std::make_shared<ChatAPI>()) {
  mChatApi->SetTokenizationOptions(TokenizationOptions::All());
}

ttv::chat::test::ChatApiTest::~ChatApiTest() {}

void ttv::chat::test::ChatApiTest::SetUpStubs() {
  ChatBaseTest::SetUpStubs();

  mChatApi->SetChatObjectFactory(mTestFactory);
}

void ttv::chat::test::ChatApiTest::SetUpComponents() {
  ChatBaseTest::SetUpComponents();

  ttv::test::InitializeModule(mCoreApi);

  mChatApi->SetCoreApi(mCoreApi);
  mChatTestManager->SetChatApi(mChatApi);

  AddModule(mCoreApi);
  AddModule(mChatApi);
}

void ttv::chat::test::ChatApiTest::TearDownComponents() {
  ShutdownModulesSync({mChatApi, mCoreApi});

  SdkBaseTest::TearDownComponents();
}

TTV_ErrorCode ttv::chat::test::ChatApiTest::LogIn(const TestParams& testParams) {
  return ttv::test::SdkBaseTest::LogIn(mCoreApi, testParams.oauth, testParams.userInfo);
}

TTV_ErrorCode ttv::chat::test::ChatApiTest::LogOut(const TestParams& testParams) {
  return ttv::test::SdkBaseTest::LogOut(mCoreApi, testParams.userInfo);
}

TTV_ErrorCode ttv::chat::test::ChatApiTest::ConnectToChannel(const TestParams& testParams) {
  return mChatApi->Connect(
    testParams.userInfo.userId, testParams.channelId, std::static_pointer_cast<IChatChannelListener>(mChatTestManager));
}

TTV_ErrorCode ttv::chat::test::ChatApiTest::DisconnectFromChannel(const TestParams& testParams) {
  return mChatApi->Disconnect(testParams.userInfo.userId, testParams.channelId);
}

TTV_ErrorCode ttv::chat::test::ChatApiTest::ShutDown() {
  return mChatApi->Shutdown(nullptr);
}

void ttv::chat::test::ChatApiTest::SendChatMessage(const TestParams& testParams, std::string message) {
  TTV_ErrorCode sentSuccess = mChatApi->SendChatMessage(testParams.userInfo.userId, testParams.channelId, message);
  EXPECT_EQ(TTV_EC_SUCCESS, sentSuccess);
}

void ttv::chat::test::ChatApiTest::TestChatApiInitializationCallback(TokenizationOptions tokenizationOptions) {
  if (tokenizationOptions.emoticons) {
    // mock emoticons set data
    mHttpRequest->AddResponse("https://api.twitch.tv/kraken/chat/emoticon_images")
      .AddRequestParam("emotesets", "0")
      .SetResponseBodyFromFile("chat/defaultemoticonimages.json");

    //      TODO: when needed add badge response
    //      std::string badgesUrl = "https://api.twitch.tv/kraken/chat/badges";
    //      testFactory->mChatManager->SetResponseForUrlFromFile(badgesUrl, HTTP_GET_REQUEST, ParamMap(), ParamMap(),
    //      200, ParamMap(), "pokemonbadges.json");
  }

  // Initialize ChatAPI with chat api listener
  mChatApi->SetTokenizationOptions(tokenizationOptions);
  mChatApi->SetListener(mChatTestManager);

  ttv::test::InitializeModule(mChatApi);
}

void ttv::chat::test::ChatApiTest::TestChatApiShutdownCallback() {
  std::function<bool()> chatApiResults = [this]() { return mChatApi->GetState() == ChatAPI::State::Uninitialized; };

  WaitUntilResult(3000, chatApiResults);
}

void ttv::chat::test::ChatApiTest::TestChatChannelStateCallback(
  const TestParams& testParams, ChatChannelState expectedState, uint waitUntilMilliSeconds) {
  std::function<bool()> chatChannelStateReached = [this, &expectedState, &testParams]() {
    if (mChatTestManager->ChannelExists(testParams.userInfo.userId, testParams.channelId)) {
      ChatChannelState chatChannelState =
        mChatTestManager->GetChatChannelStateForChannel(testParams.userInfo.userId, testParams.channelId);
      if (chatChannelState == expectedState) {
        return true;
      }
    }
    return false;
  };

  bool expectedStateReached = WaitUntilResult(waitUntilMilliSeconds, chatChannelStateReached);
  EXPECT_TRUE(expectedStateReached);
}

void ttv::chat::test::ChatApiTest::TestIrcHandshake(const TestParams& testParams) {
  std::vector<std::string> loginMessages;
  loginMessages.push_back("CAP REQ :twitch.tv/tags twitch.tv/commands\r\n");  // Capability message
  loginMessages.push_back("PASS oauth:<AUTHTOKEN>\r\n");                      // Password message
  loginMessages.push_back("NICK <USERNAME>\r\n");                             // Nickname message

  for (auto& msg : loginMessages) {
    TestClientSentMessage(SubstituteMessage(testParams, msg), 3000);
  }

  // simulate login messages from server
  std::vector<std::string> serverLoginMessages;
  serverLoginMessages.push_back(
    ":tmi.twitch.tv CAP * ACK :twitch.tv/tags twitch.tv/commands\r\n");                // capability acknowledment
  serverLoginMessages.push_back(":tmi.twitch.tv 001 <USERNAME> :Welcome, GLHF!\r\n");  // welcome msg
  serverLoginMessages.push_back(":tmi:twitch.tv 002 <USERNAME> :Your host is tmi.twitch.tv\r\n");  // host message
  serverLoginMessages.push_back(":tmi.twitch.tv 003 <USERNAME> :This server is rather new\r\n");   // created msg
  serverLoginMessages.push_back(":tmi.twitch.tv 004 <USERNAME> :-\r\n");                           // post reg greeting
  serverLoginMessages.push_back(":tmi.twitch.tv 375 <USERNAME> :-\r\n");                           // motd start
  serverLoginMessages.push_back(
    ":tmi.twitch.tv 372 <USERNAME> :You are in a maze of twisty passages, all alike.\r\n");  // mod reply
  serverLoginMessages.push_back(":tmi.twitch.tv 376 <USERNAME> :>\r\n");                     // motd end
  serverLoginMessages.push_back(":<USERNAME>!<USERNAME>@<USERNAME>.tmi.twitch.tv JOIN #<CHANNELNAME>\r\n");  // join msg

  for (auto& msg : serverLoginMessages) {
    mTransport->EnqueueIncomingData(SubstituteMessage(testParams, msg));
  };

  // wait for join message
  std::string joinMsg = SubstituteMessage(testParams, "JOIN #<CHANNELNAME>\r\n");

  TestClientSentMessage(joinMsg, 1000);

  std::vector<std::string> serverJoinMessages;
  serverJoinMessages.push_back(":tmi:twitch.tv 002 <USERNAME> :Your host is tmi.twitch.tv\r\n");  // host message
  serverJoinMessages.push_back(":tmi.twitch.tv 003 <USERNAME> :This server is rather new\r\n");   // created msg
  serverJoinMessages.push_back(":tmi.twitch.tv 004 <USERNAME> :-\r\n");                           // post reg greeting
  serverJoinMessages.push_back(":tmi.twitch.tv 375 <USERNAME> :-\r\n");                           // motd start
  serverJoinMessages.push_back(
    ":tmi.twitch.tv 372 <USERNAME> :You are in a maze of twisty passages, all alike.\r\n");  // mod reply
  serverJoinMessages.push_back(":tmi.twitch.tv 376 <USERNAME> :>\r\n");                      // motd end
  serverJoinMessages.push_back(
    ":<USERNAME>!<USERNAME>.tmi.twitch.tv JOIN #<CHANNELNAME>\r\n:<USERNAME>.tmi.twitch.tv 353 <USERNAME> = #<CHANNELNAME> :<USERNAME>\r\n:<USERNAME>.tmi.twitch.tv 366 <USERNAME> #<CHANNELNAME> :End of /NAMES list\r\n:jtv MODE #tsm_theoddone +o nfmanewhope\r\n:jtv MODE #<CHANNELNAME> +o xanbot\r\n:jtv MODE #tsm_theoddone +o immianthe\r\n:jtv MODE #<CHANNELNAME> +o tsm_theoddone\r\n:jtv MODE #tsm_theoddone +o nightbot\r\n:jtv MODE #<CHANNELNAME> +o tarfu\r\n@color=;display-name=<USERNAME>;emote-sets=0;emotesets=0;subscriber=0;turbo=0;user-type=;user_type= :tmi.twitch.tv USERSTATE #<CHANNELNAME>\r\n");  // join msg
  serverJoinMessages.push_back(
    ":<USERNAME>.tmi.twitch.tv 353 <USERNAME> = #<CHANNELNAME> :<USERNAME>\r\n");  // reply to names
  serverJoinMessages.push_back(
    ":<USERNAME>.tmi.twitch.tv 366 <USERNAME> #<CHANNELNAME> :End of /NAMES list\r\n");  // end of names

  for (auto& msg : serverJoinMessages) {
    mTransport->EnqueueIncomingData(SubstituteMessage(testParams, msg));
  };
}

std::string ttv::chat::test::ChatApiTest::SubstituteMessage(const TestParams& testParams, const std::string& msg) {
  std::string authSubstituted = ReplaceSubstring(msg, "<AUTHTOKEN>", std::string(testParams.oauth));

  std::string usernameSubstituted =
    ReplaceSubstring(authSubstituted, "<USERNAME>", std::string(testParams.userInfo.userName));

  return ttv::chat::ReplaceSubstring(usernameSubstituted, "<CHANNELNAME>", std::string(testParams.channelName));
}

void ttv::chat::test::ChatApiTest::TestClientSentMessage(std::string message, uint waitUntilMilliSeconds) {
  std::function<bool()> waitFunc = [this] { return mTransport->HasOutgoingData(); };

  bool clientHasMessages = WaitUntilResult(waitUntilMilliSeconds, waitFunc);
  /*
   * If no messages could be found before timer expires, fail test
   */

  ASSERT_TRUE(clientHasMessages);

  std::string topMsg = mTransport->NextOutgoingData();
  EXPECT_EQ(message, topMsg);
}

void ttv::chat::test::ChatApiTest::TestClientSentPrivMessage(
  const TestParams& testParams, std::string message, uint waitUntilMilliseconds) {
  std::string privMsg = SubstituteMessage(testParams, "PRIVMSG #<CHANNELNAME> :") + message + "\r\n";
  TestClientSentMessage(privMsg, waitUntilMilliseconds);
}

void ttv::chat::test::ChatApiTest::TestMessageCallback(
  const TestParams& testParams, std::vector<MessageInfo> msgList, uint waitUntilMilliSeconds) {
  std::function<bool()> gotCallback = std::bind(
    &ChatTestManager::HasMessageList, mChatTestManager, msgList, testParams.userInfo.userId, testParams.channelId);
  bool messageListSuccess = WaitUntilResult(waitUntilMilliSeconds, gotCallback);
  EXPECT_TRUE(messageListSuccess);
}
