/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/chatmessagetest.h"

#include "testutilities.h"
#include "twitchsdk/chat/internal/ircstring.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/thread.h"

using namespace ttv;
using namespace ttv::chat;

#if WIN32
#define snprintf _snprintf
#endif

void ttv::chat::test::ChatMessageTest::SetUpStubs() {
  ChatBaseTest::SetUpStubs();
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "userID"}, "1001")
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatapi_channel_info.json")
    .Done();
  mHttpRequest->AddResponse("https://api.twitch.tv/kraken/chat/emoticon_images")
    .AddRequestParam("emotesets", "0")
    .SetResponseBodyFromFile("chat/defaultemoticonimages.json");
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("chat/pokemonchatproperties.json")
    .SetType(HTTP_POST_REQUEST)
    .Done();
}

void ttv::chat::test::ChatMessageTest::SetUpComponents() {
  ChatBaseTest::SetUpComponents();

  mTransport = std::make_shared<TestTransport>();
  mTestFactory->SetTransport(std::static_pointer_cast<IChatTransport>(mTransport));

  mCoreApi = std::make_shared<CoreAPI>();
  ttv::test::InitializeModule(mCoreApi);

  mChatApi = std::make_shared<ChatAPI>();
  mChatApi->SetCoreApi(mCoreApi);
  mChatApi->SetChatObjectFactory(std::static_pointer_cast<IChatObjectFactory>(mTestFactory));

  // Initialize chat
  mChatApi->SetTokenizationOptions(TokenizationOptions::All());
  mChatChannelListenerProxy = std::make_shared<ttv::chat::ChatChannelListenerProxy>();
  ttv::test::InitializeModule(mChatApi);

  AddModule(mCoreApi);
  AddModule(mChatApi);
}

void ttv::chat::test::ChatMessageTest::TearDownComponents() {
  ShutdownModulesSync({mChatApi, mCoreApi});

  ChatBaseTest::TearDownComponents();
}

void ttv::chat::test::ChatMessageTest::SendMessage(const std::string& message) {
  std::string suffix = "\r\n";
  ChannelId channelId = 1001;

  // Login with the test user
  std::string authToken = "auth_token";
  UserInfo userInfo;
  userInfo.userId = 9001;
  userInfo.userName = "pikachu";
  userInfo.displayName = "Pikachu";

  ASSERT_TRUE(TTV_SUCCEEDED(LogIn(mCoreApi, authToken, userInfo)));

  // Kick off a connect
  ASSERT_EQ(mChatApi->Connect(
              userInfo.userId, channelId, std::static_pointer_cast<IChatChannelListener>(mChatChannelListenerProxy)),
    TTV_EC_SUCCESS);

  mTransport->SetThreadSafe(true);

  std::function<void()> updateFunction = [this]() {
    ASSERT_EQ(mCoreApi->Update(), TTV_EC_SUCCESS);
    ASSERT_EQ(mChatApi->Update(), TTV_EC_SUCCESS);
  };

  // Fake the minimum number of commands needed to connect successfully
  mTransport->EnqueueIncomingData(":tmi.twitch.tv CAP * ACK :twitch.tv/tags twitch.tv/commands\r\n");
  mTransport->EnqueueIncomingData(":tmi.twitch.tv 001 pikachu :Welcome, GLHF!\r\n");

  EXPECT_TRUE(ttv::test::WaitUntilResultWithPollTask(2000,
    [this]() {
      while (mTransport->HasOutgoingData()) {
        if (mTransport->NextOutgoingData() == "JOIN #pokemon\r\n") {
          return true;
        }
      }
      return false;
    },
    updateFunction));

  auto channelStateChangedFunc = [this](UserId /*userId*/, ChannelId channelId, ttv::chat::ChatChannelState state,
                                   TTV_ErrorCode /*ec*/) { mChannelStates[channelId] = state; };

  mChatChannelListenerProxy->chatChannelStateChangedCallback = channelStateChangedFunc;

  mTransport->EnqueueIncomingData(":pikachu!pikachu@pikachu.tmi.twitch.tv JOIN #pokemon\r\n");
  mTransport->EnqueueIncomingData(
    "@color=#5F9EA0;display-name=pikachu;emote-sets=0,42,793,2126,10947;subscriber=0;turbo=0;user-type=mod :tmi.twitch.tv USERSTATE #pokemon\r\n");

  std::function<bool()> checkChannelConnected = [this, &channelId]() {
    auto iter = mChannelStates.find(channelId);
    if (iter != mChannelStates.end()) {
      ChatChannelState chatChannelState = iter->second;
      if (chatChannelState == ChatChannelState::Connected) {
        return true;
      }
    }
    return false;
  };

  EXPECT_TRUE(ttv::test::WaitUntilResultWithPollTask(10000, checkChannelConnected, updateFunction));

  mTransport->EnqueueIncomingData(message);
  ASSERT_EQ(mChatApi->Update(), TTV_EC_SUCCESS);
}
