/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/chatstresstest.h"

#include "testutilities.h"
#include "twitchsdk/chat/internal/ircstring.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/thread.h"

using namespace ttv;
using namespace ttv::chat;

#if WIN32
#define snprintf _snprintf
#endif

ttv::chat::test::ChatStressTest::ChatStressTest()
    : mMessageFreeDelay(0), mLastMessageReceivedTime(0), mNumMessagesReceived(0), mNumMessageBatchesReceived(0) {}

void ttv::chat::test::ChatStressTest::SetUpStubs() {
  ChatBaseTest::SetUpStubs();
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "userID"}, "1001")
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatapi_channel_info.json")
    .Done();
  mHttpRequest->AddResponse("https://api.twitch.tv/kraken/chat/emoticon_images")
    .AddRequestParam("emotesets", "0")
    .SetResponseBodyFromFile("chat/defaultemoticonimages.json");
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .AddRequestRegex(std::regex("FetchChannelCheermotes"))
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("chat/pokemonchatproperties.json")
    .SetType(HTTP_POST_REQUEST)
    .Done();

  mHttpRequest->AddResponse("https://tmi.twitch.tv/group/user/dragonballz/chatters")
    .SetResponseBodyFromFile("chat/dragonballzuserlist.json");
}

void ttv::chat::test::ChatStressTest::SetUpComponents() {
  ChatBaseTest::SetUpComponents();

  // We want a shared_ptr to this class that was instantiated by gtest but we don't want to manage its lifecycle
  auto noopDeleter = [](ChatStressTest* /*p*/) {};
  mThis = std::shared_ptr<ChatStressTest>(this, noopDeleter);

  mTransport = std::make_shared<TestTransport>();
  mTestFactory->SetTransport(std::static_pointer_cast<IChatTransport>(mTransport));

  mCoreApi = std::make_shared<CoreAPI>();
  ttv::test::InitializeModule(mCoreApi);

  mChatApi = std::make_shared<ChatAPI>();
  mChatApi->SetCoreApi(mCoreApi);
  mChatApi->SetChatObjectFactory(std::static_pointer_cast<IChatObjectFactory>(mTestFactory));

  // Initialize chat
  mChatApi->SetTokenizationOptions(TokenizationOptions::All());
  mChatApi->SetListener(mThis);
  ttv::test::InitializeModule(mChatApi);

  AddModule(mCoreApi);
  AddModule(mChatApi);

  // Ready for stress testing
}

void ttv::chat::test::ChatStressTest::TearDownComponents() {
  std::vector<std::shared_ptr<IModule>> modules;

  if (mChatApi != nullptr) {
    mChatApi->SetListener(nullptr);
    modules.push_back(mChatApi);
  }

  ttv::test::ShutdownModules(modules);

  // NOTE: This should not cause a crash if the noop deleter is working correctly
  mThis.reset();

  ChatBaseTest::TearDownComponents();
}

void ttv::chat::test::ChatStressTest::RunStressTest(
  uint64_t durationSeconds, const std::string& prefix, const std::string& message) {
  std::string suffix = "\r\n";
  ChannelId channelId = 1001;

  // Login with the test user
  std::string authToken = "auth_token";
  UserInfo userInfo;
  userInfo.userId = 9001;
  userInfo.userName = "pikachu";
  userInfo.displayName = "Pikachu";

  ASSERT_TRUE(TTV_SUCCEEDED(LogIn(mCoreApi, authToken, userInfo)));

  // Kick off a connect
  ASSERT_EQ(mChatApi->Connect(userInfo.userId, channelId, std::static_pointer_cast<IChatChannelListener>(mThis)),
    TTV_EC_SUCCESS);

  mTransport->SetThreadSafe(true);

  std::function<void()> updateFunction = [this]() {
    ASSERT_EQ(mCoreApi->Update(), TTV_EC_SUCCESS);
    ASSERT_EQ(mChatApi->Update(), TTV_EC_SUCCESS);
  };

  // Fake the minimum number of commands needed to connect successfully
  mTransport->EnqueueIncomingData(":tmi.twitch.tv CAP * ACK :twitch.tv/tags twitch.tv/commands\r\n");
  mTransport->EnqueueIncomingData(":tmi.twitch.tv 001 pikachu :Welcome, GLHF!\r\n");

  EXPECT_TRUE(ttv::test::WaitUntilResultWithPollTask(2000,
    [this]() {
      while (mTransport->HasOutgoingData()) {
        if (mTransport->NextOutgoingData() == "JOIN #pokemon\r\n") {
          return true;
        }
      }
      return false;
    },
    updateFunction));

  mTransport->EnqueueIncomingData(":pikachu!pikachu@pikachu.tmi.twitch.tv JOIN #pokemon\r\n");

  mTransport->EnqueueIncomingData(
    "@color=#5F9EA0;display-name=pikachu;emote-sets=0,42,793,2126,10947;subscriber=0;turbo=0;user-type=mod :tmi.twitch.tv USERSTATE #pokemon\r\n");

  std::function<bool()> checkChannelConnected = [this, &channelId]() {
    auto iter = mChannelStates.find(channelId);
    if (iter != mChannelStates.end()) {
      ChatChannelState chatChannelState = iter->second;
      if (chatChannelState == ChatChannelState::Connected) {
        return true;
      }
    }
    return false;
  };

  EXPECT_TRUE(ttv::test::WaitUntilResultWithPollTask(10000, checkChannelConnected, updateFunction));

  std::cout << "Running stress test for " << durationSeconds << " seconds..." << std::endl;

  uint32_t numIncomingMessages = 0;
  uint64_t startTime = GetSystemTimeMilliseconds();
  char messageIndexBuffer[64];

  while (GetSystemTimeMilliseconds() - startTime < durationSeconds * 1000) {
    (void)snprintf(messageIndexBuffer, sizeof(messageIndexBuffer), "%u", numIncomingMessages);
    messageIndexBuffer[sizeof(messageIndexBuffer) - 1] = 0;

    std::string full = prefix + message + messageIndexBuffer + suffix;

    numIncomingMessages++;

    mTransport->EnqueueIncomingData(full);
    ASSERT_EQ(mChatApi->Update(), TTV_EC_SUCCESS);

    if (numIncomingMessages % 32 == 0) {
      ttv::Sleep(100);
    }
  }

  std::cout << "Chat incoming message throughput: " << mNumMessagesReceived << " / " << durationSeconds << " seconds"
            << std::endl;
  std::cout << "Chat incoming message throughput: "
            << (static_cast<float>(mNumMessagesReceived) / static_cast<float>(durationSeconds)) << " / second"
            << std::endl;

  mTransport->ClearIncomingData();

  // Flush out messages still being processed
  while ((GetSystemTimeMilliseconds() - mLastMessageReceivedTime) < 3000) {
    updateFunction();
    ttv::Sleep(10);
  }

  // Logout the test user
  ASSERT_TRUE(TTV_SUCCEEDED(LogOut(mCoreApi, userInfo)));
}

void ttv::chat::test::ChatStressTest::ChatChannelStateChanged(
  UserId /*userId*/, ChannelId channelId, ChatChannelState state, TTV_ErrorCode /*ec*/) {
  mChannelStates[channelId] = state;
}

void ttv::chat::test::ChatStressTest::ChatChannelMessagesReceived(
  UserId /*userId*/, ChannelId /*channelId*/, const std::vector<LiveChatMessage>& messageList) {
  if (mMessageFreeDelay > 0) {
    ttv::Sleep(mMessageFreeDelay);
  }

  mLastMessageReceivedTime = GetSystemTimeMilliseconds();
  mNumMessagesReceived += static_cast<uint32_t>(messageList.size());
  mNumMessageBatchesReceived++;
}

void ttv::chat::test::ChatStressTest::ChatChannelSubscriptionNoticeReceived(
  UserId /*userId*/, ChannelId /*channelId*/, const SubscriptionNotice& /*notice*/) {}

void ttv::chat::test::ChatStressTest::ChatChannelFirstTimeChatterNoticeReceived(
  UserId /*userId*/, ChannelId /*channelId*/, const FirstTimeChatterNotice& /*notice*/) {}

void ttv::chat::test::ChatStressTest::ChatChannelRaidNoticeReceived(
  UserId /*userId*/, ChannelId /*channelId*/, const RaidNotice& /*notice*/) {}

void ttv::chat::test::ChatStressTest::ChatChannelUnraidNoticeReceived(
  UserId /*userId*/, ChannelId /*channelId*/, const UnraidNotice& /*notice*/) {}

void ttv::chat::test::ChatStressTest::ChatChannelGenericNoticeReceived(
  UserId /*userId*/, ChannelId /*channelId*/, const GenericMessageNotice& /*notice*/) {}
