/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/chattokenizationtest.h"

#include "testutilities.h"
#include "twitchsdk/chat/internal/chatmessageparsing.h"
#include "twitchsdk/chat/internal/chatuserbadges.h"
#include "twitchsdk/chat/internal/ircstring.h"
#include "twitchsdk/core/thread.h"

using namespace ttv;
using namespace ttv::test;
using namespace ttv::chat;

ttv::chat::test::ChatTokenizationTest::ChatTokenizationTest() {}

void ttv::chat::test::ChatTokenizationTest::SetUpStubs() {
  ChatBaseTest::SetUpStubs();

  mHttpRequest->AddResponse("https://api.twitch.tv/kraken/chat/emoticon_images")
    .AddRequestParam("emotesets", "0")
    .SetResponseBodyFromFile("chat/defaultemoticonimages.json");

  mHttpRequest->AddResponse("https://tmi.twitch.tv/group/user/dragonballz/chatters")
    .SetResponseBodyFromFile("chat/dragonballzuserlist.json");

  mHttpRequest->AddResponse("https://badges.twitch.tv/v1/badges/global/display")
    .AddRequestParam("language", "EN")
    .SetResponseBodyFromFile("chat/chatemoticonstest_global_badges.json");
}

void ttv::chat::test::ChatTokenizationTest::SetUpComponents() {
  ChatBaseTest::SetUpComponents();

  mChatTestManager = std::make_shared<ChatTestManager>();

  auto taskRunner = CreateTaskRunner();

  std::shared_ptr<UserRepository> userRepository = CreateUserRepository();
  InitializeComponent(userRepository);

  std::shared_ptr<ChatTestApiState> chatApiState = mChatTestManager->GetChatApiState();

  mHttpRequest->AddResponse("https://api.twitch.tv/kraken/chat/emoticon_images")
    .AddRequestParam("emotesets", "0")
    .SetResponseBodyFromFile("chat/defaultemoticonimages.json");

  mHttpRequest->AddResponse("https://tmi.twitch.tv/group/user/dragonballz/chatters")
    .SetResponseBodyFromFile("chat/dragonballzuserlist.json");

  mHttpRequest->AddResponse("https://badges.twitch.tv/v1/badges/global/display")
    .AddRequestParam("language", "EN")
    .SetResponseBodyFromFile("chat/chatemoticonstest_global_badges.json");

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatuseremoticons.json")
    .Done();

  mTestUserId = 9001;
  userRepository->RegisterUser(mTestUserId);
  std::shared_ptr<User> user = userRepository->GetUser(mTestUserId);
  user->SetOAuthToken(std::make_shared<OAuthToken>("auth_token"));

  std::shared_ptr<PubSubClient> pubsub = user->GetComponentContainer()->GetComponent<PubSubClient>();
  if (pubsub == nullptr) {
    pubsub = std::make_shared<PubSubClient>(user, nullptr);
    pubsub->SetTaskRunner(taskRunner);
    pubsub->Initialize();
  }
  user->GetComponentContainer()->SetComponent(PubSubClient::GetComponentName(), pubsub);

  bool setsFetched = false;
  std::shared_ptr<UserEmoticonSetsListenerProxy> emoticonSetsListener =
    std::make_shared<UserEmoticonSetsListenerProxy>();
  emoticonSetsListener->mUserEmoticonSetsChangedFunc =
    [&setsFetched](UserId /*userId*/, const std::vector<EmoticonSet>& /*emoticonSets*/) { setsFetched = true; };

  std::shared_ptr<UserEmoticonSets> userEmoticonSets = user->GetComponentContainer()->GetComponent<UserEmoticonSets>();
  if (userEmoticonSets == nullptr) {
    userEmoticonSets = std::make_shared<UserEmoticonSets>(user, TokenizationOptions::All());
    userEmoticonSets->SetTaskRunner(taskRunner);
    userEmoticonSets->SetListener(emoticonSetsListener);
    userEmoticonSets->Initialize();
  }
  user->GetComponentContainer()->SetComponent(UserEmoticonSets::GetComponentName(), userEmoticonSets);

  std::shared_ptr<ChatUserBadges> userBadges = user->GetComponentContainer()->GetComponent<ChatUserBadges>();
  if (userBadges == nullptr) {
    userBadges = std::make_shared<ChatUserBadges>(user);
    userBadges->SetTaskRunner(taskRunner);
    userBadges->Initialize();
  }
  user->GetComponentContainer()->SetComponent(ChatUserBadges::GetComponentName(), userBadges);

  WaitUntilResultWithPollTask(1000, [&setsFetched] { return setsFetched; }, GetDefaultUpdateFunc());
}

void ttv::chat::test::ChatTokenizationTest::TestTokenizeLocalMessage(
  const std::string& message, const std::string& expectedEmotesTag) {
  std::string emotesMessageTag;
  std::string badgesMessageTag;
  TokenizeLocalMessage(mUserRepository->GetUser(mTestUserId), 0, message, emotesMessageTag, badgesMessageTag);
  EXPECT_EQ(std::string(expectedEmotesTag), std::string(emotesMessageTag));
}

void ttv::chat::test::ChatTokenizationTest::TestTokenizeServerMessage(const std::string& message,
  const std::string& emotesMessageTags, const std::string& sender,
  std::vector<std::unique_ptr<MessageToken>>&& expectedTokenList) {
  TestTokenizeServerMessage(message, emotesMessageTags, nullptr, 0, sender, std::move(expectedTokenList));
}

void ttv::chat::test::ChatTokenizationTest::TestTokenizeServerMessage(const std::string& message,
  const std::string& emotesMessageTags, const std::shared_ptr<BitsConfiguration>& bitsConfig, uint32_t numBits,
  const std::string& /*sender*/, std::vector<std::unique_ptr<MessageToken>>&& expectedTokenList) {
  MessageInfo tokenizedMessage;

  MessageInfo::Flags messageFlags;
  if (numBits > 0) {
    messageFlags.containsBits = true;
  }

  MessageInfo expected;
  expected.userName = "vegeta";
  expected.tokens = std::move(expectedTokenList);
  expected.flags = messageFlags;
  expected.numBitsSent = numBits;

  tokenizedMessage.flags = messageFlags;
  tokenizedMessage.userName = "vegeta";

  TokenizationOptions tokenizationOptions;
  tokenizationOptions.emoticons = true;
  tokenizationOptions.mentions = true;
  tokenizationOptions.urls = true;
  tokenizationOptions.bits = true;

  if (numBits <= 0) {
    tokenizationOptions.bits = false;
  }

  TokenizeServerMessage(message, tokenizationOptions, emotesMessageTags, bitsConfig, {}, tokenizedMessage);

  bool tokenizedMessageEqual = mChatTestManager->AssertMessage(tokenizedMessage, expected);
  EXPECT_TRUE(tokenizedMessageEqual) << message;
}
