/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/chatapitest.h"
#include "fixtures/sdkbasetest.h"
#include "testutilities.h"
#include "twitchsdk/chat/chatapi.h"
#include "twitchsdk/chat/imultiviewnotifications.h"
#include "twitchsdk/chat/imultiviewnotificationslistener.h"
#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/stringutilities.h"

using namespace ttv;
using namespace ttv::chat;
using namespace ttv::test;
using namespace ttv::chat::test;

namespace {
class TestMultiviewNotificationsListener : public ttv::chat::IMultiviewNotificationsListener {
 public:
  using ChanletUpdatedCallback = std::function<void(UserId userId, ChannelId channelId, Chanlet&& chanlet)>;

  virtual void ChanletUpdated(UserId userId, ChannelId channelId, Chanlet&& chanlet) override {
    if (chanletUpdatedCallback != nullptr) {
      chanletUpdatedCallback(userId, channelId, std::move(chanlet));
    }
  }

  ChanletUpdatedCallback chanletUpdatedCallback;
};

class MultiviewNotificationsTest : public SdkBaseTest {
 public:
  virtual void SetUpComponents() override {
    SdkBaseTest::SetUpComponents();

    mCoreApi = std::make_shared<CoreAPI>();
    mChatApi = std::make_shared<ChatAPI>();

    mChatApi->SetCoreApi(mCoreApi);
    mChatApi->SetTokenizationOptions(TokenizationOptions::All());

    InitializeModule(mCoreApi);
    InitializeModule(mChatApi);

    AddModule(mCoreApi);
    AddModule(mChatApi);

    mUserInfo.userId = 12345;
    mUserInfo.userName = "user";
    mUserInfo.displayName = "user";

    TTV_ErrorCode ec = LogIn(mCoreApi, "auth_token", mUserInfo);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));

    mMultiviewListener = std::make_shared<TestMultiviewNotificationsListener>();
  }

 protected:
  UserInfo mUserInfo;
  std::shared_ptr<TestMultiviewNotificationsListener> mMultiviewListener;
  std::shared_ptr<ChatAPI> mChatApi;

 private:
  std::shared_ptr<CoreAPI> mCoreApi;
};
}  // namespace

TEST_F(MultiviewNotificationsTest, ChanletUpdated) {
  bool receivedCallback = false;
  mMultiviewListener->chanletUpdatedCallback = [&receivedCallback](
                                                 UserId userId, ChannelId channelId, Chanlet&& chanlet) {
    ASSERT_EQ(userId, 12345);
    ASSERT_EQ(channelId, 1001);
    ASSERT_EQ(chanlet.chanletId, 167136666);
    ASSERT_EQ(chanlet.attributes.size(), 1);

    const auto& attribute = chanlet.attributes[0];
    ASSERT_EQ(attribute.ownerChannelId, 167136666);
    ASSERT_EQ(attribute.attributeId, "54940958-cc76-42bd-9584-048f776df543");
    ASSERT_EQ(attribute.key, "group");
    ASSERT_EQ(attribute.name, "Group");
    ASSERT_EQ(attribute.value, "Broadcaster");
    ASSERT_EQ(attribute.parentId, "");
    ASSERT_EQ(attribute.parentKey, "");
    ASSERT_EQ(attribute.valueShortName, "");

    receivedCallback = true;
  };

  std::shared_ptr<IMultiviewNotifications> multiviewNotifications;
  auto result = mChatApi->CreateMultiviewNotifications(12345, 1001, mMultiviewListener);
  ASSERT_TRUE(result.IsSuccess());

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [this]() { return mPubSubTestUtility.IsSubscribedToTopic("multiview-chanlet-update.1001"); },
    GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage("multiview-chanlet-update.1001",
    "{\"type\":\"chanlet\",\"chanlet\":{\"id\":\"167136666\",\"contentAttributes\":[{\"ownerChannel\":{\"id\":\"167136666\"},\"id\":\"54940958-cc76-42bd-9584-048f776df543\",\"key\":\"group\",\"name\":\"Group\",\"value\":\"Broadcaster\"}]}}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));
}
