/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/chatapitest.h"
#include "fixtures/sdkbasetest.h"
#include "testutilities.h"
#include "twitchsdk/chat/chatapi.h"
#include "twitchsdk/chat/isquadnotificationslistener.h"
#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/stringutilities.h"

using namespace ttv;
using namespace ttv::chat;
using namespace ttv::test;
using namespace ttv::chat::test;

namespace {
class TestSquadNotificationsListener : public ttv::chat::ISquadNotificationsListener {
 public:
  using SquadUpdatedCallback = std::function<void(SquadInfo&&)>;
  using SquadEndedCallback = std::function<void()>;

  virtual void SquadUpdated(SquadInfo&& squad) override {
    if (squadUpdatedCallback != nullptr) {
      squadUpdatedCallback(std::move(squad));
    }
  }

  virtual void SquadEnded() override {
    if (squadEndedCallback != nullptr) {
      squadEndedCallback();
    }
  }

  SquadUpdatedCallback squadUpdatedCallback;
  SquadEndedCallback squadEndedCallback;
};

class SquadNotificationsTest : public SdkBaseTest {
 public:
  virtual void SetUpComponents() override {
    SdkBaseTest::SetUpComponents();

    mCoreApi = std::make_shared<CoreAPI>();
    mChatApi = std::make_shared<ChatAPI>();

    mChatApi->SetCoreApi(mCoreApi);
    mChatApi->SetTokenizationOptions(TokenizationOptions::All());

    InitializeModule(mCoreApi);
    InitializeModule(mChatApi);

    AddModule(mCoreApi);
    AddModule(mChatApi);

    mUserInfo.userId = 12345;
    mUserInfo.userName = "user";
    mUserInfo.displayName = "user";

    TTV_ErrorCode ec = LogIn(mCoreApi, "auth_token", mUserInfo);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));

    mSquadListener = std::make_shared<TestSquadNotificationsListener>();
  }

 protected:
  UserInfo mUserInfo;
  std::shared_ptr<TestSquadNotificationsListener> mSquadListener;
  std::shared_ptr<ChatAPI> mChatApi;

 private:
  std::shared_ptr<CoreAPI> mCoreApi;
};
}  // namespace

TEST_F(SquadNotificationsTest, SquadUpdated) {
  bool receivedCallback = false;
  mSquadListener->squadUpdatedCallback = [&receivedCallback](SquadInfo&& squad) {
    ASSERT_EQ(squad.squadId, "6kDhYISxTcuKFAYY7w91ow");
    ASSERT_EQ(squad.ownerId, 12345);
    ASSERT_EQ(squad.status, SquadStatus::Live);
    ASSERT_EQ(squad.members.size(), 2);

    ASSERT_EQ(squad.members[0].channelId, 12826);
    ASSERT_EQ(squad.members[0].userLogin, "twitch");
    ASSERT_EQ(squad.members[0].userDisplayName, "Twitch");
    ASSERT_EQ(squad.members[0].profileImageUrl150, "imageurl.png");

    ASSERT_EQ(squad.members[1].channelId, 12827);
    ASSERT_EQ(squad.members[1].userLogin, "twitch2");
    ASSERT_EQ(squad.members[1].userDisplayName, "Twitch2");
    ASSERT_EQ(squad.members[1].profileImageUrl150, "imageurl.jpg");

    receivedCallback = true;
  };

  std::shared_ptr<ISquadNotifications> squadNotifications;
  auto result = mChatApi->CreateSquadNotifications(12345, "6kDhYISxTcuKFAYY7w91ow", mSquadListener);
  ASSERT_TRUE(result.IsSuccess());

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [this]() { return mPubSubTestUtility.IsSubscribedToTopic("squad-updates.6kDhYISxTcuKFAYY7w91ow"); },
    GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage("squad-updates.6kDhYISxTcuKFAYY7w91ow",
    "{\"squad\":{\"id\":\"6kDhYISxTcuKFAYY7w91ow\",\"members\":[{\"id\":\"12826\",\"login\":\"twitch\",\"display_name\":\"Twitch\",\"profile_image_url_150\":\"imageurl.png\"},{\"id\":\"12827\",\"login\":\"twitch2\",\"display_name\":\"Twitch2\",\"profile_image_url_150\":\"imageurl.jpg\"}],\"owner_id\":\"12345\",\"status\":\"LIVE\"},\"timestamp\":\"2018-10-03T23:58:41.599121546Z\",\"type\":\"squad\"}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));
}

TEST_F(SquadNotificationsTest, SquadEnded) {
  bool receivedCallback = false;
  mSquadListener->squadEndedCallback = [&receivedCallback]() { receivedCallback = true; };

  std::shared_ptr<ISquadNotifications> squadNotifications;
  auto result = mChatApi->CreateSquadNotifications(12345, "6kDhYISxTcuKFAYY7w91ow", mSquadListener);
  ASSERT_TRUE(result.IsSuccess());

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [this]() { return mPubSubTestUtility.IsSubscribedToTopic("squad-updates.6kDhYISxTcuKFAYY7w91ow"); },
    GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage("squad-updates.6kDhYISxTcuKFAYY7w91ow",
    "{\"squad\":{\"id\":\"6kDhYISxTcuKFAYY7w91ow\",\"members\":[],\"status\":\"ENDED\"},\"timestamp\":\"2018-10-03T23:58:41.599121546Z\",\"type\":\"squad\"}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));
}
