#include "testtransport.h"

#include "gtest/gtest.h"

using namespace ttv::chat;

ttv::chat::test::TestTransport::TestTransport() : mSocketOpen(false), mThreadSafe(true) {}

ttv::chat::test::TestTransport::~TestTransport() {}

void ttv::chat::test::TestTransport::EnqueueOutgoingData(const std::string& data) {
  if (mThreadSafe) {
    mThreadSafeOutgoingQueue.push(data);
  } else {
    mOutgoingQueue.push(data);
  }
}

void ttv::chat::test::TestTransport::EnqueueIncomingData(const std::string& data) {
  if (mThreadSafe) {
    mThreadSafeIncomingQueue.push(data);
  } else {
    mIncomingQueue.push(data);
  }
}

std::string ttv::chat::test::TestTransport::NextOutgoingData() {
  std::string data;

  if (mThreadSafe) {
    bool popped = false;
    while (!popped) {
      popped = mThreadSafeOutgoingQueue.try_pop(data);
    }
  } else {
    if (!mOutgoingQueue.empty()) {
      data = mOutgoingQueue.front();
      mOutgoingQueue.pop();
    }
  }

  return data;
}

std::string ttv::chat::test::TestTransport::NextIncomingData() {
  if (mPersistentIncomingData != "") {
    return mPersistentIncomingData;
  }

  std::string data;

  if (mThreadSafe) {
    bool popped = false;
    while (!popped) {
      popped = mThreadSafeIncomingQueue.try_pop(data);
    }
  } else {
    if (!mIncomingQueue.empty()) {
      data = mIncomingQueue.front();
      mIncomingQueue.pop();
    }
  }

  return data;
}

TTV_ErrorCode ttv::chat::test::TestTransport::Connect(const std::string& /*uri*/) {
  mSocketOpen = true;
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::test::TestTransport::Close() {
  mSocketOpen = false;
  return TTV_EC_SUCCESS;
}

bool ttv::chat::test::TestTransport::IsOpen() {
  return mSocketOpen;
}

void ttv::chat::test::TestTransport::SetReader(std::shared_ptr<IChatTransportReader> reader) {
  mReader = reader;
}

TTV_ErrorCode ttv::chat::test::TestTransport::ProcessIncomingEvent(bool& handled) {
  handled = false;

  if (!HasIncomingData()) {
    return TTV_EC_SUCCESS;
  }

  handled = true;

  std::string serverMsg = NextIncomingData();

  if (mReader) {
    mReader->OnRead(serverMsg.c_str(), serverMsg.size());
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::test::TestTransport::Write(const char* pData, size_t nSize) {
  EnqueueOutgoingData(std::string(pData, nSize));

  return TTV_EC_SUCCESS;
}

bool ttv::chat::test::TestTransport::HasIncomingData() {
  if (mPersistentIncomingData != "") {
    return true;
  } else {
    if (mThreadSafe) {
      return !mThreadSafeIncomingQueue.empty();
    } else {
      return !mIncomingQueue.empty();
    }
  }
}

void ttv::chat::test::TestTransport::ClearIncomingData() {
  mThreadSafeIncomingQueue.clear();
  mIncomingQueue = std::queue<std::string>();
}
