package tv.twitch.chat;

import tv.twitch.*;

public class ChatAPI extends NativeProxy implements IModule {
    public interface BlockChangeCallback { void invoke(ErrorCode ec); }
    public interface FetchEmoticonSetsCallback { void invoke(ErrorCode ec, ChatEmoticonSet[] emoticonSets); }
    public interface FetchBadgesCallback { void invoke(ErrorCode ec, ChatBadgeSet badgeSets); }
    public interface SendMessageCallback { void invoke(ErrorCode ec, int messageId); }
    public interface FetchBlockedUsersCallback { void invoke(ErrorCode ec, UserInfo[] userIds); }
    public interface BanUserCallback { void invoke(ErrorCode ec, BanUserError error); }
    public interface UnbanUserCallback { void invoke(ErrorCode ec, UnbanUserError error); }
    public interface ModUserCallback { void invoke(ErrorCode ec, ModUserError error); }
    public interface GrantVIPCallback { void invoke(ErrorCode ec, GrantVIPErrorCode error); }
    public interface RevokeVIPCallback { void invoke(ErrorCode ec, RevokeVIPErrorCode error); }
    public interface UnmodUserCallback { void invoke(ErrorCode ec, UnmodUserError error); }
    public interface UpdateUserColorCallback { void invoke(ErrorCode ec); }
    public interface FetchChannelModeratorsCallback { void invoke(ErrorCode ec, String[] modNames, String nextCursor); }
    public interface FetchChannelVIPsCallback { void invoke(ErrorCode ec, String[] vipNames); }
    public interface FetchChannelVodCommentSettingsCallback {
        void invoke(ErrorCode ec, ChannelVodCommentSettings settings);
    }
    public interface SetChannelVodCommentSettingsCallback { void invoke(ErrorCode ec); }

    static { ChatErrorCode.forceClassInit(); }

    /**
     * Legacy Constructor
     * @Deprecated use {@link ChatAPI ( IJniThreadChecker )}
     */
    @Deprecated
    public ChatAPI() {
        super(PassThroughJniThreadValidator.INSTANCE);
    }

    public ChatAPI(IJniThreadChecker jniThreadChecker) { super(new JniThreadValidator(jniThreadChecker)); }

    private native long CreateNativeInstance();
    // AND-9927 Disposing a ChatAPI is safe from any thread
    private native void DisposeNativeInstance(long nativeObjectPointer);

    private native ErrorCode SetCoreApi(long nativeObjectPointer, CoreAPI coreApi);
    private native ErrorCode SetTokenizationOptions(
        long nativeObjectPointer, ChatTokenizationOptions tokenizationOptions);
    private native ErrorCode SetListener(long nativeObjectPointer, IChatAPIListener listener);
    private native ErrorCode SetEnabledFeatures(long nativeObjectPointer, ChatFeatureFlags features);

    private native ModuleState GetState(long nativeObjectPointer);
    private native String GetModuleName(long nativeObjectPointer);
    private native ErrorCode Initialize(long nativeObjectPointer, InitializeCallback callback);
    private native ErrorCode Shutdown(long nativeObjectPointer, ShutdownCallback callback);
    private native ErrorCode Update(long nativeObjectPointer);

    private native ErrorCode Connect(
        long nativeObjectPointer, int userId, int channelId, IChatChannelListener channelListener);
    private native ErrorCode Disconnect(long nativeObjectPointer, int userId, int channelId);
    private native ErrorCode SendMessage(long nativeObjectPointer, int userId, int channelId, String message);

    private native ErrorCode BlockUser(long nativeObjectPointer, int userId, int bloblockUserId, String reason,
        boolean whisper, BlockChangeCallback callback);
    private native ErrorCode UnblockUser(
        long nativeObjectPointer, int userId, int blockUserId, BlockChangeCallback callback);
    private native ErrorCode GetUserBlocked(
        long nativeObjectPointer, int userId, int blockUserId, ResultContainer<Boolean> result);
    private native ErrorCode FetchBlockedUsers(
        long nativeObjectPointer, int userId, FetchBlockedUsersCallback callback);

    private native ErrorCode FetchUserEmoticonSets(
        long nativeObjectPointer, int userId, boolean forceFetch, FetchEmoticonSetsCallback callback);

    private native ErrorCode FetchGlobalBadges(long nativeObjectPointer, FetchBadgesCallback callback);
    private native ErrorCode FetchChannelBadges(long nativeObjectPointer, int channelId, FetchBadgesCallback callback);

    private native ErrorCode SetUserThreadsListener(
        long nativeObjectPointer, int userId, IChatUserThreadsListener listener);

    private native long GetMessageFlushInterval(long nativeObjectPointer);
    private native ErrorCode SetMessageFlushInterval(long nativeObjectPointer, long milliseconds);

    private native ErrorCode GetEmoticonUrl(
        long nativeObjectPointer, String emoticonId, float scale, ResultContainer<String> result);

    private native ErrorCode CreateChannelChatRoomManager(long nativeObjectPointer, int userId, int channelId,
        IChannelChatRoomManagerListener listener, ResultContainer<IChannelChatRoomManager> result);

    private native ErrorCode CreateChatRoom(long nativeObjectPointer, int userId, String roomId, int channelId,
        IChatRoomListener listener, ResultContainer<IChatRoom> result);

    private native ErrorCode CreateChatRoomNotifications(long nativeObjectPointer, int userId,
        IChatRoomNotificationsListener listener, ResultContainer<IChatRoomNotifications> result);

    private native ErrorCode CreateChatChannel(long nativeObjectPointer, IJniThreadValidator jniThreadValidator,
        int userId, int channelId, IChatChannelListener listener, ResultContainer<IChatChannel> channel);
    private native ErrorCode CreateChatChannelProperties(long nativeObjectPointer, int userId, int channelId,
        IChatChannelPropertyListener listener, ResultContainer<IChatChannelProperties> result);

    private native Result<IChatCommentManager> CreateChatCommentManager(long nativeObjectPointer,
        IJniThreadValidator jniThreadValidator, int userId, String vodId, IChatCommentListener listener);
    private native ErrorCode FetchChannelVodCommentSettings(
        long nativeObjectPointer, int userId, int channelId, FetchChannelVodCommentSettingsCallback callback);
    private native ErrorCode SetChannelVodFollowersOnlyDuration(long nativeObjectPointer, int userId, int channelId,
        int followersOnlyDuration, SetChannelVodCommentSettingsCallback callback);
    private native ErrorCode SetChannelVodPublishingMode(long nativeObjectPointer, int userId, int channelId,
        CommentPublishingMode mode, SetChannelVodCommentSettingsCallback callback);

    private native ErrorCode CreateChatRaid(long nativeObjectPointer, IJniThreadValidator jniThreadValidator,
        int userId, int sourceUserId, IChatRaidListener listener, ResultContainer<IChatRaid> result);
    // AND-9927 Disposing an IChatRaid is safe from any thread
    private native ErrorCode DisposeChatRaid(long nativeObjectPointer, IChatRaid chatRaid);

    private native ErrorCode CreateBitsStatus(
        long nativeObjectPointer, int userId, IBitsListener listener, ResultContainer<IBitsStatus> result);
    // AND-9927 Disposing an IBitsStatus is safe from any thread
    private native ErrorCode DisposeBitsStatus(long nativeObjectPointer, IBitsStatus bitsStatus);

    private native ErrorCode CreateFollowersStatus(long nativeObjectPointer, int userId, int channelId,
        IFollowersListener listener, ResultContainer<IFollowersStatus> result);
    // AND-9927 Disposing an IFollowersStatus is safe from any thread
    private native ErrorCode DisposeFollowersStatus(long nativeObjectPointer, IFollowersStatus followersStatus);

    private native ErrorCode CreateFollowingStatus(
        long nativeObjectPointer, int userId, IFollowingListener listener, ResultContainer<IFollowingStatus> result);

    private native ErrorCode CreateSubscribersStatus(long nativeObjectPointer, int userId,
        ISubscribersListener listener, ResultContainer<ISubscribersStatus> result);
    // AND-9927 Disposing an ISubscribersStatus is safe from any thread
    private native ErrorCode DisposeSubscribersStatus(long nativeObjectPointer, ISubscribersStatus subscribersStatus);

    private native Result<ISubscriptionsNotifications> CreateSubscriptionsNotifications(
        long nativeObjectPointer, int userId, ISubscriptionsNotificationsListener listener);
    private native Result<ISquadNotifications> CreateSquadNotifications(
        long nativeObjectPointer, int userId, String squadId, ISquadNotificationsListener listener);
    private native Result<IMultiviewNotifications> CreateMultiviewNotifications(
        long nativeObjectPointer, int userId, int channelId, IMultiviewNotificationsListener listener);

    private native ErrorCode BanUser(long nativeObjectPointer, int userId, int channelId, String bannedUserName,
        int duration, BanUserCallback callback);
    private native ErrorCode UnbanUser(
        long nativeObjectPointer, int userId, int channelId, String bannedUserName, UnbanUserCallback callback);
    private native ErrorCode ModUser(
        long nativeObjectPointer, int userId, int channelId, String modUserName, ModUserCallback callback);
    private native ErrorCode GrantVIP(
        long nativeObjectPointer, int userId, int channelId, String vipUserName, GrantVIPCallback callback);
    private native ErrorCode RevokeVIP(
        long nativeObjectPointer, int userId, int channelId, String unvipUserName, RevokeVIPCallback callback);
    private native ErrorCode UnmodUser(
        long nativeObjectPointer, int userId, int channelId, String unmodUserName, UnmodUserCallback callback);
    private native ErrorCode UpdateUserColor(
        long nativeObjectPointer, int userId, String color, UpdateUserColorCallback callback);
    private native ErrorCode FetchChannelModerators(
        long nativeObjectPointer, int channelId, String cursor, FetchChannelModeratorsCallback callback);
    private native ErrorCode FetchChannelVIPs(
        long nativeObjectPointer, int channelId, FetchChannelVIPsCallback callback);

    // AND-9927 TokenizeServerMessage is safe to call from any thread
    private static native boolean TokenizeServerMessage(String message, ChatTokenizationOptions tokenizationOptions,
        String emoticonRanges, String[] localUserNames, ResultContainer<ChatMessageInfo> result);

    @Override
    protected long createNativeInstance() {
        return mJniThreadValidator.callJniCallable(new IJniCallable<Long>() {
            @Override
            public Long call() {
                return CreateNativeInstance();
            }
        });
    }

    @Override
    protected void disposeNativeInstance(long nativeObjectPointer) {
        // AND-9927 Disposing a ChatAPI is safe from any thread
        DisposeNativeInstance(nativeObjectPointer);
    }

    public ErrorCode setCoreApi(final CoreAPI coreApi) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetCoreApi(getNativeObjectPointer(), coreApi);
            }
        });
    }

    public ErrorCode setTokenizationOptions(final ChatTokenizationOptions tokenizationOptions) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetTokenizationOptions(getNativeObjectPointer(), tokenizationOptions);
            }
        });
    }

    public ErrorCode setListener(final IChatAPIListener listener) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetListener(getNativeObjectPointer(), listener);
            }
        });
    }

    public ErrorCode setEnabledFeatures(final ChatFeatureFlags flags) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetEnabledFeatures(getNativeObjectPointer(), flags);
            }
        });
    }

    @Override
    public ModuleState getState() {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ModuleState>() {
            @Override
            public ModuleState call() {
                return GetState(getNativeObjectPointer());
            }
        });
    }

    @Override
    public String getModuleName() {
        return mJniThreadValidator.callJniCallable(new IJniCallable<String>() {
            @Override
            public String call() {
                return GetModuleName(getNativeObjectPointer());
            }
        });
    }

    @Override
    public ErrorCode initialize(final InitializeCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return Initialize(getNativeObjectPointer(), callback);
            }
        });
    }

    @Override
    public ErrorCode shutdown(final ShutdownCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return Shutdown(getNativeObjectPointer(), callback);
            }
        });
    }

    @Override
    public ErrorCode update() {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return Update(getNativeObjectPointer());
            }
        });
    }

    public ErrorCode connect(final int userId, final int channelId, final IChatChannelListener channelListener) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return Connect(getNativeObjectPointer(), userId, channelId, channelListener);
            }
        });
    }

    public ErrorCode disconnect(final int userId, final int channelId) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return Disconnect(getNativeObjectPointer(), userId, channelId);
            }
        });
    }

    public ErrorCode sendMessage(final int userId, final int channelId, final String message) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SendMessage(getNativeObjectPointer(), userId, channelId, message);
            }
        });
    }

    public ErrorCode blockUser(final int userId, final int blockUserId, final String reason, final boolean whisper,
        final BlockChangeCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return BlockUser(getNativeObjectPointer(), userId, blockUserId, reason, whisper, callback);
            }
        });
    }

    public ErrorCode unblockUser(final int userId, final int blockUserId, final BlockChangeCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return UnblockUser(getNativeObjectPointer(), userId, blockUserId, callback);
            }
        });
    }

    public ErrorCode getUserBlocked(final int userId, final int blockUserId, final ResultContainer<Boolean> result) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return GetUserBlocked(getNativeObjectPointer(), userId, blockUserId, result);
            }
        });
    }

    public ErrorCode fetchBlockedUsers(final int userId, final FetchBlockedUsersCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return FetchBlockedUsers(getNativeObjectPointer(), userId, callback);
            }
        });
    }

    public ErrorCode fetchUserEmoticonSets(
        final int userId, final boolean forceFetch, final FetchEmoticonSetsCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return FetchUserEmoticonSets(getNativeObjectPointer(), userId, forceFetch, callback);
            }
        });
    }

    public ErrorCode fetchGlobalBadges(final FetchBadgesCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return FetchGlobalBadges(getNativeObjectPointer(), callback);
            }
        });
    }

    public ErrorCode fetchChannelBadges(final int channelId, final FetchBadgesCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return FetchChannelBadges(getNativeObjectPointer(), channelId, callback);
            }
        });
    }

    public ErrorCode setUserThreadsListener(final int userId, final IChatUserThreadsListener listener) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetUserThreadsListener(getNativeObjectPointer(), userId, listener);
            }
        });
    }

    public long getMessageFlushInterval() {
        return mJniThreadValidator.callJniCallable(new IJniCallable<Long>() {
            @Override
            public Long call() {
                return GetMessageFlushInterval(getNativeObjectPointer());
            }
        });
    }

    public ErrorCode setMessageFlushInterval(final long milliseconds) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetMessageFlushInterval(getNativeObjectPointer(), milliseconds);
            }
        });
    }

    public ErrorCode getEmoticonUrl(final String emoticonId, final float scale, final ResultContainer<String> result) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return GetEmoticonUrl(getNativeObjectPointer(), emoticonId, scale, result);
            }
        });
    }

    public ErrorCode createChannelChatRoomManager(final int userId, final int channelId,
        final IChannelChatRoomManagerListener listener, final ResultContainer<IChannelChatRoomManager> result) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return CreateChannelChatRoomManager(getNativeObjectPointer(), userId, channelId, listener, result);
            }
        });
    }

    public ErrorCode createChatRoom(final int userId, final String roomId, final int channelId,
        final IChatRoomListener listener, final ResultContainer<IChatRoom> result) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return CreateChatRoom(getNativeObjectPointer(), userId, roomId, channelId, listener, result);
            }
        });
    }

    public ErrorCode createChatRoomNotifications(final int userId, final IChatRoomNotificationsListener listener,
        final ResultContainer<IChatRoomNotifications> result) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return CreateChatRoomNotifications(getNativeObjectPointer(), userId, listener, result);
            }
        });
    }

    public ErrorCode createChatChannel(final int userId, final int channelId, final IChatChannelListener listener,
        final ResultContainer<IChatChannel> channel) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return CreateChatChannel(
                    getNativeObjectPointer(), mJniThreadValidator, userId, channelId, listener, channel);
            }
        });
    }

    public ErrorCode createChatChannelProperties(final int userId, final int channelId,
        final IChatChannelPropertyListener listener, final ResultContainer<IChatChannelProperties> result) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return CreateChatChannelProperties(getNativeObjectPointer(), userId, channelId, listener, result);
            }
        });
    }

    public Result<IChatCommentManager> createChatCommentManager(
        final int userId, final String vodId, final IChatCommentListener listener) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<Result<IChatCommentManager>>() {
            @Override
            public Result<IChatCommentManager> call() {
                return CreateChatCommentManager(getNativeObjectPointer(), mJniThreadValidator, userId, vodId, listener);
            }
        });
    }

    public ErrorCode fetchChannelVodCommentSettings(
        final int userId, final int channelId, final FetchChannelVodCommentSettingsCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return FetchChannelVodCommentSettings(getNativeObjectPointer(), userId, channelId, callback);
            }
        });
    }

    public ErrorCode setChannelVodFollowersOnlyDuration(final int userId, final int channelId,
        final int followersOnlyDuration, final SetChannelVodCommentSettingsCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetChannelVodFollowersOnlyDuration(
                    getNativeObjectPointer(), userId, channelId, followersOnlyDuration, callback);
            }
        });
    }

    public ErrorCode setChannelVodPublishingMode(final int userId, final int channelId,
        final CommentPublishingMode mode, final SetChannelVodCommentSettingsCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return SetChannelVodPublishingMode(getNativeObjectPointer(), userId, channelId, mode, callback);
            }
        });
    }

    public ErrorCode createChatRaid(final int userId, final int sourceUserId, final IChatRaidListener listener,
        final ResultContainer<IChatRaid> result) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return CreateChatRaid(
                    getNativeObjectPointer(), mJniThreadValidator, userId, sourceUserId, listener, result);
            }
        });
    }

    public ErrorCode disposeChatRaid(IChatRaid chatRaid) {
        // AND-9927 Disposing an IChatRaid is safe from any thread
        return DisposeChatRaid(getNativeObjectPointer(), chatRaid);
    }

    public ErrorCode createBitsStatus(
        final int userId, final IBitsListener listener, final ResultContainer<IBitsStatus> result) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return CreateBitsStatus(getNativeObjectPointer(), userId, listener, result);
            }
        });
    }

    public ErrorCode disposeBitsStatus(IBitsStatus bitsStatus) {
        // AND-9927 Disposing an IBitsStatus is safe from any thread
        return DisposeBitsStatus(getNativeObjectPointer(), bitsStatus);
    }

    public ErrorCode createFollowersStatus(final int userId, final int channelId, final IFollowersListener listener,
        final ResultContainer<IFollowersStatus> result) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return CreateFollowersStatus(getNativeObjectPointer(), userId, channelId, listener, result);
            }
        });
    }

    public ErrorCode disposeFollowersStatus(IFollowersStatus followersStatus) {
        // AND-9927 Disposing an IFollowersStatus is safe from any thread
        return DisposeFollowersStatus(getNativeObjectPointer(), followersStatus);
    }

    public ErrorCode createFollowingStatus(
        final int userId, final IFollowingListener listener, final ResultContainer<IFollowingStatus> result) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return CreateFollowingStatus(getNativeObjectPointer(), userId, listener, result);
            }
        });
    }

    public ErrorCode createSubscribersStatus(
        final int userId, final ISubscribersListener listener, final ResultContainer<ISubscribersStatus> result) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return CreateSubscribersStatus(getNativeObjectPointer(), userId, listener, result);
            }
        });
    }

    public ErrorCode disposeSubscribersStatus(ISubscribersStatus subscribersStatus) {
        // AND-9927 Disposing an ISubscribersStatus is safe from any thread
        return DisposeSubscribersStatus(getNativeObjectPointer(), subscribersStatus);
    }

    public Result<ISubscriptionsNotifications> createSubscriptionsNotifications(
        final int userId, final ISubscriptionsNotificationsListener listener) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<Result<ISubscriptionsNotifications>>() {
            @Override
            public Result<ISubscriptionsNotifications> call() {
                return CreateSubscriptionsNotifications(getNativeObjectPointer(), userId, listener);
            }
        });
    }

    public Result<ISquadNotifications> createSquadNotifications(
        final int userId, final String squadId, final ISquadNotificationsListener listener) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<Result<ISquadNotifications>>() {
            @Override
            public Result<ISquadNotifications> call() {
                return CreateSquadNotifications(getNativeObjectPointer(), userId, squadId, listener);
            }
        });
    }

    public Result<IMultiviewNotifications> createMultiviewNotifications(
        final int userId, final int channelId, final IMultiviewNotificationsListener listener) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<Result<IMultiviewNotifications>>() {
            @Override
            public Result<IMultiviewNotifications> call() {
                return CreateMultiviewNotifications(getNativeObjectPointer(), userId, channelId, listener);
            }
        });
    }

    public ErrorCode banUser(final int userId, final int channelId, final String bannedUserName, final int duration,
        final BanUserCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return BanUser(getNativeObjectPointer(), userId, channelId, bannedUserName, duration, callback);
            }
        });
    }

    public ErrorCode unbanUser(
        final int userId, final int channelId, final String bannedUserName, final UnbanUserCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return UnbanUser(getNativeObjectPointer(), userId, channelId, bannedUserName, callback);
            }
        });
    }

    public ErrorCode modUser(
        final int userId, final int channelId, final String modUserName, final ModUserCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return ModUser(getNativeObjectPointer(), userId, channelId, modUserName, callback);
            }
        });
    }

    public ErrorCode unmodUser(
        final int userId, final int channelId, final String unmodUserName, final UnmodUserCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return UnmodUser(getNativeObjectPointer(), userId, channelId, unmodUserName, callback);
            }
        });
    }

    public ErrorCode grantVIP(
        final int userId, final int channelId, final String vipUserName, final GrantVIPCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return GrantVIP(getNativeObjectPointer(), userId, channelId, vipUserName, callback);
            }
        });
    }

    public ErrorCode revokeVIP(
        final int userId, final int channelId, final String unvipUserName, final RevokeVIPCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return RevokeVIP(getNativeObjectPointer(), userId, channelId, unvipUserName, callback);
            }
        });
    }

    public ErrorCode updateUserColor(final int userId, final String color, final UpdateUserColorCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return UpdateUserColor(getNativeObjectPointer(), userId, color, callback);
            }
        });
    }

    public ErrorCode fetchChannelModerators(
        final int channelId, final String cursor, final FetchChannelModeratorsCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return FetchChannelModerators(getNativeObjectPointer(), channelId, cursor, callback);
            }
        });
    }

    public ErrorCode fetchChannelVIPs(final int channelId, final FetchChannelVIPsCallback callback) {
        return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
            @Override
            public ErrorCode call() {
                return FetchChannelVIPs(getNativeObjectPointer(), channelId, callback);
            }
        });
    }

    public static boolean tokenizeServerMessage(String message, ChatTokenizationOptions tokenizationOptions,
        String emoticonRanges, String[] localUserNames, ResultContainer<ChatMessageInfo> result) {
        // AND-9927 TokenizeServerMessage is safe to call from any thread
        return TokenizeServerMessage(message, tokenizationOptions, emoticonRanges, localUserNames, result);
    }
}
