package tv.twitch.chat;

import tv.twitch.CoreErrorCode;
import tv.twitch.ErrorCode;
import tv.twitch.IJniCallable;
import tv.twitch.IJniThreadValidator;
import tv.twitch.NativeProxy;
import tv.twitch.PassThroughJniThreadValidator;
import tv.twitch.Result;
import tv.twitch.ErrorResult;

public class ChatCommentManagerProxy extends NativeProxy implements IChatCommentManager {
    private native ErrorCode Play(long nativeObjectPointer);
    private native ErrorCode Pause(long nativeObjectPointer);
    private native ErrorCode UpdatePlayhead(long nativeObjectPointer, int timestampMilliseconds);
    private native Result<Integer> GetPlayheadTime(long nativeObjectPointer);
    private native Result<Integer> GetChannelId(long nativeObjectPointer);
    private native Result<PlayingState> GetPlayingState(long nativeObjectPointer);
    private native ErrorCode FetchCommentsByTimestamp(
        long nativeObjectPointer, int timestampMilliseconds, int limit, FetchCommentsCallback callback);
    private native ErrorCode FetchCommentsByCursor(
        long nativeObjectPointer, String cursor, int limit, FetchCommentsCallback callback);
    private native ErrorCode FetchComment(long nativeObjectPointer, String commentId, FetchCommentCallback callback);
    private native ErrorCode DeleteComment(long nativeObjectPointer, String commentId, DeleteCommentCallback callback);
    private native ErrorCode PostComment(
        long nativeObjectPointer, String message, int timestampMilliseconds, PostCommentCallback callback);
    private native ErrorCode ReportComment(
        long nativeObjectPointer, String commentId, String reason, String description, ReportCommentCallback callback);
    private native ErrorCode FetchCommentReplies(
        long nativeObjectPointer, String commentId, FetchCommentRepliesCallback callback);
    private native ErrorCode PostCommentReply(
        long nativeObjectPointer, String parentCommentId, String message, PostCommentReplyCallback callback);
    // AND-9927 Disposing a ChatCommentManagerProxy is safe from any thread
    private native void DisposeNativeInstance(long nativeObjectPointer);

    public ChatCommentManagerProxy(long nativeObjectPointer, IJniThreadValidator jniThreadValidator) {
        super(nativeObjectPointer, jniThreadValidator);
    }

    @Override
    protected void disposeNativeInstance(long nativeObjectPointer) {
        // AND-9927 Disposing a ChatCommentManagerProxy is safe from any thread
        DisposeNativeInstance(nativeObjectPointer);
    }

    @Override
    public void dispose() {
        super.dispose();
    }

    @Override
    public ErrorCode play() {
        final long ptr = getNativeObjectPointer();

        if (ptr != 0) {
            return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
                @Override
                public ErrorCode call() {
                    return Play(ptr);
                }
            });
        } else {
            return CoreErrorCode.TTV_EC_INVALID_INSTANCE;
        }
    }

    @Override
    public ErrorCode pause() {
        final long ptr = getNativeObjectPointer();

        if (ptr != 0) {
            return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
                @Override
                public ErrorCode call() {
                    return Pause(ptr);
                }
            });
        } else {
            return CoreErrorCode.TTV_EC_INVALID_INSTANCE;
        }
    }

    @Override
    public ErrorCode updatePlayhead(final int timestampMilliseconds) {
        final long ptr = getNativeObjectPointer();

        if (ptr != 0) {
            return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
                @Override
                public ErrorCode call() {
                    return UpdatePlayhead(ptr, timestampMilliseconds);
                }
            });
        } else {
            return CoreErrorCode.TTV_EC_INVALID_INSTANCE;
        }
    }

    @Override
    public Result<Integer> getPlayheadTime() {
        final long ptr = getNativeObjectPointer();

        if (ptr != 0) {
            return mJniThreadValidator.callJniCallable(new IJniCallable<Result<Integer>>() {
                @Override
                public Result<Integer> call() {
                    return GetPlayheadTime(ptr);
                }
            });
        } else {
            return new ErrorResult<Integer>(CoreErrorCode.TTV_EC_INVALID_INSTANCE);
        }
    }

    @Override
    public Result<Integer> getChannelId() {
        final long ptr = getNativeObjectPointer();

        if (ptr != 0) {
            return mJniThreadValidator.callJniCallable(new IJniCallable<Result<Integer>>() {
                @Override
                public Result<Integer> call() {
                    return GetChannelId(ptr);
                }
            });
        } else {
            return new ErrorResult<Integer>(CoreErrorCode.TTV_EC_INVALID_INSTANCE);
        }
    }

    @Override
    public Result<PlayingState> getPlayingState() {
        final long ptr = getNativeObjectPointer();

        if (ptr != 0) {
            return mJniThreadValidator.callJniCallable(new IJniCallable<Result<PlayingState>>() {
                @Override
                public Result<PlayingState> call() {
                    return GetPlayingState(ptr);
                }
            });
        } else {
            return new ErrorResult<PlayingState>(CoreErrorCode.TTV_EC_INVALID_INSTANCE);
        }
    }

    @Override
    public ErrorCode fetchCommentsByTimestamp(
        final int timestampMilliseconds, final int limit, final FetchCommentsCallback callback) {
        final long ptr = getNativeObjectPointer();

        if (ptr != 0) {
            return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
                @Override
                public ErrorCode call() {
                    return FetchCommentsByTimestamp(ptr, timestampMilliseconds, limit, callback);
                }
            });
        } else {
            return CoreErrorCode.TTV_EC_INVALID_INSTANCE;
        }
    }

    @Override
    public ErrorCode fetchCommentsByCursor(final String cursor, final int limit, final FetchCommentsCallback callback) {
        final long ptr = getNativeObjectPointer();

        if (ptr != 0) {
            return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
                @Override
                public ErrorCode call() {
                    return FetchCommentsByCursor(ptr, cursor, limit, callback);
                }
            });
        } else {
            return CoreErrorCode.TTV_EC_INVALID_INSTANCE;
        }
    }

    @Override
    public ErrorCode fetchComment(final String commentId, final FetchCommentCallback callback) {
        final long ptr = getNativeObjectPointer();

        if (ptr != 0) {
            return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
                @Override
                public ErrorCode call() {
                    return FetchComment(ptr, commentId, callback);
                }
            });
        } else {
            return CoreErrorCode.TTV_EC_INVALID_INSTANCE;
        }
    }

    @Override
    public ErrorCode deleteComment(final String commentId, final DeleteCommentCallback callback) {
        final long ptr = getNativeObjectPointer();

        if (ptr != 0) {
            return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
                @Override
                public ErrorCode call() {
                    return DeleteComment(ptr, commentId, callback);
                }
            });
        } else {
            return CoreErrorCode.TTV_EC_INVALID_INSTANCE;
        }
    }

    @Override
    public ErrorCode postComment(
        final String message, final int timestampMilliseconds, final PostCommentCallback callback) {
        final long ptr = getNativeObjectPointer();

        if (ptr != 0) {
            return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
                @Override
                public ErrorCode call() {
                    return PostComment(ptr, message, timestampMilliseconds, callback);
                }
            });
        } else {
            return CoreErrorCode.TTV_EC_INVALID_INSTANCE;
        }
    }

    @Override
    public ErrorCode reportComment(
        final String commentId, final String reason, final String description, final ReportCommentCallback callback) {
        final long ptr = getNativeObjectPointer();

        if (ptr != 0) {
            return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
                @Override
                public ErrorCode call() {
                    return ReportComment(ptr, commentId, reason, description, callback);
                }
            });
        } else {
            return CoreErrorCode.TTV_EC_INVALID_INSTANCE;
        }
    }

    @Override
    public ErrorCode fetchCommentReplies(final String commentId, final FetchCommentRepliesCallback callback) {
        final long ptr = getNativeObjectPointer();

        if (ptr != 0) {
            return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
                @Override
                public ErrorCode call() {
                    return FetchCommentReplies(ptr, commentId, callback);
                }
            });
        } else {
            return CoreErrorCode.TTV_EC_INVALID_INSTANCE;
        }
    }

    @Override
    public ErrorCode postCommentReply(
        final String parentCommentId, final String message, final PostCommentReplyCallback callback) {
        final long ptr = getNativeObjectPointer();

        if (ptr != 0) {
            return mJniThreadValidator.callJniCallable(new IJniCallable<ErrorCode>() {
                @Override
                public ErrorCode call() {
                    return PostCommentReply(ptr, parentCommentId, message, callback);
                }
            });
        } else {
            return CoreErrorCode.TTV_EC_INVALID_INSTANCE;
        }
    }
}
