package tv.twitch.chat;

import java.util.HashMap;

import tv.twitch.ErrorCode;

public interface IChatChannelListener {
    /**
     * Callback signature for notifications of state changes in the channel.
     */
    void chatChannelStateChanged(int userId, int channelId, ChatChannelState state, ErrorCode ec);

    /**
     * Callback signature for notifications of info changes in the channel.
     */
    void chatChannelInfoChanged(int userId, int channelId, ChatChannelInfo channelInfo);

    /**
     * Callback signature for notifications of chat restriction changes in the channel.
     */
    void chatChannelRestrictionsChanged(int userId, int channelId, ChatChannelRestrictions restrictions);

    /**
     * Callback signature for notifications of the local user properties changing.
     */
    void chatChannelLocalUserChanged(int userId, int channelId, ChatUserInfo userInfo);

    /**
     * Callback signature for notifications of messages event occurring in chat.
     */
    void chatChannelMessagesReceived(int userId, int channelId, ChatLiveMessage[] messageList);

    /**
     * Callback for when a user shares their subscription.
     */
    void chatChannelSubscriptionNoticeReceived(int userId, int channelId, ChatSubscriptionNotice notice);

    /**
     * Callback for when a user performs a first chatter in a channel.
     */
    void chatChannelFirstTimeChatterNoticeReceived(int userId, int channelId, ChatFirstTimeChatterNotice notice);

    /**
     * Callback for when the joined channel is being raided by another channel.
     */
    void chatChannelRaidNoticeReceived(int userId, int channelId, ChatRaidNotice notice);

    /**
     * Callback for when the currently joined channel has finished raiding another channel.
     */
    void chatChannelUnraidNoticeReceived(int userId, int channelId, ChatUnraidNotice notice);

    /**
     * Callback for when the a unhandled message is received.
     */
    void chatChannelGenericNoticeReceived(int userId, int channelId, ChatGenericMessageNotice notice);

    /**
     * Callback signature for notifications that the all messages in the channel should be cleared.
     */
    void chatChannelMessagesCleared(int userId, int channelId);

    /**
     * Callback signature for notifications that the messages from the given user should be cleared. Not received by
     * moderators.
     */
    void chatChannelUserMessagesCleared(int userId, int channelId, int clearUserId);

    /**
     * Callback signature for notifications when a single message is deleted.
     */
    void chatChannelMessageDeleted(
        int userId, int channelId, String messageId, String senderLoginName, String deletedMessageContent);

    /**
     * Callback signature for a moderator when a user is timed out.
     */
    void chatChannelModNoticeUserTimedOut(
        int userId, int channelId, ModerationActionInfo modActionInfo, int timeoutDurationSeconds, String reason);

    /**
     * Callback signature for a moderator when a user is banned.
     */
    void chatChannelModNoticeUserBanned(int userId, int channelId, ModerationActionInfo modActionInfo, String reason);

    /**
     * Callback signature for a moderator when a user is no longer timed out.
     */
    void chatChannelModNoticeUserUntimedOut(int userId, int channelId, ModerationActionInfo modActionInfo);

    /**
     * Callback signature for a moderator when a user is unbanned.
     */
    void chatChannelModNoticeUserUnbanned(int userId, int channelId, ModerationActionInfo modActionInfo);

    /**
     * Callback signature for a moderator when a message is deleted.
     */
    void chatChannelModNoticeMessageDeleted(
        int userId, int channelId, ModerationActionInfo modActionInfo, String messageId, String message);

    /**
     * Callback signature for a moderator when chat is cleared.
     */
    void chatChannelModNoticeClearChat(int userId, int channelId, int modId, String modName);

    /**
     * Callback signature for a moderator when emote-only mode is turned on.
     */
    void chatChannelModNoticeEmoteOnly(int userId, int channelId, int modId, String modName);

    /**
     * Callback signature for a moderator when emote-only mode is turned off.
     */
    void chatChannelModNoticeEmoteOnlyOff(int userId, int channelId, int modId, String modName);

    /**
     * Callback signature for a moderator when followers-only mode is turned on.
     */
    void chatChannelModNoticeFollowersOnly(
        int userId, int channelId, int modId, String modName, int minimumFollowingDuration);

    /**
     * Callback signature for a moderator when followers-only mode is turned off.
     */
    void chatChannelModNoticeFollowersOnlyOff(int userId, int channelId, int modId, String modName);

    /**
     * Callback signature for a moderator when R9K mode is turned on.
     */
    void chatChannelModNoticeR9K(int userId, int channelId, int modId, String modName);

    /**
     * Callback signature for a moderator when R9K mode is turned off.
     */
    void chatChannelModNoticeR9KOff(int userId, int channelId, int modId, String modName);

    /**
     * Callback signature for a moderator when slow mode is turned on.
     */
    void chatChannelModNoticeSlow(int userId, int channelId, int modId, String modName, int slowModeDurationSeconds);

    /**
     * Callback signature for a moderator when slow mode is turned off.
     */
    void chatChannelModNoticeSlowOff(int userId, int channelId, int modId, String modName);

    /**
     * Callback signature for a moderator when sub-only mode is turned on.
     */
    void chatChannelModNoticeSubsOnly(int userId, int channelId, int modId, String modName);

    /**
     * Callback signature for a moderator when sub-only mode is turned off.
     */
    void chatChannelModNoticeSubsOnlyOff(int userId, int channelId, int modId, String modName);

    /**
     * Callback for notification of host mode changes.
     */
    void chatChannelHostTargetChanged(int userId, int channelId, String targetChannelName, int numViewers);

    /**
     * Callback for notice messages.
     */
    void chatChannelNoticeReceived(int userId, int channelId, String noticeId, HashMap<String, String> params);

    /**
     * Callback for when AutoMod flags a message the user sent.
     */
    void autoModCaughtSentMessage(int userId, int channelId);

    /**
     * Callback for when a message the user sent that was flagged for review by AutoMod was denied by a moderator.
     */
    void autoModDeniedSentMessage(int userId, int channelId);

    /**
     * Callback for when a message the user sent that was flagged for review by AutoMod was approved by a moderator.
     */
    void autoModApprovedSentMessage(int userId, int channelId);

    /**
     * Callback for moderators of a channel when AutoMod flags a message for review.
     */
    void autoModCaughtMessageForMods(
        int userId, int channelId, String messageId, String message, int senderId, String senderName, String reason);

    /**
     * Callback for moderators of a channel when a moderator approved a message under review.
     * Still fires in the case when the current user is the moderator that gave the approval.
     */
    void autoModMessageApprovedByMod(
        int userId, int channelId, String messageId, int moderatorId, String moderatorName);

    /**
     * Callback for moderators of a channel when a moderator denied a message under review.
     * Still fires in the case when the current user is the moderator that gave the denial.
     */
    void autoModMessageDeniedByMod(int userId, int channelId, String messageId, int moderatorId, String moderatorName);

    /**
     * Callback for when a cheer message the user sent that was flagged for review by AutoMod was denied by a moderator.
     * The cheer should still be sent without the message.
     */
    void autoModDeniedSentCheer(int userId, int channelId);

    /**
     * Callback for when a message the user sent that was flagged for review by AutoMod not responded to by a moderator.
     * The cheer should still be sent without the message.
     */
    void autoModTimedOutSentCheer(int userId, int channelId);

    /**
     * Callback for moderators of a channel when AutoMod flags a cheer message for review.
     */
    void autoModCaughtCheerForMods(
        int userId, int channelId, String messageId, String message, int senderId, String senderName, String reason);
}
