/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/java_chatchannellistenerproxy.h"

#include "twitchsdk/chat/chatapi.h"
#include "twitchsdk/chat/chattypes.h"
#include "twitchsdk/chat/generated/java_all.h"
#include "twitchsdk/chat/java_chatutil.h"

using namespace ttv::chat;

ttv::binding::java::JavaChatChannelListenerProxy::JavaChatChannelListenerProxy() {
  mListenerInfo = GetJavaClassInfo_IChatChannelListener(gActiveJavaEnvironment);
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelStateChanged(
  UserId userId, ChannelId channelId, ChatChannelState state, TTV_ErrorCode result) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jState,
    GetJavaInstance_SimpleEnum(
      gActiveJavaEnvironment, GetJavaClassInfo_ChatChannelState(gActiveJavaEnvironment), state));
  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jobject, jResult, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, result));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelStateChanged"],
    static_cast<jint>(userId), static_cast<jint>(channelId), jState, jResult);

  switch (state) {
    case ChatChannelState::Disconnected: {
      // Notify so the channel listener can be removed from the map
      if (mChannelDisconnectedCallback != nullptr) {
        mChannelDisconnectedCallback(userId, channelId);
      }

      break;
    }
    default:
      break;
  }
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelInfoChanged(
  UserId userId, ChannelId channelId, const ChatChannelInfo& info) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jobject, jChannelInfo, GetJavaInstance_ChatChannelInfo(gActiveJavaEnvironment, info));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelInfoChanged"],
    static_cast<jint>(userId), static_cast<jint>(channelId), jChannelInfo);
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelRestrictionsChanged(
  UserId userId, ChannelId channelId, const ChatChannelRestrictions& restrictions) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jChatRestrictions,
    GetJavaInstance_ChatChannelRestrictions(gActiveJavaEnvironment, restrictions));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelRestrictionsChanged"],
    static_cast<jint>(userId), static_cast<jint>(channelId), jChatRestrictions);
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelLocalUserChanged(
  UserId userId, ChannelId channelId, const ChatUserInfo& userInfo) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jobject, jUserInfo, GetJavaInstance_ChatUserInfo(gActiveJavaEnvironment, userInfo));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelLocalUserChanged"],
    static_cast<jint>(userId), static_cast<jint>(channelId), jUserInfo);
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelMessagesReceived(
  UserId userId, ChannelId channelId, const std::vector<LiveChatMessage>& messageList) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jArray,
    GetJavaInstance_ChatColferLiveMessageArray(gActiveJavaEnvironment, messageList));
  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelMessagesReceived"],
    static_cast<jint>(userId), static_cast<jint>(channelId), jArray);
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelSubscriptionNoticeReceived(
  UserId userId, ChannelId channelId, const SubscriptionNotice& notice) {
  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jobject, jNotice, GetJavaInstance_ChatSubscriptionNotice(gActiveJavaEnvironment, notice));

  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelSubscriptionNoticeReceived"],
    static_cast<jint>(userId), static_cast<jint>(channelId), jNotice);
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelFirstTimeChatterNoticeReceived(
  UserId userId, ChannelId channelId, const FirstTimeChatterNotice& notice) {
  AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jNotice,
    GetJavaInstance_ChatFirstTimeChatterNotice(gActiveJavaEnvironment, notice));

  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelFirstTimeChatterNoticeReceived"],
    static_cast<jint>(userId), static_cast<jint>(channelId), jNotice);
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelRaidNoticeReceived(
  UserId userId, ChannelId channelId, const RaidNotice& notice) {
  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jobject, jNotice, GetJavaInstance_ChatRaidNotice(gActiveJavaEnvironment, notice));

  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelRaidNoticeReceived"],
    static_cast<jint>(userId), static_cast<jint>(channelId), jNotice);
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelUnraidNoticeReceived(
  UserId userId, ChannelId channelId, const UnraidNotice& notice) {
  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jobject, jNotice, GetJavaInstance_ChatUnraidNotice(gActiveJavaEnvironment, notice));

  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelUnraidNoticeReceived"],
    static_cast<jint>(userId), static_cast<jint>(channelId), jNotice);
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelGenericNoticeReceived(
  UserId userId, ChannelId channelId, const GenericMessageNotice& notice) {
  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jobject, jNotice, GetJavaInstance_ChatGenericMessageNotice(gActiveJavaEnvironment, notice));

  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelGenericNoticeReceived"],
    static_cast<jint>(userId), static_cast<jint>(channelId), jNotice);
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelMessagesCleared(UserId userId, ChannelId channelId) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelMessagesCleared"],
    static_cast<jint>(userId), static_cast<jint>(channelId));
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelUserMessagesCleared(
  UserId userId, ChannelId channelId, UserId clearUserId) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelUserMessagesCleared"],
    static_cast<jint>(userId), static_cast<jint>(channelId), static_cast<jint>(clearUserId));
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelMessageDeleted(UserId userId, ChannelId channelId,
  std::string&& messageId, std::string&& senderLoginName, std::string&& deletedMessageContent) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jstring, jMessageId, GetJavaInstance_String(gActiveJavaEnvironment, messageId));
  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jstring, jSenderLoginName, GetJavaInstance_String(gActiveJavaEnvironment, senderLoginName));
  AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jstring, jDeletedMessageContent,
    GetJavaInstance_String(gActiveJavaEnvironment, deletedMessageContent));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelMessageDeleted"],
    static_cast<jint>(userId), static_cast<jint>(channelId), jMessageId, jSenderLoginName, jDeletedMessageContent);
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelModNoticeUserTimedOut(UserId userId,
  ChannelId channelId, ModerationActionInfo&& modActionInfo, uint32_t timeoutDurationSeconds, std::string&& reason) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jModActionInfo,
    GetJavaInstance_ModerationActionInfo(gActiveJavaEnvironment, modActionInfo));
  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jstring, jReason, GetJavaInstance_String(gActiveJavaEnvironment, reason));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelModNoticeUserTimedOut"],
    static_cast<jint>(userId), static_cast<jint>(channelId), jModActionInfo, static_cast<jint>(timeoutDurationSeconds),
    jReason);
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelModNoticeUserBanned(
  UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo, std::string&& reason) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jModActionInfo,
    GetJavaInstance_ModerationActionInfo(gActiveJavaEnvironment, modActionInfo));
  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jobject, jReason, GetJavaInstance_String(gActiveJavaEnvironment, reason));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelModNoticeUserBanned"],
    static_cast<jint>(userId), static_cast<jint>(channelId), jModActionInfo, jReason);
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelModNoticeUserUntimedOut(
  UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jModActionInfo,
    GetJavaInstance_ModerationActionInfo(gActiveJavaEnvironment, modActionInfo));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelModNoticeUserUntimedOut"],
    static_cast<jint>(userId), static_cast<jint>(channelId), jModActionInfo);
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelModNoticeUserUnbanned(
  UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jModActionInfo,
    GetJavaInstance_ModerationActionInfo(gActiveJavaEnvironment, modActionInfo));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelModNoticeUserUnbanned"],
    static_cast<jint>(userId), static_cast<jint>(channelId), jModActionInfo);
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelModNoticeMessageDeleted(UserId userId,
  ChannelId channelId, ModerationActionInfo&& modActionInfo, std::string&& messageId, std::string&& message) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jModActionInfo,
    GetJavaInstance_ModerationActionInfo(gActiveJavaEnvironment, modActionInfo));
  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jstring, jMessageId, GetJavaInstance_String(gActiveJavaEnvironment, messageId));
  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jstring, jMessage, GetJavaInstance_String(gActiveJavaEnvironment, message));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelModNoticeMessageDeleted"],
    static_cast<jint>(userId), static_cast<jint>(channelId), jModActionInfo, jMessageId, jMessage);
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelModNoticeClearChat(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jstring, jModName, GetJavaInstance_String(gActiveJavaEnvironment, modName));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelModNoticeClearChat"],
    static_cast<jint>(userId), static_cast<jint>(channelId), static_cast<jint>(modId), jModName);
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelModNoticeEmoteOnly(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jstring, jModName, GetJavaInstance_String(gActiveJavaEnvironment, modName));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelModNoticeEmoteOnly"],
    static_cast<jint>(userId), static_cast<jint>(channelId), static_cast<jint>(modId), jModName);
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelModNoticeEmoteOnlyOff(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jstring, jModName, GetJavaInstance_String(gActiveJavaEnvironment, modName));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelModNoticeEmoteOnlyOff"],
    static_cast<jint>(userId), static_cast<jint>(channelId), static_cast<jint>(modId), jModName);
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelModNoticeFollowersOnly(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName, uint32_t minimumFollowingDurationMinutes) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jstring, jModName, GetJavaInstance_String(gActiveJavaEnvironment, modName));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelModNoticeFollowersOnly"],
    static_cast<jint>(userId), static_cast<jint>(channelId), static_cast<jint>(modId), jModName,
    static_cast<jint>(minimumFollowingDurationMinutes));
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelModNoticeFollowersOnlyOff(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jstring, jModName, GetJavaInstance_String(gActiveJavaEnvironment, modName));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelModNoticeFollowersOnlyOff"],
    static_cast<jint>(userId), static_cast<jint>(channelId), static_cast<jint>(modId), jModName);
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelModNoticeR9K(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jstring, jModName, GetJavaInstance_String(gActiveJavaEnvironment, modName));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelModNoticeR9K"],
    static_cast<jint>(userId), static_cast<jint>(channelId), static_cast<jint>(modId), jModName);
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelModNoticeR9KOff(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jstring, jModName, GetJavaInstance_String(gActiveJavaEnvironment, modName));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelModNoticeR9KOff"],
    static_cast<jint>(userId), static_cast<jint>(channelId), static_cast<jint>(modId), jModName);
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelModNoticeSlow(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName, uint32_t slowModeDurationSeconds) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jstring, jModName, GetJavaInstance_String(gActiveJavaEnvironment, modName));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelModNoticeSlow"],
    static_cast<jint>(userId), static_cast<jint>(channelId), static_cast<jint>(modId), jModName,
    static_cast<jint>(slowModeDurationSeconds));
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelModNoticeSlowOff(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jstring, jModName, GetJavaInstance_String(gActiveJavaEnvironment, modName));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelModNoticeSlowOff"],
    static_cast<jint>(userId), static_cast<jint>(channelId), static_cast<jint>(modId), jModName);
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelModNoticeSubsOnly(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jstring, jModName, GetJavaInstance_String(gActiveJavaEnvironment, modName));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelModNoticeSubsOnly"],
    static_cast<jint>(userId), static_cast<jint>(channelId), static_cast<jint>(modId), jModName);
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelModNoticeSubsOnlyOff(
  UserId userId, ChannelId channelId, UserId modId, std::string&& modName) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jstring, jModName, GetJavaInstance_String(gActiveJavaEnvironment, modName));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelModNoticeSubsOnlyOff"],
    static_cast<jint>(userId), static_cast<jint>(channelId), static_cast<jint>(modId), jModName);
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelHostTargetChanged(
  UserId userId, ChannelId channelId, const std::string& targetChannel, uint32_t numViewers) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jstring, jTargetChannel, GetJavaInstance_String(gActiveJavaEnvironment, targetChannel));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelHostTargetChanged"],
    static_cast<jint>(userId), static_cast<jint>(channelId), jTargetChannel, static_cast<jint>(numViewers));
}

void ttv::binding::java::JavaChatChannelListenerProxy::ChatChannelNoticeReceived(
  UserId userId, ChannelId channelId, const std::string& id, const std::map<std::string, std::string>& params) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jstring, jId, GetJavaInstance_String(gActiveJavaEnvironment, id));
  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jobject, jParams, GetJavaInstance_StringHashMap(gActiveJavaEnvironment, params));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["chatChannelNoticeReceived"],
    static_cast<jint>(userId), static_cast<jint>(channelId), jId, jParams);
}

void ttv::binding::java::JavaChatChannelListenerProxy::AutoModCaughtSentMessage(UserId userId, ChannelId channelId) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["autoModCaughtSentMessage"],
    static_cast<jint>(userId), static_cast<jint>(channelId));
}

void ttv::binding::java::JavaChatChannelListenerProxy::AutoModDeniedSentMessage(UserId userId, ChannelId channelId) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["autoModDeniedSentMessage"],
    static_cast<jint>(userId), static_cast<jint>(channelId));
}

void ttv::binding::java::JavaChatChannelListenerProxy::AutoModApprovedSentMessage(UserId userId, ChannelId channelId) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["autoModApprovedSentMessage"],
    static_cast<jint>(userId), static_cast<jint>(channelId));
}

void ttv::binding::java::JavaChatChannelListenerProxy::AutoModCaughtMessageForMods(UserId userId, ChannelId channelId,
  std::string&& messageId, std::string&& message, UserId senderId, std::string&& senderName, std::string&& reason) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jstring, jMessageId, GetJavaInstance_String(gActiveJavaEnvironment, std::move(messageId)));
  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jstring, jMessage, GetJavaInstance_String(gActiveJavaEnvironment, std::move(message)));
  AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jstring, jSenderName,
    GetJavaInstance_String(gActiveJavaEnvironment, std::move(senderName)));
  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jstring, jReason, GetJavaInstance_String(gActiveJavaEnvironment, std::move(reason)));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["autoModCaughtMessageForMods"],
    static_cast<jint>(userId), static_cast<jint>(channelId), jMessageId, jMessage, static_cast<jint>(senderId),
    jSenderName, jReason);
}

void ttv::binding::java::JavaChatChannelListenerProxy::AutoModMessageApprovedByMod(
  UserId userId, ChannelId channelId, std::string&& messageId, UserId moderatorId, std::string&& moderatorName) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jstring, jMessageId, GetJavaInstance_String(gActiveJavaEnvironment, std::move(messageId)));
  AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jstring, jModeratorName,
    GetJavaInstance_String(gActiveJavaEnvironment, std::move(moderatorName)));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["autoModMessageApprovedByMod"],
    static_cast<jint>(userId), static_cast<jint>(channelId), jMessageId, static_cast<jint>(moderatorId),
    jModeratorName);
}

void ttv::binding::java::JavaChatChannelListenerProxy::AutoModMessageDeniedByMod(
  UserId userId, ChannelId channelId, std::string&& messageId, UserId moderatorId, std::string&& moderatorName) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jstring, jMessageId, GetJavaInstance_String(gActiveJavaEnvironment, std::move(messageId)));
  AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jstring, jModeratorName,
    GetJavaInstance_String(gActiveJavaEnvironment, std::move(moderatorName)));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["autoModMessageDeniedByMod"],
    static_cast<jint>(userId), static_cast<jint>(channelId), jMessageId, static_cast<jint>(moderatorId),
    jModeratorName);
}

void ttv::binding::java::JavaChatChannelListenerProxy::AutoModDeniedSentCheer(UserId userId, ChannelId channelId) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["autoModDeniedSentCheer"],
    static_cast<jint>(userId), static_cast<jint>(channelId));
}

void ttv::binding::java::JavaChatChannelListenerProxy::AutoModTimedOutSentCheer(UserId userId, ChannelId channelId) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["autoModTimedOutSentCheer"],
    static_cast<jint>(userId), static_cast<jint>(channelId));
}

void ttv::binding::java::JavaChatChannelListenerProxy::AutoModCaughtCheerForMods(UserId userId, ChannelId channelId,
  std::string&& messageId, std::string&& message, UserId senderId, std::string&& senderName, std::string&& reason) {
  jobject jListener = mJavaListener.GetInstance();
  if (jListener == nullptr) {
    return;
  }

  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jstring, jMessageId, GetJavaInstance_String(gActiveJavaEnvironment, std::move(messageId)));
  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jstring, jMessage, GetJavaInstance_String(gActiveJavaEnvironment, std::move(message)));
  AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jstring, jSenderName,
    GetJavaInstance_String(gActiveJavaEnvironment, std::move(senderName)));
  AUTO_DELETE_LOCAL_REF(
    gActiveJavaEnvironment, jstring, jReason, GetJavaInstance_String(gActiveJavaEnvironment, std::move(reason)));

  gActiveJavaEnvironment->CallVoidMethod(jListener, mListenerInfo.methods["autoModCaughtCheerForMods"],
    static_cast<jint>(userId), static_cast<jint>(channelId), jMessageId, jMessage, static_cast<jint>(senderId),
    jSenderName, jReason);
}
