/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/java_chatutil.h"

#include "twitchsdk/chat/generated/colfer_chatmessagearray.h"
#include "twitchsdk/chat/generated/java_all.h"
#include "twitchsdk/chat/ibitsstatus.h"
#include "twitchsdk/chat/ichannelchatroommanager.h"
#include "twitchsdk/chat/ichatchannel.h"
#include "twitchsdk/chat/ichatchannelproperties.h"
#include "twitchsdk/chat/ichatcommentmanager.h"
#include "twitchsdk/chat/ichatraid.h"
#include "twitchsdk/chat/ichatroom.h"
#include "twitchsdk/chat/ichatroomnotifications.h"
#include "twitchsdk/chat/ifollowersstatus.h"
#include "twitchsdk/chat/ifollowingstatus.h"
#include "twitchsdk/chat/isubscribersstatus.h"
#include "twitchsdk/chat/isubscriptionsnotifications.h"
#include "twitchsdk/core/java_coreutil.h"

#include <functional>
#include <list>

using namespace ttv;
using namespace ttv::chat;
using namespace ttv::binding::java;

extern JNIEnv* ttv::binding::java::
  gActiveJavaEnvironment;  //!< The is cached on every call into native code on the main thread so that it's current.  Never use from another thread.

ttv::binding::java::JavaNativeProxyRegistry<ChatAPI, ChatApiContext> ttv::binding::java::gChatApiNativeProxyRegistry;
ttv::binding::java::JavaNativeProxyRegistry<IChatRaid, ChatApiContext> ttv::binding::java::gIChatRaidInstanceRegistry;
ttv::binding::java::JavaNativeProxyRegistry<IChatChannel, ChatApiContext> ttv::binding::java::gIChatChannelRegistry;
ttv::binding::java::JavaNativeProxyRegistry<IChannelChatRoomManager, ChatApiContext>
  ttv::binding::java::gIChannelChatRoomManagerInstanceRegistry;
ttv::binding::java::JavaNativeProxyRegistry<IChatRoom, ChatApiContext> ttv::binding::java::gIChatRoomInstanceRegistry;
ttv::binding::java::JavaNativeProxyRegistry<IChatRoomNotifications, ChatApiContext>
  ttv::binding::java::gIChatRoomNotificationsInstanceRegistry;
ttv::binding::java::JavaNativeProxyRegistry<IBitsStatus, ChatApiContext>
  ttv::binding::java::gIBitsStatusInstanceRegistry;
ttv::binding::java::JavaNativeProxyRegistry<IFollowersStatus, ChatApiContext>
  ttv::binding::java::gIFollowersStatusInstanceRegistry;
ttv::binding::java::JavaNativeProxyRegistry<IFollowingStatus, ChatApiContext>
  ttv::binding::java::gIFollowingStatusInstanceRegistry;
ttv::binding::java::JavaNativeProxyRegistry<ISubscribersStatus, ChatApiContext>
  ttv::binding::java::gISubscribersStatusInstanceRegistry;
ttv::binding::java::JavaNativeProxyRegistry<IChatChannelProperties, ChatApiContext>
  ttv::binding::java::gIChatChannelPropertiesInstanceRegistry;
ttv::binding::java::JavaNativeProxyRegistry<IChatCommentManager, ChatApiContext>
  ttv::binding::java::gIChatCommentManagerInstanceRegistry;
ttv::binding::java::JavaNativeProxyRegistry<ChatMessageHandler, ChatMessageHandlerContext>
  ttv::binding::java::gChatMessageHandlerRegistry;
ttv::binding::java::JavaNativeProxyRegistry<ChatRoomMessageHandler, ChatRoomMessageHandlerContext>
  ttv::binding::java::gChatRoomMessageHandlerRegistry;
ttv::binding::java::JavaNativeProxyRegistry<ISubscriptionsNotifications, ChatApiContext>
  ttv::binding::java::gISubscriptionsNotificationsInstanceRegistry;
ttv::binding::java::JavaNativeProxyRegistry<ISquadNotifications, ChatApiContext>
  ttv::binding::java::gISquadNotificationsInstanceRegistry;
ttv::binding::java::JavaNativeProxyRegistry<IMultiviewNotifications, ChatApiContext>
  ttv::binding::java::gIMultiviewNotificationsInstanceRegistry;

void ttv::binding::java::LoadAllChatJavaClassInfo(JNIEnv* jEnv) {
  GetJavaClassInfo_ChatAPI_BlockChangeCallback(jEnv);
  GetJavaClassInfo_ChatAPI_FetchEmoticonSetsCallback(jEnv);
  GetJavaClassInfo_ChatAPI_FetchBadgesCallback(jEnv);
  GetJavaClassInfo_ChatAPI_SendMessageCallback(jEnv);
  GetJavaClassInfo_ChatAPI_FetchBlockedUsersCallback(jEnv);
  GetJavaClassInfo_ChatAPI_BanUserCallback(jEnv);
  GetJavaClassInfo_ChatAPI_UnbanUserCallback(jEnv);
  GetJavaClassInfo_ChatAPI_ModUserCallback(jEnv);
  GetJavaClassInfo_ChatAPI_UnmodUserCallback(jEnv);
  GetJavaClassInfo_ChatAPI_UpdateUserColorCallback(jEnv);
  GetJavaClassInfo_ChatAPI_FetchChannelModeratorsCallback(jEnv);

  GetJavaClassInfo_ChatBadge(jEnv);
  GetJavaClassInfo_ChatBadgeAction(jEnv);
  GetJavaClassInfo_ChatBadgeImage(jEnv);
  GetJavaClassInfo_ChatBadgeSet(jEnv);
  GetJavaClassInfo_ChatBadgeVersion(jEnv);
  GetJavaClassInfo_ChatBitsToken(jEnv);
  GetJavaClassInfo_ChatChannelInfo(jEnv);
  GetJavaClassInfo_ChatChannelProxy(jEnv);
  GetJavaClassInfo_ChatChannelRestrictions(jEnv);
  GetJavaClassInfo_ChatChannelState(jEnv);
  GetJavaClassInfo_ChatEmoticon(jEnv);
  GetJavaClassInfo_ChatEmoticonSet(jEnv);
  GetJavaClassInfo_ChatEmoticonToken(jEnv);
  GetJavaClassInfo_ChatFeatureFlags(jEnv);
  GetJavaClassInfo_ChatLiveMessage(jEnv);
  GetJavaClassInfo_ChatMentionToken(jEnv);
  GetJavaClassInfo_ChatMessageInfo(jEnv);
  GetJavaClassInfo_ChatMessageBadge(jEnv);
  GetJavaClassInfo_ChatMessageFlags(jEnv);
  GetJavaClassInfo_ChatMessageHandler_ICallbacks(jEnv);
  GetJavaClassInfo_ChatMessageToken(jEnv);
  GetJavaClassInfo_ChatMessageTokenType(jEnv);
  GetJavaClassInfo_ChatRaidProxy(jEnv);
  GetJavaClassInfo_ChatRaidStatus(jEnv);
  GetJavaClassInfo_ChatRestrictionReason(jEnv);
  GetJavaClassInfo_ChatRoomMessageHandler_ICallbacks(jEnv);
  GetJavaClassInfo_ChatRoomMessageHandler_CommandError(jEnv);
  GetJavaClassInfo_ChatSubscriptionNotice(jEnv);
  GetJavaClassInfo_ChatSubscriptionNoticePlan(jEnv);
  GetJavaClassInfo_ChatSubscriptionNoticeType(jEnv);
  GetJavaClassInfo_ChatSubscriptionNoticeRecipient(jEnv);
  GetJavaClassInfo_ChatTextToken(jEnv);
  GetJavaClassInfo_ChatTokenizationOptions(jEnv);
  GetJavaClassInfo_ChatUrlToken(jEnv);
  GetJavaClassInfo_ChatUserInfo(jEnv);
  GetJavaClassInfo_ChatUserMode(jEnv);
  GetJavaClassInfo_ChatWhisperMessage(jEnv);
  GetJavaClassInfo_BitsStatusProxy(jEnv);
  GetJavaClassInfo_FollowersStatusProxy(jEnv);
  GetJavaClassInfo_FollowingStatusProxy(jEnv);
  GetJavaClassInfo_SubscribersStatusProxy(jEnv);
  GetJavaClassInfo_ChatBadgeEntitlement(jEnv);
  GetJavaClassInfo_ChatBitsReceivedEvent(jEnv);
  GetJavaClassInfo_ChatFollowerAddedEvent(jEnv);
  GetJavaClassInfo_ChatSubscriberAddedEvent(jEnv);
  GetJavaClassInfo_IBitsStatus(jEnv);
  GetJavaClassInfo_IBitsListener(jEnv);
  GetJavaClassInfo_IFollowersStatus(jEnv);
  GetJavaClassInfo_IFollowersListener(jEnv);
  GetJavaClassInfo_ISubscribersStatus(jEnv);
  GetJavaClassInfo_ISubscribersListener(jEnv);
  GetJavaClassInfo_IChatAPIListener(jEnv);
  GetJavaClassInfo_IChatChannel(jEnv);
  GetJavaClassInfo_IChatChannelListener(jEnv);
  GetJavaClassInfo_IChatRaid(jEnv);
  GetJavaClassInfo_IChatRaidListener(jEnv);
  GetJavaClassInfo_IChatUserThreadsListener(jEnv);
  GetJavaClassInfo_SendRoomMessageError(jEnv);
  GetJavaClassInfo_CreateRoomError(jEnv);
  GetJavaClassInfo_UpdateRoomError(jEnv);
  GetJavaClassInfo_UpdateRoomModesError(jEnv);
  GetJavaClassInfo_ISubscriptionsNotifications(jEnv);
  GetJavaClassInfo_ExtensionMessage(jEnv);
}

jobject ttv::binding::java::GetJavaInstance_ChatUserInfo(JNIEnv* jEnv, const ChatUserInfo& user) {
  JavaClassInfo& chatUserInfo = GetJavaClassInfo_ChatUserInfo(jEnv);

  jobject jUserInstance = jEnv->NewObject(chatUserInfo.klass, chatUserInfo.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jUserName, GetJavaInstance_String(jEnv, user.userName));
  jEnv->SetObjectField(jUserInstance, chatUserInfo.fields["userName"], jUserName);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jDisplayName, GetJavaInstance_String(jEnv, user.displayName));
  jEnv->SetObjectField(jUserInstance, chatUserInfo.fields["displayName"], jDisplayName);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jChatUserMode, GetJavaInstance_ChatUserMode(jEnv, user.userMode));
  jEnv->SetObjectField(jUserInstance, chatUserInfo.fields["userMode"], jChatUserMode);

  jEnv->SetIntField(jUserInstance, chatUserInfo.fields["nameColorARGB"], user.nameColorARGB);
  jEnv->SetIntField(jUserInstance, chatUserInfo.fields["userId"], user.userId);

  return jUserInstance;
}

jobject ttv::binding::java::GetJavaInstance_ChatUserInfoArray(JNIEnv* jEnv, const std::vector<ChatUserInfo>& list) {
  JavaClassInfo& chatUserInfo = GetJavaClassInfo_ChatUserInfo(jEnv);
  jobjectArray jUserArray = jEnv->NewObjectArray(static_cast<jsize>(list.size()), chatUserInfo.klass, nullptr);

  jsize index = 0;
  for (const auto& info : list) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jUserInstance, GetJavaInstance_ChatUserInfo(jEnv, info));

    jEnv->SetObjectArrayElement(jUserArray, index, jUserInstance);
    index++;
  }

  return jUserArray;
}

jobject ttv::binding::java::GetJavaInstance_ChatChannelInfo(JNIEnv* jEnv, const ChatChannelInfo& channelInfo) {
  JavaClassInfo& chatChannelInfoClassInfo = GetJavaClassInfo_ChatChannelInfo(jEnv);

  jobject jChannelInfo = jEnv->NewObject(chatChannelInfoClassInfo.klass, chatChannelInfoClassInfo.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jChannelName, GetJavaInstance_String(jEnv, channelInfo.name));
  jEnv->SetObjectField(jChannelInfo, chatChannelInfoClassInfo.fields["name"], jChannelName);

  AUTO_DELETE_LOCAL_REF(
    jEnv, jstring, jBroadcasterLanguage, GetJavaInstance_String(jEnv, channelInfo.broadcasterLanguage));
  jEnv->SetObjectField(jChannelInfo, chatChannelInfoClassInfo.fields["broadcasterLanguage"], jBroadcasterLanguage);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jChatRestrictionReason,
    GetJavaInstance_ChatRestrictionReason(jEnv, channelInfo.localUserRestriction));
  jEnv->SetObjectField(jChannelInfo, chatChannelInfoClassInfo.fields["localUserRestriction"], jChatRestrictionReason);

  return jChannelInfo;
}

jobject ttv::binding::java::GetJavaInstance_ChatChannelRestrictions(
  JNIEnv* jEnv, const ChatChannelRestrictions& restrictions) {
  JavaClassInfo& chatChannelRestrictionsClassInfo = GetJavaClassInfo_ChatChannelRestrictions(jEnv);

  jobject jChannelRestrictions =
    jEnv->NewObject(chatChannelRestrictionsClassInfo.klass, chatChannelRestrictionsClassInfo.methods["<init>"]);

  jEnv->SetIntField(
    jChannelRestrictions, chatChannelRestrictionsClassInfo.fields["followersDuration"], restrictions.followersDuration);
  jEnv->SetIntField(
    jChannelRestrictions, chatChannelRestrictionsClassInfo.fields["slowModeDuration"], restrictions.slowModeDuration);
  jEnv->SetBooleanField(
    jChannelRestrictions, chatChannelRestrictionsClassInfo.fields["emoteOnly"], restrictions.emoteOnly);
  jEnv->SetBooleanField(
    jChannelRestrictions, chatChannelRestrictionsClassInfo.fields["verifiedOnly"], restrictions.verifiedOnly);
  jEnv->SetBooleanField(
    jChannelRestrictions, chatChannelRestrictionsClassInfo.fields["followersOnly"], restrictions.followersOnly);
  jEnv->SetBooleanField(
    jChannelRestrictions, chatChannelRestrictionsClassInfo.fields["subscribersOnly"], restrictions.subscribersOnly);
  jEnv->SetBooleanField(
    jChannelRestrictions, chatChannelRestrictionsClassInfo.fields["slowMode"], restrictions.slowMode);
  jEnv->SetBooleanField(jChannelRestrictions, chatChannelRestrictionsClassInfo.fields["r9k"], restrictions.r9k);
  jEnv->SetIntField(jChannelRestrictions, chatChannelRestrictionsClassInfo.fields["slowModeSetAt"],
    static_cast<jint>(restrictions.slowModeSetAt));

  return jChannelRestrictions;
}

jobject ttv::binding::java::GetJavaInstance_ChatUserMode(JNIEnv* jEnv, const UserMode& mode) {
  JavaClassInfo& info = GetJavaClassInfo_ChatUserMode(jEnv);
  jobject jUserMode = jEnv->NewObject(info.klass, info.methods["<init>"]);

  jEnv->SetBooleanField(jUserMode, info.fields["moderator"], mode.moderator);
  jEnv->SetBooleanField(jUserMode, info.fields["broadcaster"], mode.broadcaster);
  jEnv->SetBooleanField(jUserMode, info.fields["administrator"], mode.administrator);
  jEnv->SetBooleanField(jUserMode, info.fields["staff"], mode.staff);
  jEnv->SetBooleanField(jUserMode, info.fields["system"], mode.system);
  jEnv->SetBooleanField(jUserMode, info.fields["globalModerator"], mode.globalModerator);
  jEnv->SetBooleanField(jUserMode, info.fields["banned"], mode.banned);
  jEnv->SetBooleanField(jUserMode, info.fields["subscriber"], mode.subscriber);
  jEnv->SetBooleanField(jUserMode, info.fields["vip"], mode.vip);

  return jUserMode;
}

jobject ttv::binding::java::GetJavaInstance_ChatRestrictionReason(JNIEnv* jEnv, const RestrictionReason& reason) {
  JavaClassInfo& info = GetJavaClassInfo_ChatRestrictionReason(jEnv);
  jobject jReason = jEnv->NewObject(info.klass, info.methods["<init>"]);

  jEnv->SetBooleanField(jReason, info.fields["anonymous"], reason.anonymous);
  jEnv->SetBooleanField(jReason, info.fields["subscribersOnly"], reason.subscribersOnly);
  jEnv->SetBooleanField(jReason, info.fields["slowMode"], reason.slowMode);
  jEnv->SetBooleanField(jReason, info.fields["timeout"], reason.timeout);
  jEnv->SetBooleanField(jReason, info.fields["banned"], reason.banned);

  return jReason;
}

jobject ttv::binding::java::GetJavaInstance_ChatSubscriptionNotice(JNIEnv* jEnv, const SubscriptionNotice& notice) {
  JavaClassInfo& info = GetJavaClassInfo_ChatSubscriptionNotice(jEnv);
  jobject jNotice = jEnv->NewObject(info.klass, info.methods["<init>"]);

  if (notice.userMessage != nullptr) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jUserMessage, GetJavaInstance_ChatMessageInfo(jEnv, *notice.userMessage));
    jEnv->SetObjectField(jNotice, info.fields["userMessage"], jUserMessage);
  }

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jSystemMessage, GetJavaInstance_String(jEnv, notice.systemMessage));
  jEnv->SetObjectField(jNotice, info.fields["systemMessage"], jSystemMessage);

  jEnv->SetIntField(jNotice, info.fields["subStreakMonthCount"], notice.subStreakMonthCount);
  jEnv->SetIntField(jNotice, info.fields["subCumulativeMonthCount"], notice.subCumulativeMonthCount);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jType,
    GetJavaInstance_SimpleEnum(jEnv, GetJavaClassInfo_ChatSubscriptionNoticeType(jEnv), notice.type));
  jEnv->SetObjectField(jNotice, info.fields["type"], jType);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jPlan,
    GetJavaInstance_SimpleEnum(jEnv, GetJavaClassInfo_ChatSubscriptionNoticePlan(jEnv), notice.plan));
  jEnv->SetObjectField(jNotice, info.fields["plan"], jPlan);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jPlanDisplayName, GetJavaInstance_String(jEnv, notice.planDisplayName));
  jEnv->SetObjectField(jNotice, info.fields["planDisplayName"], jPlanDisplayName);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jMessageId, GetJavaInstance_String(jEnv, notice.messageId));
  jEnv->SetObjectField(jNotice, info.fields["messageId"], jMessageId);

  jEnv->SetBooleanField(jNotice, info.fields["shouldShowSubStreak"], notice.shouldShowSubStreak);

  if (notice.type == SubscriptionNotice::Type::SubGift) {
    AUTO_DELETE_LOCAL_REF(
      jEnv, jobject, jRecipient, GetJavaInstance_ChatSubscriptionNoticeRecipient(jEnv, notice.recipient));
    jEnv->SetObjectField(jNotice, info.fields["recipient"], jRecipient);
  } else if (notice.type == SubscriptionNotice::Type::SubMassGift) {
    jEnv->SetIntField(jNotice, info.fields["massGiftCount"], notice.massGiftCount);
  }

  if (notice.type == SubscriptionNotice::Type::SubGift || notice.type == SubscriptionNotice::Type::SubMassGift) {
    jEnv->SetIntField(jNotice, info.fields["senderCount"], notice.senderCount);
  }

  if (notice.type == SubscriptionNotice::Type::ExtendSub) {
    jEnv->SetIntField(jNotice, info.fields["benefitEndMonth"], notice.benefitEndMonth);
  }

  return jNotice;
}

jobject ttv::binding::java::GetJavaInstance_ChatSubscriptionNoticeRecipient(
  JNIEnv* jEnv, const SubscriptionNotice::Recipient& recipient) {
  JavaClassInfo& info = GetJavaClassInfo_ChatSubscriptionNoticeRecipient(jEnv);
  jobject jRecipient = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jUserName, GetJavaInstance_String(jEnv, recipient.userName));
  jEnv->SetObjectField(jRecipient, info.fields["userName"], jUserName);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jDisplayName, GetJavaInstance_String(jEnv, recipient.displayName));
  jEnv->SetObjectField(jRecipient, info.fields["displayName"], jDisplayName);

  jEnv->SetIntField(jRecipient, info.fields["userId"], static_cast<jint>(recipient.userId));

  return jRecipient;
}

jobject ttv::binding::java::GetJavaInstance_ChatFirstTimeChatterNotice(
  JNIEnv* jEnv, const FirstTimeChatterNotice& notice) {
  JavaClassInfo& info = GetJavaClassInfo_ChatFirstTimeChatterNotice(jEnv);
  jobject jNotice = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jUserMessage, GetJavaInstance_ChatMessageInfo(jEnv, notice.userMessage));
  jEnv->SetObjectField(jNotice, info.fields["userMessage"], jUserMessage);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jSystemMessage, GetJavaInstance_String(jEnv, notice.systemMessage));
  jEnv->SetObjectField(jNotice, info.fields["systemMessage"], jSystemMessage);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jMessageId, GetJavaInstance_String(jEnv, notice.messageId));
  jEnv->SetObjectField(jNotice, info.fields["messageId"], jMessageId);

  return jNotice;
}

jobject ttv::binding::java::GetJavaInstance_ChatRaidNotice(JNIEnv* jEnv, const RaidNotice& notice) {
  JavaClassInfo& info = GetJavaClassInfo_ChatRaidNotice(jEnv);
  jobject jNotice = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jSystemMessage, GetJavaInstance_String(jEnv, notice.systemMessage));
  jEnv->SetObjectField(jNotice, info.fields["systemMessage"], jSystemMessage);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jProfileImageUrl, GetJavaInstance_String(jEnv, notice.profileImageUrl));
  jEnv->SetObjectField(jNotice, info.fields["profileImageUrl"], jProfileImageUrl);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jRaidingUserInfo, GetJavaInstance_UserInfo(jEnv, notice.raidingUserInfo));
  jEnv->SetObjectField(jNotice, info.fields["raidingUserInfo"], jRaidingUserInfo);

  jEnv->SetIntField(jNotice, info.fields["viewerCount"], static_cast<jint>(notice.viewerCount));

  return jNotice;
}

jobject ttv::binding::java::GetJavaInstance_ChatUnraidNotice(JNIEnv* jEnv, const UnraidNotice& notice) {
  JavaClassInfo& info = GetJavaClassInfo_ChatUnraidNotice(jEnv);
  jobject jNotice = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jSystemMessage, GetJavaInstance_String(jEnv, notice.systemMessage));
  jEnv->SetObjectField(jNotice, info.fields["systemMessage"], jSystemMessage);

  return jNotice;
}

jobject ttv::binding::java::GetJavaInstance_ChatGenericMessageNotice(JNIEnv* jEnv, const GenericMessageNotice& notice) {
  JavaClassInfo& info = GetJavaClassInfo_ChatGenericMessageNotice(jEnv);
  jobject jNotice = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jMessageInfo, GetJavaInstance_ChatMessageInfo(jEnv, notice.messageInfo));
  jEnv->SetObjectField(jNotice, info.fields["messageInfo"], jMessageInfo);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jMessageId, GetJavaInstance_String(jEnv, notice.messageId));
  jEnv->SetObjectField(jNotice, info.fields["messageId"], jMessageId);

  return jNotice;
}

jobject ttv::binding::java::GetJavaInstance_ChatTokenizationOptions(JNIEnv* jEnv, const TokenizationOptions& options) {
  JavaClassInfo& info = GetJavaClassInfo_ChatTokenizationOptions(jEnv);
  jobject jOptions = jEnv->NewObject(info.klass, info.methods["<init>"]);

  jEnv->SetBooleanField(jOptions, info.fields["emoticons"], options.emoticons);
  jEnv->SetBooleanField(jOptions, info.fields["mentions"], options.mentions);
  jEnv->SetBooleanField(jOptions, info.fields["urls"], options.urls);
  jEnv->SetBooleanField(jOptions, info.fields["bits"], options.bits);

  return jOptions;
}

jobject ttv::binding::java::GetJavaInstance_ChatMessageFlags(JNIEnv* jEnv, const MessageInfo::Flags& flags) {
  JavaClassInfo& info = GetJavaClassInfo_ChatMessageFlags(jEnv);
  jobject jFlags = jEnv->NewObject(info.klass, info.methods["<init>"]);

  jEnv->SetBooleanField(jFlags, info.fields["action"], flags.action);
  jEnv->SetBooleanField(jFlags, info.fields["notice"], flags.notice);
  jEnv->SetBooleanField(jFlags, info.fields["ignored"], flags.ignored);
  jEnv->SetBooleanField(jFlags, info.fields["deleted"], flags.deleted);
  jEnv->SetBooleanField(jFlags, info.fields["containsBits"], flags.containsBits);

  return jFlags;
}

jobject ttv::binding::java::GetJavaInstance_ChatMessageToken(JNIEnv* jEnv, const MessageToken& token) {
  JavaClassInfo& chatTokenInfo = GetJavaClassInfo_ChatMessageToken(jEnv);
  JavaClassInfo& chatTextTokenInfo = GetJavaClassInfo_ChatTextToken(jEnv);
  JavaClassInfo& chatEmoticonTokenInfo = GetJavaClassInfo_ChatEmoticonToken(jEnv);
  JavaClassInfo& chatMentionTokenInfo = GetJavaClassInfo_ChatMentionToken(jEnv);
  JavaClassInfo& chatUrlTokenInfo = GetJavaClassInfo_ChatUrlToken(jEnv);
  JavaClassInfo& chatBitsTokenInfo = GetJavaClassInfo_ChatBitsToken(jEnv);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jToken, nullptr);
  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jType,
    GetJavaInstance_SimpleEnum(jEnv, GetJavaClassInfo_ChatMessageTokenType(jEnv), token.GetType()));

  switch (token.GetType()) {
    case MessageToken::Type::Text: {
      const auto& textToken = static_cast<const TextToken&>(token);

      jToken = jEnv->NewObject(chatTextTokenInfo.klass, chatTextTokenInfo.methods["<init>"]);
      AUTO_DELETE_LOCAL_REF(jEnv, jstring, jText, GetJavaInstance_String(jEnv, textToken.text));
      AUTO_DELETE_LOCAL_REF(jEnv, jobject, jAutoModFlags, GetJavaInstance_AutoModFlags(jEnv, textToken.autoModFlags));
      jEnv->SetObjectField(jToken, chatTextTokenInfo.fields["text"], jText);
      jEnv->SetObjectField(jToken, chatTextTokenInfo.fields["autoModFlags"], jAutoModFlags);
      break;
    }
    case MessageToken::Type::Emoticon: {
      const auto& emoticonToken = static_cast<const EmoticonToken&>(token);

      jToken = jEnv->NewObject(chatEmoticonTokenInfo.klass, chatEmoticonTokenInfo.methods["<init>"]);
      AUTO_DELETE_LOCAL_REF(jEnv, jstring, jEmoticonText, GetJavaInstance_String(jEnv, emoticonToken.emoticonText));
      jEnv->SetObjectField(jToken, chatEmoticonTokenInfo.fields["emoticonText"], jEmoticonText);

      AUTO_DELETE_LOCAL_REF(jEnv, jstring, jEmoticonId, GetJavaInstance_String(jEnv, emoticonToken.emoticonId));
      jEnv->SetObjectField(jToken, chatEmoticonTokenInfo.fields["emoticonId"], jEmoticonId);

      break;
    }
    case MessageToken::Type::Mention: {
      const auto& mentionToken = static_cast<const MentionToken&>(token);

      jToken = jEnv->NewObject(chatMentionTokenInfo.klass, chatMentionTokenInfo.methods["<init>"]);
      AUTO_DELETE_LOCAL_REF(jEnv, jstring, jUserName, GetJavaInstance_String(jEnv, mentionToken.userName));
      AUTO_DELETE_LOCAL_REF(jEnv, jstring, jText, GetJavaInstance_String(jEnv, mentionToken.text));
      jEnv->SetObjectField(jToken, chatMentionTokenInfo.fields["userName"], jUserName);
      jEnv->SetObjectField(jToken, chatMentionTokenInfo.fields["text"], jText);
      jEnv->SetBooleanField(
        jToken, chatMentionTokenInfo.fields["isLocalUser"], static_cast<jboolean>(mentionToken.isLocalUser));
      break;
    }
    case MessageToken::Type::Url: {
      const auto& urlToken = static_cast<const UrlToken&>(token);

      jToken = jEnv->NewObject(chatUrlTokenInfo.klass, chatUrlTokenInfo.methods["<init>"]);
      AUTO_DELETE_LOCAL_REF(jEnv, jstring, jText, GetJavaInstance_String(jEnv, urlToken.url));
      jEnv->SetObjectField(jToken, chatUrlTokenInfo.fields["url"], jText);
      jEnv->SetBooleanField(jToken, chatUrlTokenInfo.fields["hidden"], static_cast<jboolean>(urlToken.hidden));
      break;
    }
    case MessageToken::Type::Bits: {
      const auto& bitsToken = static_cast<const BitsToken&>(token);

      jToken = jEnv->NewObject(chatBitsTokenInfo.klass, chatBitsTokenInfo.methods["<init>"]);
      AUTO_DELETE_LOCAL_REF(jEnv, jstring, jPrefix, GetJavaInstance_String(jEnv, bitsToken.prefix));
      jEnv->SetObjectField(jToken, chatBitsTokenInfo.fields["prefix"], jPrefix);
      jEnv->SetIntField(jToken, chatBitsTokenInfo.fields["numBits"], static_cast<jint>(bitsToken.numBits));
      break;
    }
  }

  if (jToken != nullptr) {
    jEnv->SetObjectField(jToken, chatTokenInfo.fields["type"], jType);
  }

  return jToken;
}

jobject ttv::binding::java::GetJavaInstance_ChatMessageBadge(JNIEnv* jEnv, const MessageBadge& badge) {
  JavaClassInfo& info = GetJavaClassInfo_ChatMessageBadge(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jBadgeSet, GetJavaInstance_String(jEnv, badge.name));
  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jVersion, GetJavaInstance_String(jEnv, badge.version));

  jEnv->SetObjectField(jInstance, info.fields["name"], jBadgeSet);
  jEnv->SetObjectField(jInstance, info.fields["version"], jVersion);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_ChatBadgeImage(JNIEnv* jEnv, const BadgeImage& image) {
  JavaClassInfo& chatBadgeImageInfo = GetJavaClassInfo_ChatBadgeImage(jEnv);

  jobject jBadgeImage = jEnv->NewObject(chatBadgeImageInfo.klass, chatBadgeImageInfo.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jUrl, GetJavaInstance_String(jEnv, image.url));

  jEnv->SetFloatField(jBadgeImage, chatBadgeImageInfo.fields["scale"], static_cast<jfloat>(image.scale));
  jEnv->SetObjectField(jBadgeImage, chatBadgeImageInfo.fields["url"], jUrl);

  return jBadgeImage;
}

jobject ttv::binding::java::GetJavaInstance_ChatBadgeVersion(JNIEnv* jEnv, const BadgeVersion& badgeVersion) {
  JavaClassInfo& info = GetJavaClassInfo_ChatBadgeVersion(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jName, GetJavaInstance_String(jEnv, badgeVersion.name));
  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jTitle, GetJavaInstance_String(jEnv, badgeVersion.title));
  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jDescription, GetJavaInstance_String(jEnv, badgeVersion.description));
  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jClickUrl, GetJavaInstance_String(jEnv, badgeVersion.clickUrl));

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jClickAction,
    GetJavaInstance_SimpleEnum(jEnv, GetJavaClassInfo_ChatBadgeAction(jEnv), badgeVersion.clickAction));

  jEnv->SetObjectField(jInstance, info.fields["name"], jName);
  jEnv->SetObjectField(jInstance, info.fields["title"], jTitle);
  jEnv->SetObjectField(jInstance, info.fields["description"], jDescription);
  jEnv->SetObjectField(jInstance, info.fields["clickUrl"], jClickUrl);
  jEnv->SetObjectField(jInstance, info.fields["clickAction"], jClickAction);

  // Create the images array
  jobjectArray jImagesArray = GetJavaInstance_Array(jEnv, GetJavaClassInfo_ChatBadgeImage(jEnv),
    static_cast<uint32_t>(badgeVersion.images.size()), [jEnv, &badgeVersion](uint32_t index) -> jobject {
      return GetJavaInstance_ChatBadgeImage(jEnv, badgeVersion.images[index]);
    });
  AUTO_DELETE_LOCAL_REF_NO_DECLARE(jEnv, jobjectArray, jImagesArray);

  jEnv->SetObjectField(jInstance, info.fields["images"], jImagesArray);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_ChatBadge(JNIEnv* jEnv, const Badge& badge) {
  JavaClassInfo& info = GetJavaClassInfo_ChatBadge(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jName, GetJavaInstance_String(jEnv, badge.name));
  jEnv->SetObjectField(jInstance, info.fields["name"], jName);

  // Create the versions map
  jobject jVersions =
    GetJavaInstance_HashMap(jEnv, badge.versions, [jEnv](const std::pair<std::string, BadgeVersion>& pair) {
      jobject jKey = GetJavaInstance_String(jEnv, pair.first);
      jobject jValue = GetJavaInstance_ChatBadgeVersion(jEnv, pair.second);

      return std::pair<jobject, jobject>{jKey, jValue};
    });
  AUTO_DELETE_LOCAL_REF_NO_DECLARE(jEnv, jobject, jVersions);

  jEnv->SetObjectField(jInstance, info.fields["versions"], jVersions);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_ChatBadgeSet(JNIEnv* jEnv, const BadgeSet& badgeSet) {
  JavaClassInfo& info = GetJavaClassInfo_ChatBadgeSet(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jLanguage, GetJavaInstance_String(jEnv, badgeSet.language));
  jEnv->SetObjectField(jInstance, info.fields["language"], jLanguage);

  // Create the badges map
  jobject jBadges = GetJavaInstance_HashMap(jEnv, badgeSet.badges, [jEnv](const std::pair<std::string, Badge>& pair) {
    jobject jKey = GetJavaInstance_String(jEnv, pair.first);
    jobject jValue = GetJavaInstance_ChatBadge(jEnv, pair.second);

    return std::pair<jobject, jobject>{jKey, jValue};
  });
  AUTO_DELETE_LOCAL_REF_NO_DECLARE(jEnv, jobject, jBadges);

  jEnv->SetObjectField(jInstance, info.fields["badges"], jBadges);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_ChatMessageInfo(JNIEnv* jEnv, const MessageInfo& msg) {
  JavaClassInfo& chatMessageInfo = GetJavaClassInfo_ChatMessageInfo(jEnv);
  JavaClassInfo& chatTokenInfo = GetJavaClassInfo_ChatMessageToken(jEnv);
  JavaClassInfo& chatBadgeInfo = GetJavaClassInfo_ChatMessageBadge(jEnv);

  jobject jMessageInstance = jEnv->NewObject(chatMessageInfo.klass, chatMessageInfo.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jUserName, GetJavaInstance_String(jEnv, msg.userName));
  jEnv->SetObjectField(jMessageInstance, chatMessageInfo.fields["userName"], jUserName);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jDisplayName, GetJavaInstance_String(jEnv, msg.displayName));
  jEnv->SetObjectField(jMessageInstance, chatMessageInfo.fields["displayName"], jDisplayName);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jChatUserMode, GetJavaInstance_ChatUserMode(jEnv, msg.userMode));
  jEnv->SetObjectField(jMessageInstance, chatMessageInfo.fields["userMode"], jChatUserMode);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jChatMessageFlags, GetJavaInstance_ChatMessageFlags(jEnv, msg.flags));
  jEnv->SetObjectField(jMessageInstance, chatMessageInfo.fields["flags"], jChatMessageFlags);

  jEnv->SetIntField(jMessageInstance, chatMessageInfo.fields["nameColorARGB"], static_cast<jint>(msg.nameColorARGB));
  jEnv->SetIntField(jMessageInstance, chatMessageInfo.fields["timestamp"], static_cast<jint>(msg.timestamp));
  jEnv->SetIntField(jMessageInstance, chatMessageInfo.fields["userId"], static_cast<jint>(msg.userId));
  jEnv->SetIntField(jMessageInstance, chatMessageInfo.fields["numBitsSent"], static_cast<jint>(msg.numBitsSent));

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jMessageType, GetJavaInstance_String(jEnv, msg.messageType));
  jEnv->SetObjectField(jMessageInstance, chatMessageInfo.fields["messageType"], jMessageType);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jMessageTags, GetJavaInstance_StringHashMap(jEnv, msg.messageTags));
  jEnv->SetObjectField(jMessageInstance, chatMessageInfo.fields["messageTags"], jMessageTags);

  // Create the token array
  AUTO_DELETE_LOCAL_REF(jEnv, jobjectArray, jTokens,
    jEnv->NewObjectArray(static_cast<jsize>(msg.tokens.size()), chatTokenInfo.klass, nullptr));

  jsize index = 0;
  for (const auto& token : msg.tokens) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jToken, GetJavaInstance_ChatMessageToken(jEnv, *token));
    jEnv->SetObjectArrayElement(jTokens, index, jToken);
    index++;
  }

  jEnv->SetObjectField(jMessageInstance, chatMessageInfo.fields["tokens"], jTokens);

  // Create the badge array
  AUTO_DELETE_LOCAL_REF(jEnv, jobjectArray, jBadges,
    jEnv->NewObjectArray(static_cast<jsize>(msg.badges.size()), chatBadgeInfo.klass, nullptr));

  index = 0;
  for (const auto& badge : msg.badges) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jBadge, GetJavaInstance_ChatMessageBadge(jEnv, badge));
    jEnv->SetObjectArrayElement(jBadges, index, jBadge);
    index++;
  }

  jEnv->SetObjectField(jMessageInstance, chatMessageInfo.fields["badges"], jBadges);

  return jMessageInstance;
}

jobject ttv::binding::java::GetJavaInstance_ChatLiveMessage(JNIEnv* jEnv, const LiveChatMessage& msg) {
  JavaClassInfo& liveChatMessage = GetJavaClassInfo_ChatLiveMessage(jEnv);

  jobject jMessageInstance = jEnv->NewObject(liveChatMessage.klass, liveChatMessage.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jMessageId, GetJavaInstance_String(jEnv, msg.messageId));
  jEnv->SetObjectField(jMessageInstance, liveChatMessage.fields["messageId"], jMessageId);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jMessageInfo, GetJavaInstance_ChatMessageInfo(jEnv, msg.messageInfo));
  jEnv->SetObjectField(jMessageInstance, liveChatMessage.fields["messageInfo"], jMessageInfo);

  return jMessageInstance;
}

jobject ttv::binding::java::GetJavaInstance_ChatLiveMessageArray(
  JNIEnv* jEnv, const std::vector<LiveChatMessage>& messageList) {
  JavaClassInfo& liveChatMessage = GetJavaClassInfo_ChatLiveMessage(jEnv);

  // Create the message array
  jobjectArray jMessageArray =
    jEnv->NewObjectArray(static_cast<jsize>(messageList.size()), liveChatMessage.klass, nullptr);

  jsize index = 0;
  for (const auto& message : messageList) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jMessageInstance, GetJavaInstance_ChatLiveMessage(jEnv, message));

    // Add the message to the message list
    jEnv->SetObjectArrayElement(jMessageArray, index, jMessageInstance);
    index++;
  }

  return jMessageArray;
}

jobject ttv::binding::java::GetJavaInstance_ChatColferLiveMessageArray(
  JNIEnv* jEnv, const std::vector<LiveChatMessage>& messageList) {
  const size_t numMessages = messageList.size();
  std::vector<colfer_chat_message> outputMessages;
  outputMessages.resize(numMessages);

  std::list<std::vector<colfer_message_token>> outputTokensList;
  std::list<std::vector<colfer_message_badge>> outputBadgesList;
  std::list<std::vector<colfer_message_tag>> outputMessageTagsList;

  for (size_t i = 0; i < numMessages; i++) {
    const LiveChatMessage& inputMessage = messageList[i];
    const MessageInfo& inputMessageInfo = inputMessage.messageInfo;
    colfer_chat_message& outputMessage = outputMessages[i];

    // LiveChatMessage
    // std::string messageId
    outputMessage.message_id.utf8 = inputMessage.messageId.c_str();
    outputMessage.message_id.len = inputMessage.messageId.size();

    // MessageInfo
    // std::string userName
    outputMessage.user_name.utf8 = inputMessageInfo.userName.c_str();
    outputMessage.user_name.len = inputMessageInfo.userName.size();

    // std::string displayName
    outputMessage.display_name.utf8 = inputMessageInfo.displayName.c_str();
    outputMessage.display_name.len = inputMessageInfo.displayName.size();

    // std::string messageType
    outputMessage.message_type.utf8 = inputMessageInfo.messageType.c_str();
    outputMessage.message_type.len = inputMessageInfo.messageType.size();

    // UserMode userMode
    outputMessage.usermode_moderator = inputMessageInfo.userMode.moderator;
    outputMessage.usermode_broadcaster = inputMessageInfo.userMode.broadcaster;
    outputMessage.usermode_administrator = inputMessageInfo.userMode.administrator;
    outputMessage.usermode_staff = inputMessageInfo.userMode.staff;
    outputMessage.usermode_system = inputMessageInfo.userMode.system;
    outputMessage.usermode_global_moderator = inputMessageInfo.userMode.globalModerator;
    outputMessage.usermode_banned = inputMessageInfo.userMode.banned;
    outputMessage.usermode_subscriber = inputMessageInfo.userMode.subscriber;
    outputMessage.usermode_vip = inputMessageInfo.userMode.vip;

    // Flags flags
    outputMessage.messageflags_action = inputMessageInfo.flags.action;
    outputMessage.messageflags_notice = inputMessageInfo.flags.notice;
    outputMessage.messageflags_ignored = inputMessageInfo.flags.ignored;
    outputMessage.messageflags_deleted = inputMessageInfo.flags.deleted;
    outputMessage.messageflags_contains_bits = inputMessageInfo.flags.containsBits;

    // Color (uint32_t) nameColorARGB
    outputMessage.name_color_argb = inputMessageInfo.nameColorARGB;

    // Timestamp (uint32_t) timestamp
    outputMessage.timestamp = inputMessageInfo.timestamp;

    // UserId (uint32_t) userId
    outputMessage.user_id = inputMessageInfo.userId;

    // uint32_t numBitsSent
    outputMessage.num_bits_sent = inputMessageInfo.numBitsSent;

    // std::vector<std::unique_ptr<MessageToken>> tokens
    const size_t numTokens = inputMessageInfo.tokens.size();
    std::vector<colfer_message_token> outputTokens;
    outputTokens.resize(numTokens);

    for (size_t i = 0; i < numTokens; i++) {
      const auto& inputToken = inputMessageInfo.tokens[i];
      colfer_message_token& outputToken = outputTokens[i];

      // 0 = text
      // 1 = emoticon
      // 2 = mention
      // 3 = url
      // 4 = bits

      switch (inputToken->GetType()) {
        case MessageToken::Type::Text: {
          const auto& textToken = static_cast<const TextToken&>(*inputToken);

          outputToken.token_type = 0;
          outputToken.text_field_1.utf8 = textToken.text.c_str();
          outputToken.text_field_1.len = textToken.text.size();
          outputToken.text_field_2.len = 0;
          outputToken.aggressive_level = textToken.autoModFlags.aggressiveLevel;
          outputToken.identity_level = textToken.autoModFlags.identityLevel;
          outputToken.profanity_level = textToken.autoModFlags.profanityLevel;
          outputToken.sexual_level = textToken.autoModFlags.sexualLevel;
          break;
        }
        case MessageToken::Type::Emoticon: {
          const auto& emoticonToken = static_cast<const EmoticonToken&>(*inputToken);

          outputToken.token_type = 1;
          outputToken.text_field_1.utf8 = emoticonToken.emoticonText.c_str();
          outputToken.text_field_1.len = emoticonToken.emoticonText.size();
          outputToken.text_field_2.utf8 = emoticonToken.emoticonId.c_str();
          outputToken.text_field_2.len = emoticonToken.emoticonId.size();
          break;
        }
        case MessageToken::Type::Mention: {
          const auto& mentionToken = static_cast<const MentionToken&>(*inputToken);

          outputToken.token_type = 2;
          outputToken.text_field_1.utf8 = mentionToken.userName.c_str();
          outputToken.text_field_1.len = mentionToken.userName.size();
          outputToken.text_field_2.utf8 = mentionToken.text.c_str();
          outputToken.text_field_2.len = mentionToken.text.size();
          outputToken.bool_field = mentionToken.isLocalUser;
          break;
        }
        case MessageToken::Type::Url: {
          const auto& urlToken = static_cast<const UrlToken&>(*inputToken);

          outputToken.token_type = 3;
          outputToken.text_field_1.utf8 = urlToken.url.c_str();
          outputToken.text_field_1.len = urlToken.url.size();
          outputToken.bool_field = urlToken.hidden;
          outputToken.text_field_2.len = 0;
          break;
        }
        case MessageToken::Type::Bits: {
          const auto& bitsToken = static_cast<const BitsToken&>(*inputToken);

          outputToken.token_type = 4;
          outputToken.text_field_1.utf8 = bitsToken.prefix.c_str();
          outputToken.text_field_1.len = bitsToken.prefix.size();
          outputToken.num_field = bitsToken.numBits;
          outputToken.text_field_2.len = 0;
          break;
        }
      }
    }

    outputMessage.tokens.list = outputTokens.data();
    outputMessage.tokens.len = numTokens;
    outputTokensList.emplace_back(std::move(outputTokens));

    // std::vector<MessageBadge> badges
    const size_t numBadges = inputMessageInfo.badges.size();
    std::vector<colfer_message_badge> outputBadges;
    outputBadges.resize(numBadges);

    for (size_t i = 0; i < numBadges; i++) {
      const auto& inputBadge = inputMessageInfo.badges[i];
      colfer_message_badge& outputBadge = outputBadges[i];

      outputBadge.name.utf8 = inputBadge.name.c_str();
      outputBadge.name.len = inputBadge.name.size();
      outputBadge.version.utf8 = inputBadge.version.c_str();
      outputBadge.version.len = inputBadge.version.size();
    }

    outputMessage.badges.list = outputBadges.data();
    outputMessage.badges.len = numBadges;
    outputBadgesList.emplace_back(std::move(outputBadges));

    // std::vector<MessageTags> MessageTags
    const size_t numMessageTags = inputMessageInfo.messageTags.size();
    std::vector<colfer_message_tag> outputMessageTags;
    outputMessageTags.resize(inputMessageInfo.messageTags.size());

    size_t pos = 0;
    for (const auto& tag : inputMessageInfo.messageTags) {
      auto& outputMessageTag = outputMessageTags[pos++];
      outputMessageTag.key.utf8 = tag.first.c_str();
      outputMessageTag.key.len = tag.first.size();
      outputMessageTag.value.utf8 = tag.second.c_str();
      outputMessageTag.value.len = tag.second.size();
    }

    outputMessage.message_tags.list = outputMessageTags.data();
    outputMessage.message_tags.len = numMessageTags;
    outputMessageTagsList.emplace_back(std::move(outputMessageTags));
  }

  colfer_chat_message_array messageArray;
  messageArray.messages.list = outputMessages.data();
  messageArray.messages.len = numMessages;

  std::vector<uint8_t> buffer;
  buffer.reserve(1024 * 1024);

  size_t bufferSize = colfer_chat_message_array_marshal(&messageArray, buffer.data());

  AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jbyteArray, jSerializedMessage,
    gActiveJavaEnvironment->NewByteArray(static_cast<jsize>(bufferSize)));
  gActiveJavaEnvironment->SetByteArrayRegion(
    jSerializedMessage, 0, static_cast<jsize>(bufferSize), reinterpret_cast<const jbyte*>(buffer.data()));

  JavaClassInfo& chatLiveMessageInfo = GetJavaClassInfo_ChatLiveMessage(gActiveJavaEnvironment);
  static const auto methodInfo = chatLiveMessageInfo.staticMethods["deserializeColferMessage"];

  return gActiveJavaEnvironment->CallStaticObjectMethod(chatLiveMessageInfo.klass, methodInfo, jSerializedMessage);
}

jobject ttv::binding::java::GetJavaInstance_ChatWhisperMessage(JNIEnv* jEnv, const WhisperMessage& msg) {
  JavaClassInfo& chatWhisperMessage = GetJavaClassInfo_ChatWhisperMessage(jEnv);

  jobject jMessageInstance = jEnv->NewObject(chatWhisperMessage.klass, chatWhisperMessage.methods["<init>"]);

  jEnv->SetIntField(jMessageInstance, chatWhisperMessage.fields["messageId"], static_cast<jint>(msg.messageId));

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jThreadId, GetJavaInstance_String(jEnv, msg.threadId));
  jEnv->SetObjectField(jMessageInstance, chatWhisperMessage.fields["threadId"], jThreadId);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jMessageInfo, GetJavaInstance_ChatMessageInfo(jEnv, msg.messageInfo));
  jEnv->SetObjectField(jMessageInstance, chatWhisperMessage.fields["messageInfo"], jMessageInfo);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jMessageUuid, GetJavaInstance_String(jEnv, msg.messageUuid));
  jEnv->SetObjectField(jMessageInstance, chatWhisperMessage.fields["messageUuid"], jMessageUuid);

  return jMessageInstance;
}

jobject ttv::binding::java::GetJavaInstance_ChatWhisperMessageArray(
  JNIEnv* jEnv, const std::vector<WhisperMessage>& messageList) {
  JavaClassInfo& chatWhisperMessage = GetJavaClassInfo_ChatWhisperMessage(jEnv);

  // Create the message array
  jobjectArray jMessageArray =
    jEnv->NewObjectArray(static_cast<jsize>(messageList.size()), chatWhisperMessage.klass, nullptr);

  jsize index = 0;
  for (const auto& message : messageList) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jMessageInstance, GetJavaInstance_ChatWhisperMessage(jEnv, message));

    // Add the message to the message list
    jEnv->SetObjectArrayElement(jMessageArray, index, jMessageInstance);
    index++;
  }

  return jMessageArray;
}

jobject ttv::binding::java::GetJavaInstance_ChatEmoticon(JNIEnv* jEnv, const Emoticon& emoticon) {
  JavaClassInfo& chatEmoticonInfo = GetJavaClassInfo_ChatEmoticon(jEnv);

  jobject jEmoticon = jEnv->NewObject(chatEmoticonInfo.klass, chatEmoticonInfo.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jEmoticonId, GetJavaInstance_String(jEnv, emoticon.emoticonId));
  jEnv->SetObjectField(jEmoticon, chatEmoticonInfo.fields["emoticonId"], jEmoticonId);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jMatch, GetJavaInstance_String(jEnv, emoticon.match));
  jEnv->SetObjectField(jEmoticon, chatEmoticonInfo.fields["match"], jMatch);
  jEnv->SetBooleanField(jEmoticon, chatEmoticonInfo.fields["isRegex"], emoticon.isRegex);

  return jEmoticon;
}

jobject ttv::binding::java::GetJavaInstance_ChatEmoticonSet(JNIEnv* jEnv, const EmoticonSet& emoticonSet) {
  JavaClassInfo& chatEmoticonInfo = GetJavaClassInfo_ChatEmoticon(jEnv);
  JavaClassInfo& chatEmoticonSetInfo = GetJavaClassInfo_ChatEmoticonSet(jEnv);

  jobject jEmoticonSet = jEnv->NewObject(chatEmoticonSetInfo.klass, chatEmoticonSetInfo.methods["<init>"]);
  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jEmoticonSetId, GetJavaInstance_String(jEnv, emoticonSet.emoticonSetId));
  jEnv->SetObjectField(jEmoticonSet, chatEmoticonSetInfo.fields["emoticonSetId"], jEmoticonSetId);

  // Create the emoticon arrays
  jobjectArray jEmoticons = GetJavaInstance_Array(jEnv, chatEmoticonInfo,
    static_cast<jsize>(emoticonSet.emoticons.size()),
    [jEnv, &emoticonSet](uint32_t index) { return GetJavaInstance_ChatEmoticon(jEnv, emoticonSet.emoticons[index]); });
  jEnv->SetObjectField(jEmoticonSet, chatEmoticonSetInfo.fields["emoticons"], jEmoticons);
  AUTO_DELETE_LOCAL_REF_NO_DECLARE(jEnv, jobjectArray, jEmoticons);

  return jEmoticonSet;
}

jobject ttv::binding::java::GetJavaInstance_ChatEmoticonSets(
  JNIEnv* jEnv, const std::vector<EmoticonSet>& emoticonSets) {
  JavaClassInfo& emoticonSet = GetJavaClassInfo_ChatEmoticonSet(jEnv);
  jobjectArray jArray = jEnv->NewObjectArray(static_cast<jsize>(emoticonSets.size()), emoticonSet.klass, nullptr);

  jsize index = 0;
  for (const auto& emoticonSet : emoticonSets) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jEmoticonSet, GetJavaInstance_ChatEmoticonSet(jEnv, emoticonSet));

    jEnv->SetObjectArrayElement(jArray, index, jEmoticonSet);
    index++;
  }

  return jArray;
}

jobject ttv::binding::java::GetJavaInstance_ChatRaidStatus(JNIEnv* jEnv, const RaidStatus& val) {
  JavaClassInfo& info = GetJavaClassInfo_ChatRaidStatus(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jRaidId, GetJavaInstance_String(jEnv, val.raidId));
  jEnv->SetObjectField(jInstance, info.fields["raidId"], jRaidId);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jTargetUserLogin, GetJavaInstance_String(jEnv, val.targetUserLogin));
  jEnv->SetObjectField(jInstance, info.fields["targetUserLogin"], jTargetUserLogin);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jTargetUserDisplayName, GetJavaInstance_String(jEnv, val.targetUserDisplayName));
  jEnv->SetObjectField(jInstance, info.fields["targetUserDisplayName"], jTargetUserDisplayName);

  AUTO_DELETE_LOCAL_REF(
    jEnv, jstring, jTargetUserProfileImageUrl, GetJavaInstance_String(jEnv, val.targetUserProfileImageUrl));
  jEnv->SetObjectField(jInstance, info.fields["targetUserProfileImageUrl"], jTargetUserProfileImageUrl);

  jEnv->SetIntField(jInstance, info.fields["creatorUserId"], static_cast<jint>(val.creatorUserId));
  jEnv->SetIntField(jInstance, info.fields["sourceChannelId"], static_cast<jint>(val.sourceChannelId));
  jEnv->SetIntField(jInstance, info.fields["targetChannelId"], static_cast<jint>(val.targetChannelId));
  jEnv->SetIntField(jInstance, info.fields["transitionJitterSeconds"], static_cast<jint>(val.transitionJitterSeconds));
  jEnv->SetIntField(jInstance, info.fields["numUsersInRaid"], static_cast<jint>(val.numUsersInRaid));
  jEnv->SetIntField(jInstance, info.fields["forceRaidNowSeconds"], static_cast<jint>(val.forceRaidNowSeconds));
  jEnv->SetBooleanField(jInstance, info.fields["joined"], static_cast<jboolean>(val.joined));

  return jInstance;
};

jobject ttv::binding::java::GetJavaInstance_ChatBadgeEntitlement(
  JNIEnv* jEnv, const BadgeEntitlement& badgeEntitlement) {
  JavaClassInfo& info = GetJavaClassInfo_ChatBadgeEntitlement(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  jEnv->SetIntField(jInstance, info.fields["newLevel"], static_cast<jint>(badgeEntitlement.newLevel));
  jEnv->SetIntField(jInstance, info.fields["previousLevel"], static_cast<jint>(badgeEntitlement.previousLevel));
  jEnv->SetBooleanField(
    jInstance, info.fields["isNewBadgeLevel"], static_cast<jboolean>(badgeEntitlement.isNewBadgeLevel));

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_ChatBitsReceivedEvent(
  JNIEnv* jEnv, const BitsReceivedEvent& bitsReceivedEvent) {
  JavaClassInfo& info = GetJavaClassInfo_ChatBitsReceivedEvent(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jMessageInfo, GetJavaInstance_ChatMessageInfo(jEnv, bitsReceivedEvent.message));
  jEnv->SetObjectField(jInstance, info.fields["message"], jMessageInfo);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jChannelName, GetJavaInstance_String(jEnv, bitsReceivedEvent.channelName));
  jEnv->SetObjectField(jInstance, info.fields["channelName"], jChannelName);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jContext, GetJavaInstance_String(jEnv, bitsReceivedEvent.context));
  jEnv->SetObjectField(jInstance, info.fields["context"], jContext);

  jEnv->SetIntField(jInstance, info.fields["channelId"], bitsReceivedEvent.channelId);
  jEnv->SetIntField(jInstance, info.fields["bitsUsed"], bitsReceivedEvent.bitsUsed);
  jEnv->SetIntField(jInstance, info.fields["totalBitsUsed"], bitsReceivedEvent.totalBitsUsed);

  AUTO_DELETE_LOCAL_REF(
    jEnv, jobject, jBadgeEntitlement, GetJavaInstance_ChatBadgeEntitlement(jEnv, bitsReceivedEvent.badge));
  jEnv->SetObjectField(jInstance, info.fields["badge"], jBadgeEntitlement);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_ChatBitsSentEvent(JNIEnv* jEnv, const BitsSentEvent& bitsSentEvent) {
  JavaClassInfo& info = GetJavaClassInfo_ChatBitsSentEvent(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  jEnv->SetIntField(jInstance, info.fields["channelId"], bitsSentEvent.channelId);
  jEnv->SetIntField(jInstance, info.fields["userBitsBalance"], bitsSentEvent.userBitsBalance);
  jEnv->SetIntField(jInstance, info.fields["channelBitsTotal"], bitsSentEvent.channelBitsTotal);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_ChatFollowerAddedEvent(
  JNIEnv* jEnv, const FollowerAddedEvent& followerAddedEvent) {
  JavaClassInfo& info = GetJavaClassInfo_ChatFollowerAddedEvent(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jDisplayName, GetJavaInstance_String(jEnv, followerAddedEvent.displayName));
  jEnv->SetObjectField(jInstance, info.fields["displayName"], jDisplayName);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jUserName, GetJavaInstance_String(jEnv, followerAddedEvent.userName));
  jEnv->SetObjectField(jInstance, info.fields["userName"], jUserName);

  jEnv->SetIntField(jInstance, info.fields["userId"], followerAddedEvent.userId);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_ChatSubscriberAddedEvent(
  JNIEnv* jEnv, const SubscriberAddedEvent& subscriberAddedEvent) {
  JavaClassInfo& info = GetJavaClassInfo_ChatSubscriberAddedEvent(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(
    jEnv, jobject, jSubNotice, GetJavaInstance_ChatSubscriptionNotice(jEnv, subscriberAddedEvent.subNotice));
  jEnv->SetObjectField(jInstance, info.fields["subNotice"], jSubNotice);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jDisplayName, GetJavaInstance_String(jEnv, subscriberAddedEvent.displayName));
  jEnv->SetObjectField(jInstance, info.fields["displayName"], jDisplayName);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jUserName, GetJavaInstance_String(jEnv, subscriberAddedEvent.userName));
  jEnv->SetObjectField(jInstance, info.fields["userName"], jUserName);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jChannelName, GetJavaInstance_String(jEnv, subscriberAddedEvent.channelName));
  jEnv->SetObjectField(jInstance, info.fields["channelName"], jChannelName);

  jEnv->SetIntField(jInstance, info.fields["userId"], subscriberAddedEvent.userId);
  jEnv->SetIntField(jInstance, info.fields["channelId"], subscriberAddedEvent.channelId);
  jEnv->SetIntField(jInstance, info.fields["timestamp"], static_cast<jint>(subscriberAddedEvent.timestamp));

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_ChatRoomInfo(JNIEnv* jEnv, const ChatRoomInfo& value) {
  JavaClassInfo& info = GetJavaClassInfo_ChatRoomInfo(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jUserInfo, GetJavaInstance_UserInfo(jEnv, value.owner));
  jEnv->SetObjectField(jInstance, info.fields["owner"], jUserInfo);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jView, GetJavaInstance_ChatRoomView(jEnv, value.view));
  jEnv->SetObjectField(jInstance, info.fields["view"], jView);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jId, GetJavaInstance_String(jEnv, value.id));
  jEnv->SetObjectField(jInstance, info.fields["id"], jId);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jName, GetJavaInstance_String(jEnv, value.name));
  jEnv->SetObjectField(jInstance, info.fields["name"], jName);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jTopic, GetJavaInstance_String(jEnv, value.topic));
  jEnv->SetObjectField(jInstance, info.fields["topic"], jTopic);

  AUTO_DELETE_LOCAL_REF(
    jEnv, jstring, jPermissions, GetJavaInstance_ChatRoomRolePermissions(jEnv, value.rolePermissions));
  jEnv->SetObjectField(jInstance, info.fields["rolePermissions"], jPermissions);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_ChatRoomMessage(JNIEnv* jEnv, const ChatRoomMessage& value) {
  JavaClassInfo& info = GetJavaClassInfo_ChatRoomMessage(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jRoomId, GetJavaInstance_String(jEnv, value.roomId));
  jEnv->SetObjectField(jInstance, info.fields["roomId"], jRoomId);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jRoomMessageId, GetJavaInstance_String(jEnv, value.roomMessageId));
  jEnv->SetObjectField(jInstance, info.fields["roomMessageId"], jRoomMessageId);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jMessageInfo, GetJavaInstance_ChatMessageInfo(jEnv, value.messageInfo));
  jEnv->SetObjectField(jInstance, info.fields["messageInfo"], jMessageInfo);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_ChatRoomView(JNIEnv* jEnv, const ChatRoomView& value) {
  JavaClassInfo& info = GetJavaClassInfo_ChatRoomView(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jPermissions, GetJavaInstance_ChatRoomPermissions(jEnv, value.permissions));
  jEnv->SetObjectField(jInstance, info.fields["permissions"], jPermissions);

  jEnv->SetIntField(jInstance, info.fields["lastReadAt"], value.lastReadAt);
  jEnv->SetIntField(jInstance, info.fields["unreadMentionCount"], value.unreadMentionCount);
  jEnv->SetBooleanField(jInstance, info.fields["isMuted"], value.isMuted ? JNI_TRUE : JNI_FALSE);
  jEnv->SetBooleanField(jInstance, info.fields["isArchived"], value.isArchived ? JNI_TRUE : JNI_FALSE);
  jEnv->SetBooleanField(jInstance, info.fields["isUnread"], value.isUnread ? JNI_TRUE : JNI_FALSE);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_RoomMentionInfo(JNIEnv* jEnv, const RoomMentionInfo& value) {
  JavaClassInfo& info = GetJavaClassInfo_RoomMentionInfo(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jRoomOwnerName, GetJavaInstance_String(jEnv, value.roomOwnerName));
  jEnv->SetObjectField(jInstance, info.fields["roomOwnerName"], jRoomOwnerName);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jRoomOwnerLogin, GetJavaInstance_String(jEnv, value.roomOwnerLogin));
  jEnv->SetObjectField(jInstance, info.fields["roomOwnerLogin"], jRoomOwnerLogin);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jSenderName, GetJavaInstance_String(jEnv, value.senderName));
  jEnv->SetObjectField(jInstance, info.fields["senderName"], jSenderName);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jRoomId, GetJavaInstance_String(jEnv, value.roomId));
  jEnv->SetObjectField(jInstance, info.fields["roomId"], jRoomId);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jRoomName, GetJavaInstance_String(jEnv, value.roomName));
  jEnv->SetObjectField(jInstance, info.fields["roomName"], jRoomName);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jMessageId, GetJavaInstance_String(jEnv, value.messageId));
  jEnv->SetObjectField(jInstance, info.fields["messageId"], jMessageId);

  jEnv->SetIntField(jInstance, info.fields["roomOwnerId"], static_cast<jint>(value.roomOwnerId));
  jEnv->SetIntField(jInstance, info.fields["senderId"], static_cast<jint>(value.senderId));
  jEnv->SetIntField(jInstance, info.fields["sentAt"], static_cast<jint>(value.sentAt));

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_ChatRoomRolePermissions(
  JNIEnv* jEnv, const RoomRolePermissions& permissions) {
  JavaClassInfo& info = GetJavaClassInfo_RoomRolePermissions(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(
    jEnv, jobject, jRead, GetJavaInstance_SimpleEnum(jEnv, GetJavaClassInfo_RoomRole(jEnv), permissions.read));
  jEnv->SetObjectField(jInstance, info.fields["read"], jRead);

  AUTO_DELETE_LOCAL_REF(
    jEnv, jobject, jSend, GetJavaInstance_SimpleEnum(jEnv, GetJavaClassInfo_RoomRole(jEnv), permissions.send));
  jEnv->SetObjectField(jInstance, info.fields["send"], jSend);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_ChatRoomPermissions(JNIEnv* jEnv, const ChatRoomPermissions& value) {
  JavaClassInfo& info = GetJavaClassInfo_ChatRoomPermissions(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  jEnv->SetBooleanField(jInstance, info.fields["readMessages"], value.readMessages ? JNI_TRUE : JNI_FALSE);
  jEnv->SetBooleanField(jInstance, info.fields["sendMessages"], value.sendMessages ? JNI_TRUE : JNI_FALSE);
  jEnv->SetBooleanField(jInstance, info.fields["moderate"], value.moderate ? JNI_TRUE : JNI_FALSE);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_ChatRoomMessageHandlerCommandError(
  JNIEnv* jEnv, ChatRoomMessageHandler::CommandError value) {
  JavaClassInfo& info = GetJavaClassInfo_ChatRoomMessageHandler_CommandError(jEnv);
  return jEnv->CallStaticObjectMethod(info.klass, info.staticMethods["lookupValue"], static_cast<jint>(value));
}

jobject ttv::binding::java::GetJavaInstance_SendRoomMessageError(JNIEnv* jEnv, const SendRoomMessageError& value) {
  JavaClassInfo& info = GetJavaClassInfo_SendRoomMessageError(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jErrorCode,
    GetJavaInstance_SimpleEnum(jEnv, GetJavaClassInfo_ChatGraphQLErrorCode(jEnv), value.code));
  jEnv->SetObjectField(jInstance, info.fields["code"], jErrorCode);

  jEnv->SetIntField(
    jInstance, info.fields["slowModeDurationSeconds"], static_cast<jint>(value.slowModeDurationSeconds));
  jEnv->SetIntField(
    jInstance, info.fields["remainingDurationSeconds"], static_cast<jint>(value.remainingDurationSeconds));

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_CreateRoomError(JNIEnv* jEnv, const CreateRoomError& value) {
  JavaClassInfo& info = GetJavaClassInfo_CreateRoomError(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jErrorCode,
    GetJavaInstance_SimpleEnum(jEnv, GetJavaClassInfo_ChatGraphQLErrorCode(jEnv), value.code));
  jEnv->SetObjectField(jInstance, info.fields["code"], jErrorCode);

  jEnv->SetIntField(jInstance, info.fields["maxAllowedRooms"], static_cast<jint>(value.maxAllowedRooms));
  jEnv->SetIntField(jInstance, info.fields["minLength"], static_cast<jint>(value.minLength));
  jEnv->SetIntField(jInstance, info.fields["maxLength"], static_cast<jint>(value.maxLength));

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_UpdateRoomError(JNIEnv* jEnv, const UpdateRoomError& value) {
  JavaClassInfo& info = GetJavaClassInfo_UpdateRoomError(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jErrorCode,
    GetJavaInstance_SimpleEnum(jEnv, GetJavaClassInfo_ChatGraphQLErrorCode(jEnv), value.code));
  jEnv->SetObjectField(jInstance, info.fields["code"], jErrorCode);

  jEnv->SetIntField(jInstance, info.fields["minLength"], static_cast<jint>(value.minLength));
  jEnv->SetIntField(jInstance, info.fields["maxLength"], static_cast<jint>(value.maxLength));

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_UpdateRoomModesError(JNIEnv* jEnv, const UpdateRoomModesError& value) {
  JavaClassInfo& info = GetJavaClassInfo_UpdateRoomModesError(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jErrorCode,
    GetJavaInstance_SimpleEnum(jEnv, GetJavaClassInfo_ChatGraphQLErrorCode(jEnv), value.code));
  jEnv->SetObjectField(jInstance, info.fields["code"], jErrorCode);

  jEnv->SetIntField(
    jInstance, info.fields["minimumSlowModeDurationSeconds"], static_cast<jint>(value.minimumSlowModeDurationSeconds));
  jEnv->SetIntField(
    jInstance, info.fields["maximumSlowModeDurationSeconds"], static_cast<jint>(value.maximumSlowModeDurationSeconds));

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_BanUserError(JNIEnv* jEnv, const BanUserError& value) {
  JavaClassInfo& info = GetJavaClassInfo_BanUserError(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jErrorCode,
    GetJavaInstance_SimpleEnum(jEnv, GetJavaClassInfo_ChatGraphQLErrorCode(jEnv), value.code));
  jEnv->SetObjectField(jInstance, info.fields["code"], jErrorCode);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_UnbanUserError(JNIEnv* jEnv, const UnbanUserError& value) {
  JavaClassInfo& info = GetJavaClassInfo_UnbanUserError(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jErrorCode,
    GetJavaInstance_SimpleEnum(jEnv, GetJavaClassInfo_ChatGraphQLErrorCode(jEnv), value.code));
  jEnv->SetObjectField(jInstance, info.fields["code"], jErrorCode);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_ModUserError(JNIEnv* jEnv, const ModUserError& value) {
  JavaClassInfo& info = GetJavaClassInfo_ModUserError(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jErrorCode,
    GetJavaInstance_SimpleEnum(jEnv, GetJavaClassInfo_ChatGraphQLErrorCode(jEnv), value.code));
  jEnv->SetObjectField(jInstance, info.fields["code"], jErrorCode);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_UnmodUserError(JNIEnv* jEnv, const UnmodUserError& value) {
  JavaClassInfo& info = GetJavaClassInfo_UnmodUserError(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jErrorCode,
    GetJavaInstance_SimpleEnum(jEnv, GetJavaClassInfo_ChatGraphQLErrorCode(jEnv), value.code));
  jEnv->SetObjectField(jInstance, info.fields["code"], jErrorCode);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_ChatComment(JNIEnv* jEnv, const ChatComment& comment) {
  JavaClassInfo& info = GetJavaClassInfo_ChatComment(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jMessageInfo, GetJavaInstance_ChatMessageInfo(jEnv, comment.messageInfo));
  jEnv->SetObjectField(jInstance, info.fields["messageInfo"], jMessageInfo);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jReplies, GetJavaInstance_ChatCommentArray(jEnv, comment.replies));
  jEnv->SetObjectField(jInstance, info.fields["replies"], jReplies);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jCommentId, GetJavaInstance_String(jEnv, comment.commentId));
  jEnv->SetObjectField(jInstance, info.fields["commentId"], jCommentId);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jParentCommentId, GetJavaInstance_String(jEnv, comment.parentCommentId));
  jEnv->SetObjectField(jInstance, info.fields["parentCommentId"], jParentCommentId);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jContentId, GetJavaInstance_String(jEnv, comment.contentId));
  jEnv->SetObjectField(jInstance, info.fields["contentId"], jContentId);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jSource,
    GetJavaInstance_SimpleEnum(jEnv, GetJavaClassInfo_ChatCommentSource(jEnv), comment.commentSource));
  jEnv->SetObjectField(jInstance, info.fields["commentSource"], jSource);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jPublishedState,
    GetJavaInstance_SimpleEnum(jEnv, GetJavaClassInfo_ChatCommentPublishedState(jEnv), comment.publishedState));
  jEnv->SetObjectField(jInstance, info.fields["publishedState"], jPublishedState);

  jEnv->SetIntField(jInstance, info.fields["channelId"], static_cast<jint>(comment.channelId));
  jEnv->SetIntField(jInstance, info.fields["timestampMilliseconds"], static_cast<jint>(comment.timestampMilliseconds));
  jEnv->SetIntField(jInstance, info.fields["updatedAt"], static_cast<jint>(comment.updatedAt));
  jEnv->SetBooleanField(jInstance, info.fields["moreReplies"], comment.moreReplies ? JNI_TRUE : JNI_FALSE);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_ChatCommentArray(JNIEnv* jEnv, const std::vector<ChatComment>& comments) {
  JavaClassInfo& info = GetJavaClassInfo_ChatComment(jEnv);

  // Create the message array
  jobjectArray jCommentArray = jEnv->NewObjectArray(static_cast<jsize>(comments.size()), info.klass, nullptr);

  jsize index = 0;
  for (const auto& comment : comments) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jComment, GetJavaInstance_ChatComment(jEnv, comment));

    // Add the message to the message list
    jEnv->SetObjectArrayElement(jCommentArray, index, jComment);
    index++;
  }

  return jCommentArray;
}

jobject ttv::binding::java::GetJavaInstance_ChannelVodCommentSettings(
  JNIEnv* jEnv, const ChannelVodCommentSettings& settings) {
  JavaClassInfo& info = GetJavaClassInfo_ChannelVodCommentSettings(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jPublishingMode,
    GetJavaInstance_SimpleEnum(jEnv, GetJavaClassInfo_CommentPublishingMode(jEnv), settings.publishingMode));

  jEnv->SetIntField(jInstance, info.fields["channelId"], static_cast<jint>(settings.channelId));
  jEnv->SetIntField(jInstance, info.fields["createdAt"], static_cast<jint>(settings.createdAt));
  jEnv->SetIntField(jInstance, info.fields["updatedAt"], static_cast<jint>(settings.updatedAt));
  jEnv->SetIntField(
    jInstance, info.fields["followersOnlyDurationSeconds"], static_cast<jint>(settings.followersOnlyDurationSeconds));

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_ModerationActionInfo(
  JNIEnv* jEnv, const ModerationActionInfo& modActionInfo) {
  JavaClassInfo& info = GetJavaClassInfo_ModerationActionInfo(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jModeratorName, GetJavaInstance_String(jEnv, modActionInfo.moderatorName));
  jEnv->SetObjectField(jInstance, info.fields["moderatorName"], jModeratorName);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jTargetName, GetJavaInstance_String(jEnv, modActionInfo.targetName));
  jEnv->SetObjectField(jInstance, info.fields["targetName"], jTargetName);

  jEnv->SetIntField(jInstance, info.fields["moderatorId"], static_cast<jint>(modActionInfo.moderatorId));
  jEnv->SetIntField(jInstance, info.fields["targetId"], static_cast<jint>(modActionInfo.targetId));

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_ExtensionMessage(JNIEnv* jEnv, const ExtensionMessage& message) {
  JavaClassInfo& info = GetJavaClassInfo_ExtensionMessage(jEnv);
  JavaClassInfo& chatTokenInfo = GetJavaClassInfo_ChatMessageToken(jEnv);
  JavaClassInfo& chatBadgeInfo = GetJavaClassInfo_ChatMessageBadge(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jMessageId, GetJavaInstance_String(jEnv, message.messageId));
  jEnv->SetObjectField(jInstance, info.fields["messageId"], jMessageId);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jExtensionClientId, GetJavaInstance_String(jEnv, message.extensionClientId));
  jEnv->SetObjectField(jInstance, info.fields["extensionClientId"], jExtensionClientId);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jExtensionVersion, GetJavaInstance_String(jEnv, message.extensionVersion));
  jEnv->SetObjectField(jInstance, info.fields["extensionVersion"], jExtensionVersion);

  AUTO_DELETE_LOCAL_REF(
    jEnv, jstring, jExtensionDisplayName, GetJavaInstance_String(jEnv, message.extensionDisplayName));
  jEnv->SetObjectField(jInstance, info.fields["extensionDisplayName"], jExtensionDisplayName);

  jEnv->SetIntField(jInstance, info.fields["chatColor"], static_cast<jint>(message.chatColor));
  jEnv->SetIntField(jInstance, info.fields["sentAt"], static_cast<jint>(message.sentAt));

  // Create the badge array
  AUTO_DELETE_LOCAL_REF(jEnv, jobjectArray, jBadges,
    jEnv->NewObjectArray(static_cast<jsize>(message.badges.size()), chatBadgeInfo.klass, nullptr));

  jsize index = 0;
  for (const auto& badge : message.badges) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jBadge, GetJavaInstance_ChatMessageBadge(jEnv, badge));
    jEnv->SetObjectArrayElement(jBadges, index, jBadge);
    index++;
  }

  jEnv->SetObjectField(jInstance, info.fields["badges"], jBadges);

  // Create the token array
  AUTO_DELETE_LOCAL_REF(jEnv, jobjectArray, jTokens,
    jEnv->NewObjectArray(static_cast<jsize>(message.tokens.size()), chatTokenInfo.klass, nullptr));

  index = 0;
  for (const auto& token : message.tokens) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jToken, GetJavaInstance_ChatMessageToken(jEnv, *token));
    jEnv->SetObjectArrayElement(jTokens, index, jToken);
    index++;
  }

  jEnv->SetObjectField(jInstance, info.fields["tokens"], jTokens);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_AutoModFlags(JNIEnv* jEnv, const AutoModFlags& flags) {
  JavaClassInfo& info = GetJavaClassInfo_AutoModFlags(jEnv);
  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  jEnv->SetIntField(jInstance, info.fields["aggressiveLevel"], static_cast<jint>(flags.aggressiveLevel));
  jEnv->SetIntField(jInstance, info.fields["identityLevel"], static_cast<jint>(flags.identityLevel));
  jEnv->SetIntField(jInstance, info.fields["sexualLevel"], static_cast<jint>(flags.sexualLevel));
  jEnv->SetIntField(jInstance, info.fields["profanityLevel"], static_cast<jint>(flags.profanityLevel));

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_MultiviewContentAttribute(
  JNIEnv* jEnv, const MultiviewContentAttribute& attribute) {
  JavaClassInfo& info = GetJavaClassInfo_MultiviewContentAttribute(jEnv);
  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jAttributeId, GetJavaInstance_String(jEnv, attribute.attributeId));
  jEnv->SetObjectField(jInstance, info.fields["attributeId"], jAttributeId);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jKey, GetJavaInstance_String(jEnv, attribute.key));
  jEnv->SetObjectField(jInstance, info.fields["key"], jKey);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jName, GetJavaInstance_String(jEnv, attribute.name));
  jEnv->SetObjectField(jInstance, info.fields["name"], jName);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jParentId, GetJavaInstance_String(jEnv, attribute.parentId));
  jEnv->SetObjectField(jInstance, info.fields["parentId"], jParentId);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jParentKey, GetJavaInstance_String(jEnv, attribute.parentKey));
  jEnv->SetObjectField(jInstance, info.fields["parentKey"], jParentKey);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jValue, GetJavaInstance_String(jEnv, attribute.value));
  jEnv->SetObjectField(jInstance, info.fields["value"], jValue);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jImageUrl, GetJavaInstance_String(jEnv, attribute.imageUrl));
  jEnv->SetObjectField(jInstance, info.fields["imageUrl"], jImageUrl);

  AUTO_DELETE_LOCAL_REF(jEnv, jstring, jValueShortName, GetJavaInstance_String(jEnv, attribute.valueShortName));
  jEnv->SetObjectField(jInstance, info.fields["valueShortName"], jValueShortName);

  jEnv->SetIntField(jInstance, info.fields["ownerChannelId"], static_cast<jint>(attribute.ownerChannelId));
  jEnv->SetIntField(jInstance, info.fields["createdAt"], static_cast<jint>(attribute.createdAt));
  jEnv->SetIntField(jInstance, info.fields["updatedAt"], static_cast<jint>(attribute.updatedAt));

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_Chanlet(JNIEnv* jEnv, const Chanlet& chanlet) {
  JavaClassInfo& info = GetJavaClassInfo_Chanlet(jEnv);
  JavaClassInfo& attributeInfo = GetJavaClassInfo_MultiviewContentAttribute(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  jEnv->SetIntField(jInstance, info.fields["chanletId"], static_cast<jint>(chanlet.chanletId));

  // Create the squad member array
  AUTO_DELETE_LOCAL_REF(jEnv, jobjectArray, jAttributes,
    jEnv->NewObjectArray(static_cast<jsize>(chanlet.attributes.size()), attributeInfo.klass, nullptr));

  jsize index = 0;
  for (const auto& attribute : chanlet.attributes) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jAttribute, GetJavaInstance_MultiviewContentAttribute(jEnv, attribute));
    jEnv->SetObjectArrayElement(jAttributes, index, jAttribute);
    index++;
  }

  jEnv->SetObjectField(jInstance, info.fields["attributes"], jAttributes);

  return jInstance;
}

void ttv::binding::java::GetNativeInstance_ChatTokenizationOptions(
  JNIEnv* jEnv, jobject jOptions, TokenizationOptions& options) {
  JavaClassInfo& info = GetJavaClassInfo_ChatTokenizationOptions(jEnv);

  options.emoticons = jEnv->GetBooleanField(jOptions, info.fields["emoticons"]) == JNI_TRUE;
  options.mentions = jEnv->GetBooleanField(jOptions, info.fields["mentions"]) == JNI_TRUE;
  options.urls = jEnv->GetBooleanField(jOptions, info.fields["urls"]) == JNI_TRUE;
  options.bits = jEnv->GetBooleanField(jOptions, info.fields["bits"]) == JNI_TRUE;
}

void ttv::binding::java::GetNativeInstance_ChatFeatureFlags(JNIEnv* jEnv, jobject jFlags, FeatureFlags& flags) {
  JavaClassInfo& info = GetJavaClassInfo_ChatFeatureFlags(jEnv);

  flags.conversations = jEnv->GetBooleanField(jFlags, info.fields["conversations"]) == JNI_TRUE;
}

void ttv::binding::java::GetNativeInstance_RoomRolePermissions(
  JNIEnv* jEnv, jobject jPermissions, RoomRolePermissions& permissions) {
  JavaClassInfo& info = GetJavaClassInfo_RoomRolePermissions(jEnv);

  permissions.read = GetNativeFromJava_SimpleEnum<RoomRole>(
    jEnv, GetJavaClassInfo_RoomRole(jEnv), jEnv->GetObjectField(jPermissions, info.fields["read"]), RoomRole::Unknown);
  permissions.send = GetNativeFromJava_SimpleEnum<RoomRole>(
    jEnv, GetJavaClassInfo_RoomRole(jEnv), jEnv->GetObjectField(jPermissions, info.fields["send"]), RoomRole::Unknown);
}
