/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/generated/jni_channelchatroommanagerproxy.h"

#include "twitchsdk/chat/generated/java_all.h"
#include "twitchsdk/chat/internal/channelchatroommanager.h"
#include "twitchsdk/chat/java_chatutil.h"

using namespace ttv;
using namespace ttv::chat;
using namespace ttv::binding::java;

#define GET_NATIVE_PTR(x) reinterpret_cast<IChannelChatRoomManager*>(x)

JNIEXPORT void JNICALL Java_tv_twitch_chat_ChannelChatRoomManagerProxy_DisposeNativeInstance(
  JNIEnv* /*jEnv*/, jobject jThis, jlong /*jNativeObjectPointer*/) {
  auto instance = gIChannelChatRoomManagerInstanceRegistry.LookupNativeInstance(jThis);
  if (instance == nullptr) {
    return;
  }

  TTV_ErrorCode ec = instance->Dispose();

  if (TTV_SUCCEEDED(ec)) {
    gIChannelChatRoomManagerInstanceRegistry.Unregister(jThis);
  }
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChannelChatRoomManagerProxy_AddNewChatRoom(JNIEnv* jEnv,
  jobject /*jThis*/, jlong jNativeObjectPointer, jstring jRoomName, jstring jTopic, jobject jPermissions,
  jobject jCallback) {
  TTV_JNI_RETURN_ON_NULL(jEnv, jRoomName, TTV_EC_INVALID_ARG);
  TTV_JNI_RETURN_ON_NULL(jEnv, jTopic, TTV_EC_INVALID_ARG);
  TTV_JNI_RETURN_ON_NULL(jEnv, jPermissions, TTV_EC_INVALID_ARG);

  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject, jobject, jobject>(
    jEnv, jCallback, GetJavaClassInfo_IChannelChatRoomManager_AddRoomCallback(jEnv));

  ScopedJavaUTFStringConverter roomName(jEnv, jRoomName);
  ScopedJavaUTFStringConverter topic(jEnv, jTopic);

  RoomRolePermissions permissions;
  GetNativeInstance_RoomRolePermissions(jEnv, jPermissions, permissions);

  TTV_ErrorCode ec = instance->AddNewChatRoom(roomName, topic, permissions,
    [callback](TTV_ErrorCode callbackEc, CreateRoomError&& error, ChatRoomInfo&& roomInfo) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jGraphQLError, GetJavaInstance_CreateRoomError(gActiveJavaEnvironment, error));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jRoomInfo, GetJavaInstance_ChatRoomInfo(gActiveJavaEnvironment, roomInfo));

      callback(jError, jGraphQLError, jRoomInfo);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChannelChatRoomManagerProxy_JoinChatRooms(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback =
    CreateJavaCallbackWrapper<jobject>(jEnv, jCallback, GetJavaClassInfo_IChannelChatRoomManager_JoinCallback(jEnv));

  TTV_ErrorCode ec = instance->JoinChatRooms([callback](TTV_ErrorCode callbackEc) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));

    callback(jError);
  });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChannelChatRoomManagerProxy_LeaveChatRooms(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback =
    CreateJavaCallbackWrapper<jobject>(jEnv, jCallback, GetJavaClassInfo_IChannelChatRoomManager_LeaveCallback(jEnv));

  TTV_ErrorCode ec = instance->LeaveChatRooms([callback](TTV_ErrorCode callbackEc) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));

    callback(jError);
  });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChannelChatRoomManagerProxy_FetchChatRoomsInfo(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject, jobject>(
    jEnv, jCallback, GetJavaClassInfo_IChannelChatRoomManager_FetchRoomsInfoCallback(jEnv));

  TTV_ErrorCode ec =
    instance->FetchChatRoomsInfo([callback](TTV_ErrorCode callbackEc, std::vector<ChatRoomInfo>&& infos) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jRoomInfos,
        GetJavaInstance_Array(gActiveJavaEnvironment, GetJavaClassInfo_ChatRoomInfo(gActiveJavaEnvironment),
          static_cast<uint32_t>(infos.size()),
          [&infos](uint32_t index) { return GetJavaInstance_ChatRoomInfo(gActiveJavaEnvironment, infos[index]); }));

      callback(jError, jRoomInfos);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}
