/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/generated/jni_chatapi.h"

#include "twitchsdk/chat/chatapi.h"
#include "twitchsdk/chat/generated/java_all.h"
#include "twitchsdk/chat/ibitsstatus.h"
#include "twitchsdk/chat/ichannelchatroommanager.h"
#include "twitchsdk/chat/ichatchannelproperties.h"
#include "twitchsdk/chat/ichatraid.h"
#include "twitchsdk/chat/ichatroom.h"
#include "twitchsdk/chat/ichatroomnotifications.h"
#include "twitchsdk/chat/ifollowersstatus.h"
#include "twitchsdk/chat/ifollowingstatus.h"
#include "twitchsdk/chat/imultiviewnotifications.h"
#include "twitchsdk/chat/internal/chathelpers.h"
#include "twitchsdk/chat/internal/chaturlgenerator.h"
#include "twitchsdk/chat/isquadnotifications.h"
#include "twitchsdk/chat/isubscribersstatus.h"
#include "twitchsdk/chat/isubscriptionsnotifications.h"
#include "twitchsdk/chat/java_bitslistenerproxy.h"
#include "twitchsdk/chat/java_chatapilistenerproxy.h"
#include "twitchsdk/chat/java_chatchannellistenerproxy.h"
#include "twitchsdk/chat/java_chatraidlistenerproxy.h"
#include "twitchsdk/chat/java_chatuserthreadslistenerproxy.h"
#include "twitchsdk/chat/java_chatutil.h"
#include "twitchsdk/chat/java_followerslistenerproxy.h"
#include "twitchsdk/chat/java_followinglistenerproxy.h"
#include "twitchsdk/chat/java_ichannelchatroommanagerlistener_proxy.h"
#include "twitchsdk/chat/java_ichatchannelpropertylistenerproxy.h"
#include "twitchsdk/chat/java_ichatcommentlistener_proxy.h"
#include "twitchsdk/chat/java_ichatroomlistener_proxy.h"
#include "twitchsdk/chat/java_ichatroomnotificationslistener_proxy.h"
#include "twitchsdk/chat/java_imultiviewnotificationslistener_proxy.h"
#include "twitchsdk/chat/java_isquadnotificationslistener_proxy.h"
#include "twitchsdk/chat/java_isubscriptionsnotificationslistener_proxy.h"
#include "twitchsdk/chat/java_subscriberslistenerproxy.h"
#include "twitchsdk/core/assertion.h"
#include "twitchsdk/core/mutex.h"
#include "twitchsdk/core/stringutilities.h"

#include <functional>
#include <map>

#define GET_CHATAPI_PTR(x) reinterpret_cast<ChatAPI*>(x)

using namespace ttv;
using namespace ttv::chat;
using namespace ttv::binding::java;

JNIEXPORT jlong JNICALL Java_tv_twitch_chat_ChatAPI_CreateNativeInstance(JNIEnv* jEnv, jobject jThis) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  // Force load all types to ensure all classes are loaded by the time we need them.
  LoadAllChatJavaClassInfo(jEnv);

  std::shared_ptr<ChatApiContext> context = std::make_shared<ChatApiContext>();
  context->chatApi = std::make_shared<ChatAPI>();
  context->nativeListener = std::make_shared<JavaChatAPIListenerProxy>(jThis);

  gChatApiNativeProxyRegistry.Register(context->chatApi, context, jThis);

  context->chatApi->SetListener(context->nativeListener);

  return reinterpret_cast<jlong>(context->chatApi.get());
}

JNIEXPORT void JNICALL Java_tv_twitch_chat_ChatAPI_DisposeNativeInstance(
  JNIEnv* /*jEnv*/, jobject /*jThis*/, jlong jNativePointer) {
  gChatApiNativeProxyRegistry.Unregister(jNativePointer);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_SetCoreApi(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jCoreApi) {
  auto api = GET_CHATAPI_PTR(jNativePointer);
  auto coreApi = GetCoreApiInstance(jCoreApi);

  TTV_ErrorCode ec;

  if (coreApi != nullptr) {
    ec = api->SetCoreApi(coreApi);
  } else {
    ec = TTV_EC_INVALID_ARG;
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_SetTokenizationOptions(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jOptions) {
  auto api = GET_CHATAPI_PTR(jNativePointer);

  TokenizationOptions options;
  GetNativeInstance_ChatTokenizationOptions(jEnv, jOptions, options);

  TTV_ErrorCode ec = api->SetTokenizationOptions(options);

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_SetEnabledFeatures(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jFlags) {
  auto api = GET_CHATAPI_PTR(jNativePointer);

  FeatureFlags flags;
  GetNativeInstance_ChatFeatureFlags(jEnv, jFlags, flags);

  TTV_ErrorCode ec = api->SetEnabledFeatures(flags);

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_SetListener(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  TTV_JNI_RETURN_ON_NULL(jEnv, jListener, TTV_EC_INVALID_ARG);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  auto context = gChatApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    context->nativeListener->SetListener(jListener);
  } else {
    ec = TTV_EC_NOT_INITIALIZED;
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_GetState(JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  IModule::State state = api->GetState();

  JavaClassInfo& info = GetJavaClassInfo_ModuleState(jEnv);
  return GetJavaInstance_SimpleEnum(jEnv, info, state);
}

JNIEXPORT jstring JNICALL Java_tv_twitch_chat_ChatAPI_GetModuleName(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  std::string name = api->GetModuleName();

  return GetJavaInstance_String(jEnv, name.c_str());
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_Initialize(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  auto callback =
    CreateJavaCallbackWrapper<jobject>(jEnv, jCallback, GetJavaClassInfo_IModule_InitializeCallback(jEnv));

  TTV_ErrorCode ec = api->Initialize([callback](TTV_ErrorCode callbackEc) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
    callback(jError);
  });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_Shutdown(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  auto callback = CreateJavaCallbackWrapper<jobject>(jEnv, jCallback, GetJavaClassInfo_IModule_ShutdownCallback(jEnv));

  TTV_ErrorCode ec = api->Shutdown([callback](TTV_ErrorCode callbackEc) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
    callback(jError);
  });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_Connect(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jint jChannelId, jobject jIChatChannelListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jIChatChannelListener, TTV_EC_INVALID_ARG);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  std::shared_ptr<ChatApiContext> context = gChatApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    std::shared_ptr<JavaChatChannelListenerProxy> nativeChannelListener;

    auto& listeners = context->channelListeners;

    auto iter = listeners.find(static_cast<ChannelId>(jChannelId));
    if (iter != listeners.end()) {
      nativeChannelListener = iter->second;
    } else {
      nativeChannelListener = std::make_shared<JavaChatChannelListenerProxy>();
      nativeChannelListener->SetChatApi(context->chatApi);
      nativeChannelListener->SetChannelDisconnectedFunc([context](UserId /*userId*/, ChannelId channelId) {
        auto iter = context->channelListeners.find(static_cast<ChannelId>(channelId));
        if (iter != context->channelListeners.end()) {
          context->channelListeners.erase(iter);
        }
      });
    }

    ec = api->Connect(static_cast<UserId>(jUserId), static_cast<ChannelId>(jChannelId), nativeChannelListener);
    if (TTV_SUCCEEDED(ec)) {
      nativeChannelListener->SetListener(jIChatChannelListener);

      if (iter == listeners.end()) {
        listeners[static_cast<ChannelId>(jChannelId)] = nativeChannelListener;
      }
    }
  } else {
    ec = TTV_EC_NOT_INITIALIZED;
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_Disconnect(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jint jChannelId) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  // NOTE: The chatChannelListener will be cleared by the listener during a disconnection event

  TTV_ErrorCode ec = api->Disconnect(static_cast<UserId>(jUserId), static_cast<ChannelId>(jChannelId));
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_SendMessage(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jint jChannelId, jstring jMessage) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  ScopedJavaUTFStringConverter message(jEnv, jMessage);

  TTV_ErrorCode ec =
    api->SendChatMessage(static_cast<UserId>(jUserId), static_cast<ChannelId>(jChannelId), message.GetNativeString());
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_Update(JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  auto context = gChatApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    // Keep a reference to the local listeners until all callbacks are fired in case a callback wants to disconnect
    auto keepChannelListenersAlive = context->channelListeners;
    auto keepUserThreadsListenersAlive = context->userThreadsListeners;

    ec = api->Update();
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_BlockUser(JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer,
  jint jUserId, jint jBlockUserId, jstring jReason, jboolean jWhisper, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  ScopedJavaUTFStringConverter reason(jEnv, jReason);

  auto callbackWrapper =
    CreateJavaCallbackWrapper<jobject>(jEnv, jCallback, GetJavaClassInfo_ChatAPI_BlockChangeCallback(jEnv));
  TTV_ErrorCode ec = api->BlockUser(static_cast<UserId>(jUserId), static_cast<UserId>(jBlockUserId),
    reason.GetNativeString(), jWhisper == JNI_TRUE, [callbackWrapper](TTV_ErrorCode callbackEc) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      callbackWrapper(jErrorCode);
    });
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_UnblockUser(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jint jBlockUserId, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  auto callbackWrapper =
    CreateJavaCallbackWrapper<jobject>(jEnv, jCallback, GetJavaClassInfo_ChatAPI_BlockChangeCallback(jEnv));
  TTV_ErrorCode ec = api->UnblockUser(
    static_cast<UserId>(jUserId), static_cast<UserId>(jBlockUserId), [callbackWrapper](TTV_ErrorCode callbackEc) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      callbackWrapper(jErrorCode);
    });
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_GetUserBlocked(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jint jBlockUserId, jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  bool result = false;
  TTV_ErrorCode ec = api->GetUserBlocked(static_cast<UserId>(jUserId), static_cast<UserId>(jBlockUserId), result);
  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jBoolean, GetJavaInstance_Boolean(jEnv, result));
  SetResultContainerResult(jEnv, jResultContainer, jBoolean);

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_FetchBlockedUsers(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  auto callbackWrapper = CreateJavaCallbackWrapper<jobject, jobjectArray>(
    jEnv, jCallback, GetJavaClassInfo_ChatAPI_FetchBlockedUsersCallback(jEnv));
  TTV_ErrorCode ec = api->FetchBlockedUsers(
    static_cast<UserId>(jUserId), [callbackWrapper](TTV_ErrorCode ec, const std::vector<UserInfo>& blockedUsers) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, ec));
      AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobjectArray, jUsers,
        (TTV_SUCCEEDED(ec) ? GetJavaInstance_UserInfoArray(gActiveJavaEnvironment, blockedUsers) : nullptr));

      callbackWrapper(jErrorCode, jUsers);
    });
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_FetchUserEmoticonSets(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jboolean jForceFetch, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  auto callbackWrapper = CreateJavaCallbackWrapper<jobject, jobject>(
    jEnv, jCallback, GetJavaClassInfo_ChatAPI_FetchEmoticonSetsCallback(jEnv));

  TTV_ErrorCode ec = api->FetchUserEmoticonSets(static_cast<UserId>(jUserId), jForceFetch == JNI_TRUE,
    [callbackWrapper](TTV_ErrorCode ec, const std::vector<EmoticonSet>& emoticonSets) {
      auto& info = GetJavaClassInfo_ChatEmoticonSet(gActiveJavaEnvironment);
      AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jEmoticonSets,
        GetJavaInstance_Array(
          gActiveJavaEnvironment, info, static_cast<jsize>(emoticonSets.size()), [&emoticonSets](uint32_t index) {
            return GetJavaInstance_ChatEmoticonSet(gActiveJavaEnvironment, emoticonSets[index]);
          }));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, ec));

      callbackWrapper(jErrorCode, jEmoticonSets);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_FetchGlobalBadges(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  auto callbackWrapper =
    CreateJavaCallbackWrapper<jobject, jobject>(jEnv, jCallback, GetJavaClassInfo_ChatAPI_FetchBadgesCallback(jEnv));
  TTV_ErrorCode ec = api->FetchGlobalBadges([callbackWrapper](TTV_ErrorCode callbackEc, const BadgeSet& badgeSet) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jBadgeSet, GetJavaInstance_ChatBadgeSet(gActiveJavaEnvironment, badgeSet));
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));

    callbackWrapper(jErrorCode, jBadgeSet);
  });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_FetchChannelBadges(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jChannelId, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  auto callbackWrapper =
    CreateJavaCallbackWrapper<jobject, jobject>(jEnv, jCallback, GetJavaClassInfo_ChatAPI_FetchBadgesCallback(jEnv));
  TTV_ErrorCode ec = api->FetchChannelBadges(
    static_cast<ChannelId>(jChannelId), [callbackWrapper](TTV_ErrorCode callbackEc, const BadgeSet& badgeSet) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jBadgeSet, GetJavaInstance_ChatBadgeSet(gActiveJavaEnvironment, badgeSet));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));

      callbackWrapper(jErrorCode, jBadgeSet);
    });
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_SetUserThreadsListener(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  auto context = gChatApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    std::shared_ptr<JavaChatUserThreadsListenerProxy> nativeListener;

    auto& listeners = context->userThreadsListeners;
    auto iter = listeners.find(static_cast<UserId>(jUserId));

    if (iter != listeners.end()) {
      nativeListener = iter->second;
    }

    if (jListener != nullptr) {
      if (nativeListener == nullptr) {
        nativeListener = std::make_shared<JavaChatUserThreadsListenerProxy>();
      }

      ec = api->SetUserThreadsListener(static_cast<UserId>(jUserId), nativeListener);

      if (TTV_SUCCEEDED(ec)) {
        if (iter == listeners.end()) {
          listeners[static_cast<UserId>(jUserId)] = nativeListener;
        }

        nativeListener->SetListener(jListener);
      }
    } else {
      ec = api->SetUserThreadsListener(static_cast<UserId>(jUserId), nullptr);

      if (TTV_SUCCEEDED(ec) && iter != listeners.end()) {
        listeners.erase(iter);
      }
    }
  } else {
    ec = TTV_EC_NOT_INITIALIZED;
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jlong JNICALL Java_tv_twitch_chat_ChatAPI_GetMessageFlushInterval(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  return static_cast<jlong>(api->GetMessageFlushInterval());
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_SetMessageFlushInterval(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jlong jMilliseconds) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  if (jMilliseconds < 0) {
    return GetJavaInstance_ErrorCode(jEnv, TTV_EC_INVALID_ARG);
  }

  uint64_t ms = static_cast<uint64_t>(jMilliseconds);
  api->SetMessageFlushInterval(ms);

  return GetJavaInstance_ErrorCode(jEnv, TTV_EC_SUCCESS);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_GetEmoticonUrl(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jstring jEmoticonId, jfloat jScale, jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  std::string url;

  ScopedJavaUTFStringConverter emoticonId(jEnv, jEmoticonId);
  TTV_ErrorCode ec = api->GetEmoticonUrl(emoticonId.GetNativeString(), static_cast<float>(jScale), url);

  if (TTV_SUCCEEDED(ec)) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jUrl, GetJavaInstance_String(jEnv, url));
    SetResultContainerResult(jEnv, jResultContainer, jUrl);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_CreateChannelChatRoomManager(JNIEnv* jEnv, jobject /*jThis*/,
  jlong jNativePointer, jint jUserId, jint jChannelId, jobject jListener, jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jListener, TTV_EC_INVALID_ARG);
  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;
  std::shared_ptr<IChannelChatRoomManager> result;

  auto context = gChatApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context == nullptr) {
    ec = TTV_EC_INVALID_ARG;
  }

  if (TTV_SUCCEEDED(ec)) {
    std::shared_ptr<JavaIChannelChatRoomManagerListenerProxy> listenerProxy =
      std::make_shared<JavaIChannelChatRoomManagerListenerProxy>();
    listenerProxy->SetListener(jListener);

    ec = api->CreateChannelChatRoomManager(
      static_cast<UserId>(jUserId), static_cast<ChannelId>(jChannelId), listenerProxy, result);
  }

  if (TTV_SUCCEEDED(ec)) {
    TTV_ASSERT(result != nullptr);

    JavaClassInfo& info = GetJavaClassInfo_ChannelChatRoomManagerProxy(jEnv);

    // Create the proxy java object and pass the native raw pointer to it
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jProxy,
      jEnv->NewObject(info.klass, info.methods["<init>"], reinterpret_cast<jlong>(result.get())));
    SetResultContainerResult(jEnv, jResultContainer, jProxy);

    gIChannelChatRoomManagerInstanceRegistry.Register(result, context, jProxy);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_CreateChatRoom(JNIEnv* jEnv, jobject /*jThis*/,
  jlong jNativePointer, jint jUserId, jstring jRoomId, jint jChannelId, jobject jListener, jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jRoomId, TTV_EC_INVALID_ARG);
  TTV_JNI_RETURN_ON_NULL(jEnv, jListener, TTV_EC_INVALID_ARG);
  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;
  std::shared_ptr<IChatRoom> result;

  auto context = gChatApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context == nullptr) {
    ec = TTV_EC_INVALID_ARG;
  }

  if (TTV_SUCCEEDED(ec)) {
    ScopedJavaUTFStringConverter roomId(jEnv, jRoomId);

    std::shared_ptr<JavaIChatRoomListenerProxy> listenerProxy = std::make_shared<JavaIChatRoomListenerProxy>();
    listenerProxy->SetListener(jListener);

    ec = api->CreateChatRoom(
      static_cast<UserId>(jUserId), roomId, static_cast<ChannelId>(jChannelId), listenerProxy, result);
  }

  if (TTV_SUCCEEDED(ec)) {
    TTV_ASSERT(result != nullptr);

    JavaClassInfo& info = GetJavaClassInfo_ChatRoomProxy(jEnv);

    // Create the proxy java object and pass the native raw pointer to it
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jProxy,
      jEnv->NewObject(info.klass, info.methods["<init>"], reinterpret_cast<jlong>(result.get())));
    SetResultContainerResult(jEnv, jResultContainer, jProxy);

    gIChatRoomInstanceRegistry.Register(result, context, jProxy);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_CreateChatRoomNotifications(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jobject jListener, jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jListener, TTV_EC_INVALID_ARG);
  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;
  std::shared_ptr<IChatRoomNotifications> result;

  auto context = gChatApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context == nullptr) {
    ec = TTV_EC_INVALID_ARG;
  }

  if (TTV_SUCCEEDED(ec)) {
    std::shared_ptr<JavaIChatRoomNotificationsListenerProxy> listenerProxy =
      std::make_shared<JavaIChatRoomNotificationsListenerProxy>();
    listenerProxy->SetListener(jListener);

    ec = api->CreateChatRoomNotifications(static_cast<UserId>(jUserId), listenerProxy, result);
  }

  if (TTV_SUCCEEDED(ec)) {
    TTV_ASSERT(result != nullptr);

    JavaClassInfo& info = GetJavaClassInfo_ChatRoomNotificationsProxy(jEnv);

    // Create the proxy java object and pass the native raw pointer to it
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jProxy,
      jEnv->NewObject(info.klass, info.methods["<init>"], reinterpret_cast<jlong>(result.get())));
    SetResultContainerResult(jEnv, jResultContainer, jProxy);

    gIChatRoomNotificationsInstanceRegistry.Register(result, context, jProxy);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_CreateChatChannel(JNIEnv* jEnv, jobject /*jThis*/,
  jlong jNativePointer, jobject jJniThreadValidator, jint jUserId, jint jChannelId, jobject jListener,
  jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jListener, TTV_EC_INVALID_ARG);
  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;
  std::shared_ptr<IChatChannel> result;

  auto context = gChatApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context == nullptr) {
    ec = TTV_EC_INVALID_ARG;
  }

  if (TTV_SUCCEEDED(ec)) {
    std::shared_ptr<JavaChatChannelListenerProxy> listenerProxy = std::make_shared<JavaChatChannelListenerProxy>();
    listenerProxy->SetListener(jListener);

    ec =
      api->CreateChatChannel(static_cast<UserId>(jUserId), static_cast<ChannelId>(jChannelId), listenerProxy, result);
  }

  if (TTV_SUCCEEDED(ec)) {
    TTV_ASSERT(result != nullptr);

    JavaClassInfo& info = GetJavaClassInfo_ChatChannelProxy(jEnv);

    // Create the proxy java object and pass the native raw pointer to it
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jChatChannel,
      jEnv->NewObject(info.klass, info.methods["<init>"], reinterpret_cast<jlong>(result.get()), jJniThreadValidator));
    SetResultContainerResult(jEnv, jResultContainer, jChatChannel);

    gIChatChannelRegistry.Register(result, context, jChatChannel);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_CreateChatRaid(JNIEnv* jEnv, jobject /*jThis*/,
  jlong jNativePointer, jobject jJniThreadValidator, jint jUserId, jint jChannelId, jobject jListener,
  jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jListener, TTV_EC_INVALID_ARG);
  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;
  std::shared_ptr<IChatRaid> result;

  auto context = gChatApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context == nullptr) {
    ec = TTV_EC_INVALID_ARG;
  }

  if (TTV_SUCCEEDED(ec)) {
    std::shared_ptr<JavaChatRaidListenerProxy> listenerProxy = std::make_shared<JavaChatRaidListenerProxy>();
    listenerProxy->SetListener(jListener);

    ec = api->CreateChatRaid(static_cast<UserId>(jUserId), static_cast<ChannelId>(jChannelId), listenerProxy, result);
  }

  if (TTV_SUCCEEDED(ec)) {
    TTV_ASSERT(result != nullptr);

    JavaClassInfo& info = GetJavaClassInfo_ChatRaidProxy(jEnv);

    // Create the proxy java object and pass the native raw pointer to it
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jChatRaid,
      jEnv->NewObject(info.klass, info.methods["<init>"], reinterpret_cast<jlong>(result.get()), jJniThreadValidator));
    SetResultContainerResult(jEnv, jResultContainer, jChatRaid);

    gIChatRaidInstanceRegistry.Register(result, context, jChatRaid);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_CreateChatChannelProperties(JNIEnv* jEnv, jobject /*jThis*/,
  jlong jNativePointer, jint jUserId, jint jChannelId, jobject jListener, jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jListener, TTV_EC_INVALID_ARG);
  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;
  std::shared_ptr<IChatChannelProperties> result;

  auto context = gChatApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context == nullptr) {
    ec = TTV_EC_INVALID_ARG;
  }

  if (TTV_SUCCEEDED(ec)) {
    std::shared_ptr<JavaIChatChannelPropertyListenerProxy> listenerProxy =
      std::make_shared<JavaIChatChannelPropertyListenerProxy>();
    listenerProxy->SetListener(jListener);

    ec = api->CreateChatChannelProperties(
      static_cast<UserId>(jUserId), static_cast<ChannelId>(jChannelId), listenerProxy, result);
  }

  if (TTV_SUCCEEDED(ec)) {
    TTV_ASSERT(result != nullptr);

    JavaClassInfo& info = GetJavaClassInfo_ChatChannelPropertiesProxy(jEnv);

    // Create the proxy java object and pass the native raw pointer to it
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jProxy,
      jEnv->NewObject(info.klass, info.methods["<init>"], reinterpret_cast<jlong>(result.get())));
    SetResultContainerResult(jEnv, jResultContainer, jProxy);

    gIChatChannelPropertiesInstanceRegistry.Register(result, context, jProxy);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_CreateChatCommentManager(JNIEnv* jEnv, jobject /*jThis*/,
  jlong jNativePointer, jobject jJniThreadValidator, jint jUserId, jstring jVodId, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jListener, TTV_EC_INVALID_ARG);

  std::shared_ptr<IChatCommentManager> result;

  auto context = gChatApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context == nullptr) {
    return GetJavaInstance_ErrorResult(jEnv, TTV_EC_INVALID_ARG);
  }

  std::shared_ptr<JavaIChatCommentListenerProxy> listenerProxy = std::make_shared<JavaIChatCommentListenerProxy>();
  listenerProxy->SetListener(jListener);

  ScopedJavaUTFStringConverter vodId(jEnv, jVodId);

  return GetJavaInstance_Result(jEnv,
    api->CreateChatCommentManager(static_cast<UserId>(jUserId), vodId.GetNativeString(), listenerProxy),
    [jEnv, jJniThreadValidator, &context](const std::shared_ptr<IChatCommentManager>& result) {
      JavaClassInfo& info = GetJavaClassInfo_ChatCommentManagerProxy(jEnv);

      // Create the proxy java object and pass the native raw pointer to it
      jobject jProxy =
        jEnv->NewObject(info.klass, info.methods["<init>"], reinterpret_cast<jlong>(result.get()), jJniThreadValidator);

      gIChatCommentManagerInstanceRegistry.Register(result, context, jProxy);

      return jProxy;
    });
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_FetchChannelVodCommentSettings(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jint jChannelId, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  auto callbackWrapper = CreateJavaCallbackWrapper<jobject, jobject>(
    jEnv, jCallback, GetJavaClassInfo_ChatAPI_FetchChannelVodCommentSettingsCallback(jEnv));
  TTV_ErrorCode ec =
    api->FetchChannelVodCommentSettings(static_cast<UserId>(jUserId), static_cast<ChannelId>(jChannelId),
      [callbackWrapper](TTV_ErrorCode callbackEc, ChannelVodCommentSettings&& settings) {
        AUTO_DELETE_LOCAL_REF(
          gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
        AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jCommentSettings,
          GetJavaInstance_ChannelVodCommentSettings(gActiveJavaEnvironment, settings));
        callbackWrapper(jErrorCode, jCommentSettings);
      });
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_SetChannelVodFollowersOnlyDuration(JNIEnv* jEnv,
  jobject /*jThis*/, jlong jNativePointer, jint jUserId, jint jChannelId, jint jFollowersOnlyDuration,
  jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  auto callbackWrapper = CreateJavaCallbackWrapper<jobject>(
    jEnv, jCallback, GetJavaClassInfo_ChatAPI_SetChannelVodCommentSettingsCallback(jEnv));
  TTV_ErrorCode ec =
    api->SetChannelVodFollowersOnlyDuration(static_cast<UserId>(jUserId), static_cast<ChannelId>(jChannelId),
      static_cast<uint32_t>(jFollowersOnlyDuration), [callbackWrapper](TTV_ErrorCode callbackEc) {
        AUTO_DELETE_LOCAL_REF(
          gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
        callbackWrapper(jErrorCode);
      });
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_SetChannelVodPublishingMode(JNIEnv* jEnv, jobject /*jThis*/,
  jlong jNativePointer, jint jUserId, jint jChannelId, jobject jPublishingMode, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  CommentPublishingMode publishingMode = GetNativeFromJava_SimpleEnum<CommentPublishingMode>(
    jEnv, GetJavaClassInfo_CommentPublishingMode(jEnv), jPublishingMode, CommentPublishingMode::Unknown);

  auto callbackWrapper = CreateJavaCallbackWrapper<jobject>(
    jEnv, jCallback, GetJavaClassInfo_ChatAPI_SetChannelVodCommentSettingsCallback(jEnv));
  TTV_ErrorCode ec = api->SetChannelVodPublishingMode(static_cast<UserId>(jUserId), static_cast<ChannelId>(jChannelId),
    publishingMode, [callbackWrapper](TTV_ErrorCode callbackEc) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      callbackWrapper(jErrorCode);
    });
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_DisposeChatRaid(
  JNIEnv* jEnv, jobject /*jThis*/, jlong /*jNativePointer*/, jobject jChatRaid) {
  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  auto instance = gIChatRaidInstanceRegistry.LookupNativeInstance(jChatRaid);
  if (instance == nullptr) {
    ec = TTV_EC_INVALID_ARG;
  }

  if (TTV_SUCCEEDED(ec)) {
    ec = instance->Dispose();
  }

  if (TTV_SUCCEEDED(ec)) {
    gIChatRaidInstanceRegistry.Unregister(jChatRaid);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_CreateBitsStatus(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jobject jListener, jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jListener, TTV_EC_INVALID_ARG);
  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;
  std::shared_ptr<IBitsStatus> result;

  auto context = gChatApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context == nullptr) {
    ec = TTV_EC_INVALID_ARG;
  }

  if (TTV_SUCCEEDED(ec)) {
    std::shared_ptr<JavaBitsListenerProxy> listenerProxy = std::make_shared<JavaBitsListenerProxy>();
    listenerProxy->SetListener(jListener);

    ec = api->CreateBitsStatus(static_cast<UserId>(jUserId), listenerProxy, result);
  }

  if (TTV_SUCCEEDED(ec)) {
    TTV_ASSERT(result != nullptr);

    JavaClassInfo& info = GetJavaClassInfo_BitsStatusProxy(jEnv);

    // Create the proxy java object and pass the native raw pointer to it
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jBitsStatus,
      jEnv->NewObject(info.klass, info.methods["<init>"], reinterpret_cast<jlong>(result.get())));
    SetResultContainerResult(jEnv, jResultContainer, jBitsStatus);

    gIBitsStatusInstanceRegistry.Register(result, context, jBitsStatus);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_DisposeBitsStatus(
  JNIEnv* jEnv, jobject /*jThis*/, jlong /*jNativePointer*/, jobject jBitsStatus) {
  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  auto instance = gIBitsStatusInstanceRegistry.LookupNativeInstance(jBitsStatus);
  if (instance == nullptr) {
    ec = TTV_EC_INVALID_ARG;
  }

  if (TTV_SUCCEEDED(ec)) {
    ec = instance->Dispose();
  }

  if (TTV_SUCCEEDED(ec)) {
    gIBitsStatusInstanceRegistry.Unregister(jBitsStatus);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_CreateFollowersStatus(JNIEnv* jEnv, jobject /*jThis*/,
  jlong jNativePointer, jint jUserId, jint jChannelId, jobject jListener, jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jListener, TTV_EC_INVALID_ARG);
  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;
  std::shared_ptr<IFollowersStatus> result;

  auto context = gChatApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context == nullptr) {
    ec = TTV_EC_INVALID_ARG;
  }

  if (TTV_SUCCEEDED(ec)) {
    std::shared_ptr<JavaFollowersListenerProxy> listenerProxy = std::make_shared<JavaFollowersListenerProxy>();
    listenerProxy->SetListener(jListener);

    ec = api->CreateFollowersStatus(
      static_cast<UserId>(jUserId), static_cast<ChannelId>(jChannelId), listenerProxy, result);
  }

  if (TTV_SUCCEEDED(ec)) {
    TTV_ASSERT(result != nullptr);

    JavaClassInfo& info = GetJavaClassInfo_FollowersStatusProxy(jEnv);

    // Create the proxy java object and pass the native raw pointer to it
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jFollowersStatus,
      jEnv->NewObject(info.klass, info.methods["<init>"], reinterpret_cast<jlong>(result.get())));
    SetResultContainerResult(jEnv, jResultContainer, jFollowersStatus);

    gIFollowersStatusInstanceRegistry.Register(result, context, jFollowersStatus);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_DisposeFollowersStatus(
  JNIEnv* jEnv, jobject /*jThis*/, jlong /*jNativePointer*/, jobject jFollowersStatus) {
  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  auto instance = gIFollowersStatusInstanceRegistry.LookupNativeInstance(jFollowersStatus);
  if (instance == nullptr) {
    ec = TTV_EC_INVALID_ARG;
  }

  if (TTV_SUCCEEDED(ec)) {
    ec = instance->Dispose();
  }

  if (TTV_SUCCEEDED(ec)) {
    gIFollowersStatusInstanceRegistry.Unregister(jFollowersStatus);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_CreateFollowingStatus(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jobject jListener, jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jListener, TTV_EC_INVALID_ARG);
  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;
  std::shared_ptr<IFollowingStatus> result;

  auto context = gChatApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context == nullptr) {
    ec = TTV_EC_INVALID_ARG;
  }

  if (TTV_SUCCEEDED(ec)) {
    std::shared_ptr<JavaFollowingListenerProxy> listenerProxy = std::make_shared<JavaFollowingListenerProxy>();
    listenerProxy->SetListener(jListener);

    ec = api->CreateFollowingStatus(static_cast<UserId>(jUserId), listenerProxy, result);
  }

  if (TTV_SUCCEEDED(ec)) {
    TTV_ASSERT(result != nullptr);

    JavaClassInfo& info = GetJavaClassInfo_FollowingStatusProxy(jEnv);

    // Create the proxy java object and pass the native raw pointer to it
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jFollowingStatus,
      jEnv->NewObject(info.klass, info.methods["<init>"], reinterpret_cast<jlong>(result.get())));
    SetResultContainerResult(jEnv, jResultContainer, jFollowingStatus);

    gIFollowingStatusInstanceRegistry.Register(result, context, jFollowingStatus);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_CreateSubscribersStatus(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jobject jListener, jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jListener, TTV_EC_INVALID_ARG);
  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;
  std::shared_ptr<ISubscribersStatus> result;

  auto context = gChatApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context == nullptr) {
    ec = TTV_EC_INVALID_ARG;
  }

  if (TTV_SUCCEEDED(ec)) {
    std::shared_ptr<JavaSubscribersListenerProxy> listenerProxy = std::make_shared<JavaSubscribersListenerProxy>();
    listenerProxy->SetListener(jListener);

    ec = api->CreateSubscribersStatus(static_cast<UserId>(jUserId), listenerProxy, result);
  }

  if (TTV_SUCCEEDED(ec)) {
    TTV_ASSERT(result != nullptr);

    JavaClassInfo& info = GetJavaClassInfo_SubscribersStatusProxy(jEnv);

    // Create the proxy java object and pass the native raw pointer to it
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jSubscribersStatus,
      jEnv->NewObject(info.klass, info.methods["<init>"], reinterpret_cast<jlong>(result.get())));
    SetResultContainerResult(jEnv, jResultContainer, jSubscribersStatus);

    gISubscribersStatusInstanceRegistry.Register(result, context, jSubscribersStatus);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_DisposeSubscribersStatus(
  JNIEnv* jEnv, jobject /*jThis*/, jlong /*jNativePointer*/, jobject jSubscribersStatus) {
  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  auto instance = gISubscribersStatusInstanceRegistry.LookupNativeInstance(jSubscribersStatus);
  if (instance == nullptr) {
    ec = TTV_EC_INVALID_ARG;
  }

  if (TTV_SUCCEEDED(ec)) {
    ec = instance->Dispose();
  }

  if (TTV_SUCCEEDED(ec)) {
    gISubscribersStatusInstanceRegistry.Unregister(jSubscribersStatus);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_CreateSubscriptionsNotifications(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jListener, TTV_EC_INVALID_ARG);

  std::shared_ptr<ISubscriptionsNotifications> result;

  auto context = gChatApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context == nullptr) {
    return GetJavaInstance_ErrorResult(jEnv, TTV_EC_INVALID_ARG);
  }

  std::shared_ptr<JavaISubscriptionsNotificationsListenerProxy> listenerProxy =
    std::make_shared<JavaISubscriptionsNotificationsListenerProxy>();
  listenerProxy->SetListener(jListener);

  return GetJavaInstance_Result(jEnv,
    api->CreateSubscriptionsNotifications(static_cast<UserId>(jUserId), listenerProxy),
    [jEnv, &context](const std::shared_ptr<ISubscriptionsNotifications>& result) {
      JavaClassInfo& info = GetJavaClassInfo_SubscriptionsNotificationsProxy(jEnv);

      // Create the proxy java object and pass the native raw pointer to it
      jobject jProxy = jEnv->NewObject(info.klass, info.methods["<init>"], reinterpret_cast<jlong>(result.get()));

      gISubscriptionsNotificationsInstanceRegistry.Register(result, context, jProxy);

      return jProxy;
    });
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_CreateSquadNotifications(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jstring jSquadId, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jListener, TTV_EC_INVALID_ARG);

  std::shared_ptr<ISquadNotifications> result;

  auto context = gChatApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context == nullptr) {
    return GetJavaInstance_ErrorResult(jEnv, TTV_EC_INVALID_ARG);
  }

  ScopedJavaUTFStringConverter squadId(jEnv, jSquadId);

  std::shared_ptr<JavaISquadNotificationsListenerProxy> listenerProxy =
    std::make_shared<JavaISquadNotificationsListenerProxy>();
  listenerProxy->SetListener(jListener);

  return GetJavaInstance_Result(jEnv,
    api->CreateSquadNotifications(static_cast<UserId>(jUserId), squadId, listenerProxy),
    [jEnv, &context](const std::shared_ptr<ISquadNotifications>& result) {
      JavaClassInfo& info = GetJavaClassInfo_SquadNotificationsProxy(jEnv);

      // Create the proxy java object and pass the native raw pointer to it
      jobject jProxy = jEnv->NewObject(info.klass, info.methods["<init>"], reinterpret_cast<jlong>(result.get()));

      gISquadNotificationsInstanceRegistry.Register(result, context, jProxy);

      return jProxy;
    });
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_CreateMultiviewNotifications(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jint jChannelId, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jListener, TTV_EC_INVALID_ARG);

  std::shared_ptr<IMultiviewNotifications> result;

  auto context = gChatApiNativeProxyRegistry.LookupNativeContext(jNativePointer);
  if (context == nullptr) {
    return GetJavaInstance_ErrorResult(jEnv, TTV_EC_INVALID_ARG);
  }

  std::shared_ptr<JavaIMultiviewNotificationsListenerProxy> listenerProxy =
    std::make_shared<JavaIMultiviewNotificationsListenerProxy>();
  listenerProxy->SetListener(jListener);

  return GetJavaInstance_Result(jEnv,
    api->CreateMultiviewNotifications(static_cast<UserId>(jUserId), static_cast<UserId>(jChannelId), listenerProxy),
    [jEnv, &context](const std::shared_ptr<IMultiviewNotifications>& result) {
      JavaClassInfo& info = GetJavaClassInfo_MultiviewNotificationsProxy(jEnv);

      // Create the proxy java object and pass the native raw pointer to it
      jobject jProxy = jEnv->NewObject(info.klass, info.methods["<init>"], reinterpret_cast<jlong>(result.get()));

      gIMultiviewNotificationsInstanceRegistry.Register(result, context, jProxy);

      return jProxy;
    });
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_BanUser(JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer,
  jint jUserId, jint jChannelId, jstring jBannedUserName, jint jDuration, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  ScopedJavaUTFStringConverter bannedUserName(jEnv, jBannedUserName);

  auto callbackWrapper =
    CreateJavaCallbackWrapper<jobject, jobject>(jEnv, jCallback, GetJavaClassInfo_ChatAPI_BanUserCallback(jEnv));
  TTV_ErrorCode ec =
    api->BanUser(static_cast<UserId>(jUserId), static_cast<ChannelId>(jChannelId), bannedUserName.GetNativeString(),
      static_cast<uint32_t>(jDuration), [callbackWrapper](TTV_ErrorCode callbackEc, BanUserError&& error) {
        AUTO_DELETE_LOCAL_REF(
          gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
        AUTO_DELETE_LOCAL_REF(
          gActiveJavaEnvironment, jobject, jGraphQLError, GetJavaInstance_BanUserError(gActiveJavaEnvironment, error));
        callbackWrapper(jErrorCode, jGraphQLError);
      });
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_UnbanUser(JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer,
  jint jUserId, jint jChannelId, jstring jBannedUserName, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  ScopedJavaUTFStringConverter bannedUserName(jEnv, jBannedUserName);

  auto callbackWrapper =
    CreateJavaCallbackWrapper<jobject, jobject>(jEnv, jCallback, GetJavaClassInfo_ChatAPI_UnbanUserCallback(jEnv));
  TTV_ErrorCode ec = api->UnbanUser(static_cast<UserId>(jUserId), static_cast<ChannelId>(jChannelId),
    bannedUserName.GetNativeString(), [callbackWrapper](TTV_ErrorCode callbackEc, UnbanUserError&& error) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jGraphQLError, GetJavaInstance_UnbanUserError(gActiveJavaEnvironment, error));
      callbackWrapper(jErrorCode, jGraphQLError);
    });
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_ModUser(JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer,
  jint jUserId, jint jChannelId, jstring jModUserName, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  ScopedJavaUTFStringConverter modUserName(jEnv, jModUserName);

  auto callbackWrapper =
    CreateJavaCallbackWrapper<jobject, jobject>(jEnv, jCallback, GetJavaClassInfo_ChatAPI_ModUserCallback(jEnv));
  TTV_ErrorCode ec = api->ModUser(static_cast<UserId>(jUserId), static_cast<ChannelId>(jChannelId),
    modUserName.GetNativeString(), [callbackWrapper](TTV_ErrorCode callbackEc, ModUserError&& error) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jGraphQLError, GetJavaInstance_ModUserError(gActiveJavaEnvironment, error));
      callbackWrapper(jErrorCode, jGraphQLError);
    });
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_UnmodUser(JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer,
  jint jUserId, jint jChannelId, jstring jUnmodUserName, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  ScopedJavaUTFStringConverter unmodUserName(jEnv, jUnmodUserName);

  auto callbackWrapper =
    CreateJavaCallbackWrapper<jobject, jobject>(jEnv, jCallback, GetJavaClassInfo_ChatAPI_UnmodUserCallback(jEnv));
  TTV_ErrorCode ec = api->UnmodUser(static_cast<UserId>(jUserId), static_cast<ChannelId>(jChannelId),
    unmodUserName.GetNativeString(), [callbackWrapper](TTV_ErrorCode callbackEc, UnmodUserError&& error) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jGraphQLError, GetJavaInstance_UnmodUserError(gActiveJavaEnvironment, error));
      callbackWrapper(jErrorCode, jGraphQLError);
    });
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_GrantVIP(JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer,
  jint jUserId, jint jChannelId, jstring jVipUserName, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  ScopedJavaUTFStringConverter vipUserName(jEnv, jVipUserName);

  auto callbackWrapper =
    CreateJavaCallbackWrapper<jobject, jobject>(jEnv, jCallback, GetJavaClassInfo_ChatAPI_GrantVIPCallback(jEnv));
  TTV_ErrorCode ec = api->GrantVIP(static_cast<UserId>(jUserId), static_cast<ChannelId>(jChannelId),
    vipUserName.GetNativeString(), [callbackWrapper](TTV_ErrorCode callbackEc, GrantVIPErrorCode error) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jGraphQLErrorCode,
        GetJavaInstance_SimpleEnum(
          gActiveJavaEnvironment, GetJavaClassInfo_GrantVIPErrorCode(gActiveJavaEnvironment), error));
      callbackWrapper(jErrorCode, jGraphQLErrorCode);
    });
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_RevokeVIP(JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer,
  jint jUserId, jint jChannelId, jstring jUnvipUserName, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  ScopedJavaUTFStringConverter unvipUserName(jEnv, jUnvipUserName);

  auto callbackWrapper =
    CreateJavaCallbackWrapper<jobject, jobject>(jEnv, jCallback, GetJavaClassInfo_ChatAPI_RevokeVIPCallback(jEnv));
  TTV_ErrorCode ec = api->RevokeVIP(static_cast<UserId>(jUserId), static_cast<ChannelId>(jChannelId),
    unvipUserName.GetNativeString(), [callbackWrapper](TTV_ErrorCode callbackEc, RevokeVIPErrorCode error) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jGraphQLErrorCode,
        GetJavaInstance_SimpleEnum(
          gActiveJavaEnvironment, GetJavaClassInfo_RevokeVIPErrorCode(gActiveJavaEnvironment), error));
      callbackWrapper(jErrorCode, jGraphQLErrorCode);
    });
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_UpdateUserColor(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jstring jColor, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  ScopedJavaUTFStringConverter color(jEnv, jColor);

  auto callbackWrapper =
    CreateJavaCallbackWrapper<jobject>(jEnv, jCallback, GetJavaClassInfo_ChatAPI_UpdateUserColorCallback(jEnv));
  TTV_ErrorCode ec = api->UpdateUserColor(
    static_cast<UserId>(jUserId), color.GetNativeString(), [callbackWrapper](TTV_ErrorCode callbackEc) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      callbackWrapper(jErrorCode);
    });
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_FetchChannelModerators(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jChannelId, jstring jCursor, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  std::string cursor;
  if (jCursor != nullptr) {
    ScopedJavaUTFStringConverter temp(jEnv, jCursor);
    cursor = temp.GetNativeString();
  }

  auto callbackWrapper = CreateJavaCallbackWrapper<jobject, jobjectArray, jstring>(
    jEnv, jCallback, GetJavaClassInfo_ChatAPI_FetchChannelModeratorsCallback(jEnv));
  TTV_ErrorCode ec = api->FetchChannelModerators(static_cast<ChannelId>(jChannelId), cursor,
    [callbackWrapper](
      TTV_ErrorCode callbackEc, const std::vector<std::string>& modNames, const std::string& nextCursor) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobjectArray, jModNames, GetJavaInstance_StringArray(gActiveJavaEnvironment, modNames));

      jstring jNextCursor = nullptr;
      if (!nextCursor.empty()) {
        jNextCursor = GetJavaInstance_String(gActiveJavaEnvironment, nextCursor);
      }

      AUTO_DELETE_LOCAL_REF_NO_DECLARE(gActiveJavaEnvironment, jstring, jNextCursor);

      callbackWrapper(jErrorCode, jModNames, jNextCursor);
    });
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_FetchChannelVIPs(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jChannelId, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_CHATAPI_PTR(jNativePointer);

  auto callbackWrapper = CreateJavaCallbackWrapper<jobject, jobjectArray>(
    jEnv, jCallback, GetJavaClassInfo_ChatAPI_FetchChannelVIPsCallback(jEnv));
  TTV_ErrorCode ec = api->FetchChannelVIPs(static_cast<ChannelId>(jChannelId),
    [callbackWrapper](TTV_ErrorCode callbackEc, std::vector<std::string>&& vipNames) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jErrorCode, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobjectArray, jVipNames, GetJavaInstance_StringArray(gActiveJavaEnvironment, vipNames));
      callbackWrapper(jErrorCode, jVipNames);
    });
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

// NOTE: Exposing this is a hack and should be removed when we move newsfeeds to the SDK
#include "twitchsdk/chat/internal/chatmessageparsing.h"

JNIEXPORT jboolean JNICALL Java_tv_twitch_chat_ChatAPI_TokenizeServerMessage(JNIEnv* jEnv, jclass /*jKlass*/,
  jstring jMessage, jobject jTokenizationOptions, jstring jEmoteRanges, jobjectArray jLocalUserNames,
  jobject jResultContainer) {
  if (jMessage == nullptr || jEmoteRanges == nullptr || jResultContainer == nullptr) {
    return JNI_FALSE;
  }

  ScopedJavaUTFStringConverter message(jEnv, jMessage);
  ScopedJavaUTFStringConverter emotesMessageTag(jEnv, jEmoteRanges);

  std::map<std::string, std::vector<EmoteRange>> emoticonRanges;
  bool ret = ParseEmotesMessageTag(emotesMessageTag.GetNativeString(), emoticonRanges);
  if (!ret) {
    return JNI_FALSE;
  }

  TokenizationOptions options;
  GetNativeInstance_ChatTokenizationOptions(jEnv, jTokenizationOptions, options);

  std::vector<std::string> localUserNames;
  GetNativeInstance_StringVector(jEnv, jLocalUserNames, localUserNames);

  MessageInfo tokenizedMessage;
  ttv::chat::TokenizeServerMessage(
    message.GetNativeString(), options, emoticonRanges, nullptr, localUserNames, tokenizedMessage);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jResult, GetJavaInstance_ChatMessageInfo(jEnv, tokenizedMessage));
  SetResultContainerResult(jEnv, jResultContainer, jResult);

  return JNI_TRUE;
}
