/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/generated/jni_chatcommentmanagerproxy.h"

#include "twitchsdk/chat/generated/java_all.h"
#include "twitchsdk/chat/internal/chatcommentmanager.h"
#include "twitchsdk/chat/java_chatutil.h"

using namespace ttv;
using namespace ttv::chat;
using namespace ttv::binding::java;

#define GET_NATIVE_PTR(x) reinterpret_cast<IChatCommentManager*>(x)

JNIEXPORT void JNICALL Java_tv_twitch_chat_ChatCommentManagerProxy_DisposeNativeInstance(
  JNIEnv* /*jEnv*/, jobject jThis, jlong /*jNativeObjectPointer*/) {
  auto instance = gIChatCommentManagerInstanceRegistry.LookupNativeInstance(jThis);
  if (instance == nullptr) {
    return;
  }

  TTV_ErrorCode ec = instance->Dispose();

  if (TTV_SUCCEEDED(ec)) {
    gIChatCommentManagerInstanceRegistry.Unregister(jThis);
  }
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatCommentManagerProxy_Play(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);

  TTV_ErrorCode ec = instance->Play();

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatCommentManagerProxy_Pause(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);

  TTV_ErrorCode ec = instance->Pause();

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatCommentManagerProxy_UpdatePlayhead(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jint jTimestampMilliseconds) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);

  TTV_ErrorCode ec = instance->UpdatePlayhead(static_cast<uint64_t>(jTimestampMilliseconds));

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatCommentManagerProxy_GetPlayheadTime(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);

  return GetJavaInstance_Result(jEnv, instance->GetPlayheadTime(),
    [jEnv](uint64_t playheadTime) { return GetJavaInstance_Long(jEnv, playheadTime); });
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatCommentManagerProxy_GetChannelId(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);

  return GetJavaInstance_Result(
    jEnv, instance->GetChannelId(), [jEnv](ChannelId channelId) { return GetJavaInstance_Integer(jEnv, channelId); });
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatCommentManagerProxy_GetPlayingState(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);

  return GetJavaInstance_Result(
    jEnv, instance->GetPlayingState(), [jEnv](IChatCommentManager::PlayingState playingState) {
      return GetJavaInstance_SimpleEnum(jEnv, GetJavaClassInfo_PlayingState(jEnv), playingState);
    });
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatCommentManagerProxy_FetchCommentsByTimestamp(JNIEnv* jEnv,
  jobject /*jThis*/, jlong jNativeObjectPointer, jint jTimestampMilliseconds, jint jLimit, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject, jobjectArray, jobject>(
    jEnv, jCallback, GetJavaClassInfo_IChatCommentManager_FetchCommentsCallback(jEnv));

  TTV_ErrorCode ec =
    instance->FetchComments(static_cast<uint64_t>(jTimestampMilliseconds), static_cast<uint32_t>(jLimit),
      [callback](TTV_ErrorCode callbackEc, std::vector<ChatComment>&& comments, std::string&& nextCursor) {
        AUTO_DELETE_LOCAL_REF(
          gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
        AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobjectArray, jComments,
          GetJavaInstance_ChatCommentArray(gActiveJavaEnvironment, comments));
        AUTO_DELETE_LOCAL_REF(
          gActiveJavaEnvironment, jobject, jNextCursor, GetJavaInstance_String(gActiveJavaEnvironment, nextCursor));

        callback(jError, jComments, jNextCursor);
      });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatCommentManagerProxy_FetchCommentsByCursor(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jstring jCursor, jint jLimit, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject, jobjectArray, jobject>(
    jEnv, jCallback, GetJavaClassInfo_IChatCommentManager_FetchCommentsCallback(jEnv));

  ScopedJavaUTFStringConverter cursor(jEnv, jCursor);

  TTV_ErrorCode ec = instance->FetchComments(cursor.GetNativeString(), static_cast<uint32_t>(jLimit),
    [callback](TTV_ErrorCode callbackEc, std::vector<ChatComment>&& comments, std::string&& nextCursor) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobjectArray, jComments,
        GetJavaInstance_ChatCommentArray(gActiveJavaEnvironment, comments));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jNextCursor, GetJavaInstance_String(gActiveJavaEnvironment, nextCursor));

      callback(jError, jComments, jNextCursor);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatCommentManagerProxy_FetchComment(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jstring jCommentId, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject, jobject>(
    jEnv, jCallback, GetJavaClassInfo_IChatCommentManager_FetchCommentCallback(jEnv));

  ScopedJavaUTFStringConverter commentId(jEnv, jCommentId);

  TTV_ErrorCode ec =
    instance->FetchComment(commentId.GetNativeString(), [callback](TTV_ErrorCode callbackEc, ChatComment&& comment) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jComment, GetJavaInstance_ChatComment(gActiveJavaEnvironment, comment));

      callback(jError, jComment);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatCommentManagerProxy_DeleteComment(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jstring jCommentId, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject>(
    jEnv, jCallback, GetJavaClassInfo_IChatCommentManager_DeleteCommentCallback(jEnv));

  ScopedJavaUTFStringConverter commentId(jEnv, jCommentId);

  TTV_ErrorCode ec = instance->DeleteComment(commentId.GetNativeString(), [callback](TTV_ErrorCode callbackEc) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));

    callback(jError);
  });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatCommentManagerProxy_PostComment(JNIEnv* jEnv, jobject /*jThis*/,
  jlong jNativeObjectPointer, jstring jMessage, jint timestampMilliseconds, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject, jobject, jobject>(
    jEnv, jCallback, GetJavaClassInfo_IChatCommentManager_PostCommentCallback(jEnv));

  ScopedJavaUTFStringConverter message(jEnv, jMessage);

  TTV_ErrorCode ec = instance->PostComment(message.GetNativeString(), static_cast<jint>(timestampMilliseconds),
    [callback](TTV_ErrorCode callbackEc, ChatComment&& comment, std::string&& errorMessage) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jComment, GetJavaInstance_ChatComment(gActiveJavaEnvironment, comment));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jErrorMessage, GetJavaInstance_String(gActiveJavaEnvironment, errorMessage));

      callback(jError, jComment, jErrorMessage);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatCommentManagerProxy_ReportComment(JNIEnv* jEnv, jobject /*jThis*/,
  jlong jNativeObjectPointer, jstring jCommentId, jstring jReason, jstring jDescription, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject>(
    jEnv, jCallback, GetJavaClassInfo_IChatCommentManager_ReportCommentCallback(jEnv));

  ScopedJavaUTFStringConverter commentId(jEnv, jCommentId);
  ScopedJavaUTFStringConverter reason(jEnv, jReason);
  ScopedJavaUTFStringConverter description(jEnv, jDescription);

  TTV_ErrorCode ec = instance->ReportComment(commentId.GetNativeString(), reason.GetNativeString(),
    description.GetNativeString(), [callback](TTV_ErrorCode callbackEc) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));

      callback(jError);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatCommentManagerProxy_FetchCommentReplies(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jstring jCommentId, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject, jobject>(
    jEnv, jCallback, GetJavaClassInfo_IChatCommentManager_FetchCommentRepliesCallback(jEnv));

  ScopedJavaUTFStringConverter commentId(jEnv, jCommentId);

  TTV_ErrorCode ec = instance->FetchCommentReplies(
    commentId.GetNativeString(), [callback](TTV_ErrorCode callbackEc, std::vector<ChatComment>&& replies) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jReplies, GetJavaInstance_ChatCommentArray(gActiveJavaEnvironment, replies));

      callback(jError, jReplies);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatCommentManagerProxy_PostCommentReply(JNIEnv* jEnv, jobject /*jThis*/,
  jlong jNativeObjectPointer, jstring jParentCommentId, jstring jMessage, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject, jobject>(
    jEnv, jCallback, GetJavaClassInfo_IChatCommentManager_PostCommentReplyCallback(jEnv));

  ScopedJavaUTFStringConverter parentCommentId(jEnv, jParentCommentId);
  ScopedJavaUTFStringConverter message(jEnv, jMessage);

  TTV_ErrorCode ec = instance->PostCommentReply(parentCommentId.GetNativeString(), message.GetNativeString(),
    [callback](TTV_ErrorCode callbackEc, ChatComment&& reply) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jReply, GetJavaInstance_ChatComment(gActiveJavaEnvironment, reply));

      callback(jError, jReply);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}
