/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/generated/jni_chatraidproxy.h"

#include "twitchsdk/chat/generated/java_ichatraid_cancelcallback.h"
#include "twitchsdk/chat/generated/java_ichatraid_joincallback.h"
#include "twitchsdk/chat/generated/java_ichatraid_leavecallback.h"
#include "twitchsdk/chat/generated/java_ichatraid_raidnowcallback.h"
#include "twitchsdk/chat/generated/java_ichatraid_startcallback.h"
#include "twitchsdk/chat/ichatraid.h"
#include "twitchsdk/chat/java_chatutil.h"

using namespace ttv;
using namespace ttv::chat;
using namespace ttv::binding::java;

#define GET_NATIVE_PTR(x) reinterpret_cast<IChatRaid*>(x)

JNIEXPORT void JNICALL Java_tv_twitch_chat_ChatRaidProxy_DisposeNativeInstance(
  JNIEnv* /*jEnv*/, jobject jThis, jlong /*jNativeObjectPointer*/) {
  auto context = gIChatRaidInstanceRegistry.LookupNativeContext(jThis);
  if (context == nullptr) {
    return;
  }

  auto instance = gIChatRaidInstanceRegistry.LookupNativeInstance(jThis);
  if (instance == nullptr) {
    return;
  }

  TTV_ErrorCode ec = instance->Dispose();

  if (TTV_SUCCEEDED(ec)) {
    gIChatRaidInstanceRegistry.Unregister(jThis);
  }
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatRaidProxy_Join(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jstring jRaidId, jobject jCallback) {
  TTV_JNI_RETURN_ON_NULL(jEnv, jRaidId, TTV_EC_INVALID_ARG);

  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  ScopedJavaUTFStringConverter raidId(jEnv, jRaidId);

  IChatRaid* raid = GET_NATIVE_PTR(jNativeObjectPointer);

  auto callback = CreateJavaCallbackWrapper<jobject>(jEnv, jCallback, GetJavaClassInfo_IChatRaid_JoinCallback(jEnv));

  TTV_ErrorCode ec = raid->Join(raidId.GetNativeString(), [callback](TTV_ErrorCode callbackEc) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
    callback(jError);
  });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatRaidProxy_Leave(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jstring jRaidId, jobject jCallback) {
  TTV_JNI_RETURN_ON_NULL(jEnv, jRaidId, TTV_EC_INVALID_ARG);

  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  ScopedJavaUTFStringConverter raidId(jEnv, jRaidId);

  IChatRaid* raid = GET_NATIVE_PTR(jNativeObjectPointer);

  auto callback = CreateJavaCallbackWrapper<jobject>(jEnv, jCallback, GetJavaClassInfo_IChatRaid_LeaveCallback(jEnv));

  TTV_ErrorCode ec = raid->Leave(raidId.GetNativeString(), [callback](TTV_ErrorCode callbackEc) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
    callback(jError);
  });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatRaidProxy_Start(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jint jTargetUserId, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  IChatRaid* raid = GET_NATIVE_PTR(jNativeObjectPointer);

  auto callback = CreateJavaCallbackWrapper<jobject>(jEnv, jCallback, GetJavaClassInfo_IChatRaid_StartCallback(jEnv));

  TTV_ErrorCode ec = raid->Start(static_cast<UserId>(jTargetUserId), [callback](TTV_ErrorCode callbackEc) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));

    callback(jError);
  });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatRaidProxy_RaidNow(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  IChatRaid* raid = GET_NATIVE_PTR(jNativeObjectPointer);

  auto callback = CreateJavaCallbackWrapper<jobject>(jEnv, jCallback, GetJavaClassInfo_IChatRaid_RaidNowCallback(jEnv));

  TTV_ErrorCode ec = raid->RaidNow([callback](TTV_ErrorCode callbackEc) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
    callback(jError);
  });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatRaidProxy_Cancel(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  IChatRaid* raid = GET_NATIVE_PTR(jNativeObjectPointer);

  auto callback = CreateJavaCallbackWrapper<jobject>(jEnv, jCallback, GetJavaClassInfo_IChatRaid_CancelCallback(jEnv));

  TTV_ErrorCode ec = raid->Cancel([callback](TTV_ErrorCode callbackEc) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
    callback(jError);
  });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}
