/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/generated/jni_chatroomproxy.h"

#include "twitchsdk/chat/generated/java_all.h"
#include "twitchsdk/chat/internal/chatroom.h"
#include "twitchsdk/chat/java_chatutil.h"

using namespace ttv;
using namespace ttv::chat;
using namespace ttv::binding::java;

#define GET_NATIVE_PTR(x) reinterpret_cast<IChatRoom*>(x)

JNIEXPORT void JNICALL Java_tv_twitch_chat_ChatRoomProxy_DisposeNativeInstance(
  JNIEnv* /*jEnv*/, jobject jThis, jlong /*jNativeObjectPointer*/) {
  auto instance = gIChatRoomInstanceRegistry.LookupNativeInstance(jThis);
  if (instance == nullptr) {
    return;
  }

  TTV_ErrorCode ec = instance->Dispose();

  if (TTV_SUCCEEDED(ec)) {
    gIChatRoomInstanceRegistry.Unregister(jThis);
  }
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatRoomProxy_DeleteRoom(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback =
    CreateJavaCallbackWrapper<jobject>(jEnv, jCallback, GetJavaClassInfo_IChatRoom_DeleteRoomCallback(jEnv));

  TTV_ErrorCode ec = instance->DeleteRoom([callback](TTV_ErrorCode callbackEc) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));

    callback(jError);
  });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatRoomProxy_SendMessage(JNIEnv* jEnv, jobject /*jThis*/,
  jlong jNativeObjectPointer, jstring jMessage, jobject jResultContainer, jobject jCallback) {
  TTV_JNI_RETURN_ON_NULL(jEnv, jMessage, TTV_EC_INVALID_ARG);
  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject, jobject, jobject>(
    jEnv, jCallback, GetJavaClassInfo_IChatRoom_SendMessageCallback(jEnv));
  ScopedJavaUTFStringConverter messageString(jEnv, jMessage);

  ChatRoomMessage placeholder;

  TTV_ErrorCode ec = instance->SendMessage(messageString.GetNativeString(), placeholder,
    [callback](TTV_ErrorCode callbackEc, SendRoomMessageError&& error, ChatRoomMessage&& message) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jGraphQLError,
        GetJavaInstance_SendRoomMessageError(gActiveJavaEnvironment, error));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jMessage, GetJavaInstance_ChatRoomMessage(gActiveJavaEnvironment, message));

      callback(jError, jGraphQLError, jMessage);
    });

  if (TTV_SUCCEEDED(ec)) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jResult, GetJavaInstance_ChatRoomMessage(jEnv, placeholder));
    SetResultContainerResult(jEnv, jResultContainer, jResult);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatRoomProxy_EditMessage(JNIEnv* jEnv, jobject /*jThis*/,
  jlong jNativeObjectPointer, jstring jMessageId, jstring jMessageStr, jobject jResultContainer, jobject jCallback) {
  TTV_JNI_RETURN_ON_NULL(jEnv, jMessageStr, TTV_EC_INVALID_ARG);
  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback =
    CreateJavaCallbackWrapper<jobject, jobject>(jEnv, jCallback, GetJavaClassInfo_IChatRoom_EditMessageCallback(jEnv));
  ScopedJavaUTFStringConverter messageId(jEnv, jMessageId);
  ScopedJavaUTFStringConverter messageStr(jEnv, jMessageStr);

  ChatRoomMessage placeholder;

  TTV_ErrorCode ec = instance->EditMessage(
    messageId, messageStr, placeholder, [callback](TTV_ErrorCode callbackEc, ChatRoomMessage&& message) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jMessage, GetJavaInstance_ChatRoomMessage(gActiveJavaEnvironment, message));

      callback(jError, jMessage);
    });

  if (TTV_SUCCEEDED(ec)) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jResult, GetJavaInstance_ChatRoomMessage(jEnv, placeholder));
    SetResultContainerResult(jEnv, jResultContainer, jResult);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatRoomProxy_DeleteMessage(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jstring jMessageId, jobject jCallback) {
  TTV_JNI_RETURN_ON_NULL(jEnv, jMessageId, TTV_EC_INVALID_ARG);

  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback =
    CreateJavaCallbackWrapper<jobject>(jEnv, jCallback, GetJavaClassInfo_IChatRoom_DeleteMessageCallback(jEnv));
  ScopedJavaUTFStringConverter messageId(jEnv, jMessageId);

  TTV_ErrorCode ec = instance->DeleteMessage(messageId, [callback](TTV_ErrorCode callbackEc) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));

    callback(jError);
  });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatRoomProxy_FetchMessagesBeforeCursor(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jstring jCursor, jint jLimit, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject, jobject, jobject, jboolean>(
    jEnv, jCallback, GetJavaClassInfo_IChatRoom_FetchMessagesCallback(jEnv));

  std::string cursor;

  if (jCursor != nullptr) {
    ScopedJavaUTFStringConverter temp(jEnv, jCursor);
    cursor = temp.GetNativeString();
  }

  TTV_ErrorCode ec = instance->FetchMessagesBeforeCursor(cursor, static_cast<uint32_t>(jLimit),
    [callback](
      TTV_ErrorCode callbackEc, std::vector<ChatRoomMessage>&& messages, std::string&& nextCursor, bool moreMessages) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jMessages,
        GetJavaInstance_Array(gActiveJavaEnvironment, GetJavaClassInfo_ChatRoomMessage(gActiveJavaEnvironment),
          static_cast<uint32_t>(messages.size()), [&messages](uint32_t index) {
            return GetJavaInstance_ChatRoomMessage(gActiveJavaEnvironment, messages[index]);
          }));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jNextCursor, GetJavaInstance_String(gActiveJavaEnvironment, nextCursor));

      callback(jError, jMessages, jNextCursor, moreMessages ? JNI_TRUE : JNI_FALSE);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatRoomProxy_FetchMessagesAfterCursor(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jstring jCursor, jint jLimit, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject, jobject, jobject, jboolean>(
    jEnv, jCallback, GetJavaClassInfo_IChatRoom_FetchMessagesCallback(jEnv));

  std::string cursor;

  if (jCursor != nullptr) {
    ScopedJavaUTFStringConverter temp(jEnv, jCursor);
    cursor = temp.GetNativeString();
  }

  TTV_ErrorCode ec = instance->FetchMessagesAfterCursor(cursor, static_cast<uint32_t>(jLimit),
    [callback](
      TTV_ErrorCode callbackEc, std::vector<ChatRoomMessage>&& messages, std::string&& nextCursor, bool moreMessages) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jMessages,
        GetJavaInstance_Array(gActiveJavaEnvironment, GetJavaClassInfo_ChatRoomMessage(gActiveJavaEnvironment),
          static_cast<uint32_t>(messages.size()), [&messages](uint32_t index) {
            return GetJavaInstance_ChatRoomMessage(gActiveJavaEnvironment, messages[index]);
          }));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jNextCursor, GetJavaInstance_String(gActiveJavaEnvironment, nextCursor));

      callback(jError, jMessages, jNextCursor, moreMessages ? JNI_TRUE : JNI_FALSE);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatRoomProxy_FetchMessagesBeforeTimestamp(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jint jTimestamp, jint jLimit, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject, jobject, jobject, jboolean>(
    jEnv, jCallback, GetJavaClassInfo_IChatRoom_FetchMessagesCallback(jEnv));

  TTV_ErrorCode ec = instance->FetchMessagesBeforeTimestamp(jTimestamp, static_cast<uint32_t>(jLimit),
    [callback](
      TTV_ErrorCode callbackEc, std::vector<ChatRoomMessage>&& messages, std::string&& nextCursor, bool moreMessages) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jMessages,
        GetJavaInstance_Array(gActiveJavaEnvironment, GetJavaClassInfo_ChatRoomMessage(gActiveJavaEnvironment),
          static_cast<uint32_t>(messages.size()), [&messages](uint32_t index) {
            return GetJavaInstance_ChatRoomMessage(gActiveJavaEnvironment, messages[index]);
          }));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jNextCursor, GetJavaInstance_String(gActiveJavaEnvironment, nextCursor));

      callback(jError, jMessages, jNextCursor, moreMessages ? JNI_TRUE : JNI_FALSE);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatRoomProxy_FetchMessagesAfterTimestamp(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jint jTimestamp, jint jLimit, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject, jobject, jobject, jboolean>(
    jEnv, jCallback, GetJavaClassInfo_IChatRoom_FetchMessagesCallback(jEnv));

  TTV_ErrorCode ec = instance->FetchMessagesAfterTimestamp(jTimestamp, static_cast<uint32_t>(jLimit),
    [callback](
      TTV_ErrorCode callbackEc, std::vector<ChatRoomMessage>&& messages, std::string&& nextCursor, bool moreMessages) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jMessages,
        GetJavaInstance_Array(gActiveJavaEnvironment, GetJavaClassInfo_ChatRoomMessage(gActiveJavaEnvironment),
          static_cast<uint32_t>(messages.size()), [&messages](uint32_t index) {
            return GetJavaInstance_ChatRoomMessage(gActiveJavaEnvironment, messages[index]);
          }));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jNextCursor, GetJavaInstance_String(gActiveJavaEnvironment, nextCursor));

      callback(jError, jMessages, jNextCursor, moreMessages ? JNI_TRUE : JNI_FALSE);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatRoomProxy_SetRoomName(
  JNIEnv* jEnv, jobject jThis, jlong jNativeObjectPointer, jstring jName, jobject jCallback) {
  TTV_JNI_RETURN_ON_NULL(jEnv, jName, TTV_EC_INVALID_ARG);

  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject, jobject, jobject>(
    jEnv, jCallback, GetJavaClassInfo_IChatRoom_UpdateRoomInfoCallback(jEnv));
  ScopedJavaUTFStringConverter name(jEnv, jName);

  TTV_ErrorCode ec =
    instance->SetRoomName(name, [callback](TTV_ErrorCode callbackEc, UpdateRoomError&& error, ChatRoomInfo&& info) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jGraphQLError, GetJavaInstance_UpdateRoomError(gActiveJavaEnvironment, error));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jRoomInfo, GetJavaInstance_ChatRoomInfo(gActiveJavaEnvironment, info));

      callback(jError, jGraphQLError, jRoomInfo);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatRoomProxy_SetTopic(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jstring jTopic, jobject jCallback) {
  TTV_JNI_RETURN_ON_NULL(jEnv, jTopic, TTV_EC_INVALID_ARG);

  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject, jobject, jobject>(
    jEnv, jCallback, GetJavaClassInfo_IChatRoom_UpdateRoomInfoCallback(jEnv));
  ScopedJavaUTFStringConverter topic(jEnv, jTopic);

  TTV_ErrorCode ec =
    instance->SetTopic(topic, [callback](TTV_ErrorCode callbackEc, UpdateRoomError&& error, ChatRoomInfo&& info) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jGraphQLError, GetJavaInstance_UpdateRoomError(gActiveJavaEnvironment, error));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jRoomInfo, GetJavaInstance_ChatRoomInfo(gActiveJavaEnvironment, info));

      callback(jError, jGraphQLError, jRoomInfo);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatRoomProxy_SetRoomRolePermissions(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jobject jPermissions, jobject jCallback) {
  TTV_JNI_RETURN_ON_NULL(jEnv, jPermissions, TTV_EC_INVALID_ARG);

  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject, jobject, jobject>(
    jEnv, jCallback, GetJavaClassInfo_IChatRoom_UpdateRoomInfoCallback(jEnv));

  RoomRolePermissions permissions;
  GetNativeInstance_RoomRolePermissions(jEnv, jPermissions, permissions);

  TTV_ErrorCode ec = instance->SetRoomRolePermissions(
    permissions, [callback](TTV_ErrorCode callbackEc, UpdateRoomError&& error, ChatRoomInfo&& info) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jGraphQLError, GetJavaInstance_UpdateRoomError(gActiveJavaEnvironment, error));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jRoomInfo, GetJavaInstance_ChatRoomInfo(gActiveJavaEnvironment, info));

      callback(jError, jGraphQLError, jRoomInfo);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatRoomProxy_EnableSlowMode(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jint jDurationSeconds, jobject jCallback) {
  if (jDurationSeconds <= 0) {
    return GetJavaInstance_ErrorCode(jEnv, TTV_EC_INVALID_ARG);
  }

  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject, jobject, jobject>(
    jEnv, jCallback, GetJavaClassInfo_IChatRoom_UpdateRoomModesCallback(jEnv));

  TTV_ErrorCode ec = instance->EnableSlowMode(static_cast<uint32_t>(jDurationSeconds),
    [callback](TTV_ErrorCode callbackEc, UpdateRoomModesError&& error, ChatRoomInfo&& info) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jGraphQLError,
        GetJavaInstance_UpdateRoomModesError(gActiveJavaEnvironment, error));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jRoomInfo, GetJavaInstance_ChatRoomInfo(gActiveJavaEnvironment, info));

      callback(jError, jGraphQLError, jRoomInfo);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatRoomProxy_DisableSlowMode(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject, jobject, jobject>(
    jEnv, jCallback, GetJavaClassInfo_IChatRoom_UpdateRoomModesCallback(jEnv));

  TTV_ErrorCode ec =
    instance->DisableSlowMode([callback](TTV_ErrorCode callbackEc, UpdateRoomModesError&& error, ChatRoomInfo&& info) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jGraphQLError,
        GetJavaInstance_UpdateRoomModesError(gActiveJavaEnvironment, error));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jRoomInfo, GetJavaInstance_ChatRoomInfo(gActiveJavaEnvironment, info));

      callback(jError, jGraphQLError, jRoomInfo);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatRoomProxy_EnableR9kMode(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject, jobject, jobject>(
    jEnv, jCallback, GetJavaClassInfo_IChatRoom_UpdateRoomModesCallback(jEnv));

  TTV_ErrorCode ec =
    instance->EnableR9kMode([callback](TTV_ErrorCode callbackEc, UpdateRoomModesError&& error, ChatRoomInfo&& info) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jGraphQLError,
        GetJavaInstance_UpdateRoomModesError(gActiveJavaEnvironment, error));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jRoomInfo, GetJavaInstance_ChatRoomInfo(gActiveJavaEnvironment, info));

      callback(jError, jGraphQLError, jRoomInfo);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatRoomProxy_DisableR9kMode(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject, jobject, jobject>(
    jEnv, jCallback, GetJavaClassInfo_IChatRoom_UpdateRoomModesCallback(jEnv));

  TTV_ErrorCode ec =
    instance->DisableR9kMode([callback](TTV_ErrorCode callbackEc, UpdateRoomModesError&& error, ChatRoomInfo&& info) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jGraphQLError,
        GetJavaInstance_UpdateRoomModesError(gActiveJavaEnvironment, error));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jRoomInfo, GetJavaInstance_ChatRoomInfo(gActiveJavaEnvironment, info));

      callback(jError, jGraphQLError, jRoomInfo);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatRoomProxy_EnableEmotesOnlyMode(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject, jobject, jobject>(
    jEnv, jCallback, GetJavaClassInfo_IChatRoom_UpdateRoomModesCallback(jEnv));

  TTV_ErrorCode ec = instance->EnableEmotesOnlyMode(
    [callback](TTV_ErrorCode callbackEc, UpdateRoomModesError&& error, ChatRoomInfo&& info) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jGraphQLError,
        GetJavaInstance_UpdateRoomModesError(gActiveJavaEnvironment, error));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jRoomInfo, GetJavaInstance_ChatRoomInfo(gActiveJavaEnvironment, info));

      callback(jError, jGraphQLError, jRoomInfo);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatRoomProxy_DisableEmotesOnlyMode(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject, jobject, jobject>(
    jEnv, jCallback, GetJavaClassInfo_IChatRoom_UpdateRoomModesCallback(jEnv));

  TTV_ErrorCode ec = instance->DisableEmotesOnlyMode(
    [callback](TTV_ErrorCode callbackEc, UpdateRoomModesError&& error, ChatRoomInfo&& info) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jGraphQLError,
        GetJavaInstance_UpdateRoomModesError(gActiveJavaEnvironment, error));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jRoomInfo, GetJavaInstance_ChatRoomInfo(gActiveJavaEnvironment, info));

      callback(jError, jGraphQLError, jRoomInfo);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatRoomProxy_SetLastReadAt(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jint jLastReadAt, jobject jCallback) {
  if (jLastReadAt <= 0) {
    return GetJavaInstance_ErrorCode(jEnv, TTV_EC_INVALID_ARG);
  }

  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject, jobject>(
    jEnv, jCallback, GetJavaClassInfo_IChatRoom_UpdateRoomViewCallback(jEnv));

  TTV_ErrorCode ec = instance->SetLastReadAt(
    static_cast<Timestamp>(jLastReadAt), [callback](TTV_ErrorCode callbackEc, ChatRoomInfo&& info) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jRoomInfo, GetJavaInstance_ChatRoomInfo(gActiveJavaEnvironment, info));

      callback(jError, jRoomInfo);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatRoomProxy_SetMuted(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jboolean jIsMuted, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject, jobject>(
    jEnv, jCallback, GetJavaClassInfo_IChatRoom_UpdateRoomViewCallback(jEnv));

  TTV_ErrorCode ec =
    instance->SetMuted(jIsMuted == JNI_TRUE, [callback](TTV_ErrorCode callbackEc, ChatRoomInfo&& info) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jRoomInfo, GetJavaInstance_ChatRoomInfo(gActiveJavaEnvironment, info));

      callback(jError, jRoomInfo);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatRoomProxy_SetArchived(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jboolean jIsArchived, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject, jobject>(
    jEnv, jCallback, GetJavaClassInfo_IChatRoom_UpdateRoomViewCallback(jEnv));

  TTV_ErrorCode ec =
    instance->SetArchived(jIsArchived == JNI_TRUE, [callback](TTV_ErrorCode callbackEc, ChatRoomInfo&& info) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jRoomInfo, GetJavaInstance_ChatRoomInfo(gActiveJavaEnvironment, info));

      callback(jError, jRoomInfo);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatRoomProxy_FetchRoomInfo(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativeObjectPointer, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  auto instance = GET_NATIVE_PTR(jNativeObjectPointer);
  auto callback = CreateJavaCallbackWrapper<jobject, jobject>(
    jEnv, jCallback, GetJavaClassInfo_IChatRoom_FetchRoomInfoCallback(jEnv));

  TTV_ErrorCode ec = instance->FetchRoomInfo([callback](TTV_ErrorCode callbackEc, ChatRoomInfo&& info) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, callbackEc));
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jRoomInfo, GetJavaInstance_ChatRoomInfo(gActiveJavaEnvironment, info));

    callback(jError, jRoomInfo);
  });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}
