/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/generated/jni_chattest.h"

#include "twitchsdk/chat/java_bitslistenerproxy.h"
#include "twitchsdk/chat/java_chatapilistenerproxy.h"
#include "twitchsdk/chat/java_chatchannellistenerproxy.h"
#include "twitchsdk/chat/java_chatraidlistenerproxy.h"
#include "twitchsdk/chat/java_chatuserthreadslistenerproxy.h"
#include "twitchsdk/chat/java_chatutil.h"
#include "twitchsdk/chat/java_followerslistenerproxy.h"
#include "twitchsdk/chat/java_followinglistenerproxy.h"
#include "twitchsdk/chat/java_ichatchannelpropertylistenerproxy.h"
#include "twitchsdk/chat/java_ichatcommentlistener_proxy.h"
#include "twitchsdk/chat/java_ichatroomlistener_proxy.h"
#include "twitchsdk/chat/java_ichatroomnotificationslistener_proxy.h"
#include "twitchsdk/chat/java_imultiviewnotificationslistener_proxy.h"
#include "twitchsdk/chat/java_isquadnotificationslistener_proxy.h"
#include "twitchsdk/chat/java_isubscriptionsnotificationslistener_proxy.h"
#include "twitchsdk/chat/java_subscriberslistenerproxy.h"
#include "twitchsdk/core/stringutilities.h"

using namespace ttv;
using namespace ttv::chat;
using namespace ttv::binding::java;

namespace {
void SetChatUserInfo(ChatUserInfo& userInfo, const std::string& userName, const std::string& displayName,
  UserMode modes, Color nameColorARGB) {
  userInfo.userName = userName;
  userInfo.displayName = displayName;
  userInfo.userMode = modes;
  userInfo.nameColorARGB = nameColorARGB;
}

void SetupChatWhisperMessageList(std::vector<WhisperMessage>& messageList, std::function<void()> func) {
  UserMode mode;
  mode.broadcaster = true;

  MessageInfo::Flags flags;
  flags.action = true;

  // Stack allocate the temporary messages
  messageList.resize(2);
  messageList[0].messageInfo.userName = "userName0";
  messageList[0].messageInfo.displayName = "displayName0";
  messageList[0].messageInfo.userId = 1111;
  messageList[0].threadId = "threadId0";
  messageList[0].messageInfo.userMode = mode;
  messageList[0].messageInfo.nameColorARGB = 123456;
  messageList[0].messageInfo.flags = flags;
  messageList[0].messageInfo.timestamp = 98989898;
  messageList[0].messageId = 82746;
  messageList[0].messageInfo.numBitsSent = 625;

  messageList[0].messageInfo.tokens.push_back(std::make_unique<TextToken>("yo "));
  messageList[0].messageInfo.tokens.push_back(std::make_unique<EmoticonToken>("Kappa", "25"));
  messageList[0].messageInfo.tokens.push_back(std::make_unique<TextToken>(" "));
  messageList[0].messageInfo.tokens.push_back(std::make_unique<EmoticonToken>("Kappa", "25"));

  messageList[1].messageInfo.userName = "userName1";
  messageList[1].messageInfo.displayName = "displayName1";
  messageList[1].messageInfo.userId = 2222;
  messageList[1].threadId = "threadId1";
  messageList[1].messageInfo.userMode = mode;
  messageList[1].messageInfo.nameColorARGB = 234567;
  messageList[1].messageInfo.flags = flags;
  messageList[1].messageInfo.timestamp = 878878887;
  messageList[1].messageId = 45454455;
  messageList[1].messageInfo.numBitsSent = 111;
  messageList[1].messageInfo.tokens.push_back(std::make_unique<TextToken>("yo "));
  messageList[1].messageInfo.tokens.push_back(std::make_unique<EmoticonToken>("Kappa", "25"));
  messageList[1].messageInfo.tokens.push_back(std::make_unique<TextToken>(" "));
  messageList[1].messageInfo.tokens.push_back(std::make_unique<EmoticonToken>("Kappa", "25"));

  // Call the callback the that will use the data before the stack frees it
  func();
}

void SetupChatLiveMessageList(std::vector<LiveChatMessage>& messageList, std::function<void()> func) {
  // Stack allocate the temporary messages
  messageList.resize(5);
  messageList[0].messageInfo.userName = "";
  messageList[0].messageInfo.displayName = "displayName0";
  messageList[0].messageInfo.userId = 0;
  messageList[0].messageInfo.userMode.moderator = true;
  messageList[0].messageInfo.userMode.broadcaster = true;
  messageList[0].messageInfo.userMode.administrator = true;
  messageList[0].messageInfo.userMode.staff = true;
  messageList[0].messageInfo.userMode.system = true;
  messageList[0].messageInfo.userMode.globalModerator = true;
  messageList[0].messageInfo.userMode.banned = true;
  messageList[0].messageInfo.userMode.subscriber = true;
  messageList[0].messageInfo.nameColorARGB = 123456;
  messageList[0].messageInfo.flags.action = true;
  messageList[0].messageInfo.flags.notice = true;
  messageList[0].messageInfo.flags.ignored = true;
  messageList[0].messageInfo.flags.deleted = true;
  messageList[0].messageInfo.flags.containsBits = true;
  messageList[0].messageInfo.timestamp = 012345;  // 5349
  messageList[0].messageId = "messageId0";
  messageList[0].messageInfo.numBitsSent = 0;
  messageList[0].messageInfo.tokens.push_back(std::make_unique<TextToken>(""));
  messageList[0].messageInfo.tokens.push_back(std::make_unique<EmoticonToken>("", ""));
  messageList[0].messageInfo.tokens.push_back(std::make_unique<MentionToken>("", "", false));
  messageList[0].messageInfo.tokens.push_back(std::make_unique<UrlToken>("", false));
  messageList[0].messageInfo.tokens.push_back(std::make_unique<BitsToken>("", 0));
  messageList[0].messageInfo.badges.push_back({"", ""});
  messageList[0].messageInfo.badges.push_back({"1", ""});
  messageList[0].messageInfo.badges.push_back({"2", ""});
  messageList[0].messageInfo.badges.push_back({"blah", "lol"});
  messageList[0].messageInfo.badges.push_back({"who", "wat"});
  messageList[0].messageInfo.messageType = "messageType0";
  messageList[0].messageInfo.messageTags.insert(
    // clang-format off
    {{"key00", "value00"},
     {"key01", "value01"},
     {"key02", "value02"},
     {"key03", "value03"},
     {"key04", "value04"}}
    // clang-format on
  );

  messageList[1].messageInfo.userName = "user-name 1";
  messageList[1].messageInfo.displayName = "displayName1";
  messageList[1].messageInfo.userId = 1111;
  messageList[1].messageInfo.userMode.moderator = false;
  messageList[1].messageInfo.userMode.broadcaster = false;
  messageList[1].messageInfo.userMode.administrator = false;
  messageList[1].messageInfo.userMode.staff = false;
  messageList[1].messageInfo.userMode.system = false;
  messageList[1].messageInfo.userMode.globalModerator = false;
  messageList[1].messageInfo.userMode.banned = false;
  messageList[1].messageInfo.userMode.subscriber = false;
  messageList[1].messageInfo.nameColorARGB = 234567;
  messageList[1].messageInfo.flags.action = false;
  messageList[1].messageInfo.flags.notice = false;
  messageList[1].messageInfo.flags.ignored = false;
  messageList[1].messageInfo.flags.deleted = false;
  messageList[1].messageInfo.flags.containsBits = false;
  messageList[1].messageInfo.timestamp = 111111;
  messageList[1].messageId = "messageId1";
  messageList[1].messageInfo.numBitsSent = 111;
  messageList[1].messageInfo.messageType = "messageType1";
  messageList[1].messageInfo.messageTags.insert(
    // clang-format off
    {{"key10", "value10"},
     {"key11", "value11"},
     {"key12", "value12"},
     {"key13", "value13"},
     {"key14", "value14"}}
    // clang-format on
  );
  // No Tokens or Badges

  messageList[2].messageInfo.userName = "USERNAME2";
  messageList[2].messageInfo.displayName = "displayName2";
  messageList[2].messageInfo.userId = 2222222;
  messageList[2].messageInfo.userMode.moderator = true;
  messageList[2].messageInfo.userMode.broadcaster = false;
  messageList[2].messageInfo.userMode.administrator = true;
  messageList[2].messageInfo.userMode.staff = false;
  messageList[2].messageInfo.userMode.system = true;
  messageList[2].messageInfo.userMode.globalModerator = false;
  messageList[2].messageInfo.userMode.banned = true;
  messageList[2].messageInfo.userMode.subscriber = false;
  messageList[2].messageInfo.nameColorARGB = 0;
  messageList[2].messageInfo.flags.action = true;
  messageList[2].messageInfo.flags.notice = false;
  messageList[2].messageInfo.flags.ignored = true;
  messageList[2].messageInfo.flags.deleted = false;
  messageList[2].messageInfo.flags.containsBits = true;
  messageList[2].messageInfo.timestamp = 2222;
  messageList[2].messageId = "messageId2";
  messageList[2].messageInfo.numBitsSent = 2222;
  messageList[2].messageInfo.tokens.push_back(std::make_unique<EmoticonToken>("Kappa", "2222"));
  messageList[2].messageInfo.tokens.push_back(std::make_unique<MentionToken>("user name", "text", false));
  messageList[2].messageInfo.tokens.push_back(std::make_unique<UrlToken>("urlurlulr", false));
  messageList[2].messageInfo.tokens.push_back(std::make_unique<BitsToken>("someBitsSent", 0));
  messageList[2].messageInfo.badges.push_back({"name", "version"});
  messageList[2].messageInfo.messageType = "messageType2";
  messageList[2].messageInfo.messageTags.insert(
    // clang-format off
    {{"key20", "value20"},
     {"key21", "value21"},
     {"key22", "value22"},
     {"key23", "value23"},
     {"key24", "value24"}}
    // clang-format on
  );

  messageList[3].messageInfo.userName = "33333";
  messageList[3].messageInfo.displayName = "displayName3";
  messageList[3].messageInfo.userId = 33333333;
  messageList[3].messageInfo.userMode.moderator = false;
  messageList[3].messageInfo.userMode.broadcaster = true;
  messageList[3].messageInfo.userMode.administrator = false;
  messageList[3].messageInfo.userMode.staff = true;
  messageList[3].messageInfo.userMode.system = false;
  messageList[3].messageInfo.userMode.globalModerator = true;
  messageList[3].messageInfo.userMode.banned = false;
  messageList[3].messageInfo.userMode.subscriber = true;
  messageList[3].messageInfo.nameColorARGB = 33333333;
  messageList[3].messageInfo.flags.action = false;
  messageList[3].messageInfo.flags.notice = true;
  messageList[3].messageInfo.flags.ignored = false;
  messageList[3].messageInfo.flags.deleted = true;
  messageList[3].messageInfo.flags.containsBits = false;
  messageList[3].messageInfo.timestamp = 33333;
  messageList[3].messageId = "messageId3";
  messageList[3].messageInfo.numBitsSent = 3333;
  messageList[3].messageInfo.tokens.push_back(std::make_unique<TextToken>("text3"));
  messageList[3].messageInfo.tokens.push_back(std::make_unique<MentionToken>("username3", "text3", true));
  messageList[3].messageInfo.tokens.push_back(std::make_unique<UrlToken>("urlurl", true));
  messageList[3].messageInfo.badges.push_back({"1", "4"});
  messageList[3].messageInfo.badges.push_back({"2", "4"});
  messageList[3].messageInfo.badges.push_back({"3", "4"});
  messageList[3].messageInfo.badges.push_back({"4", "4"});
  messageList[3].messageInfo.messageType = "messageType3";
  messageList[3].messageInfo.messageTags.insert(
    // clang-format off
    {{"key30", "value30"},
     {"key31", "value31"},
     {"key32", "value32"},
     {"key33", "value33"},
     {"key34", "value34"}}
    // clang-format on
  );

  messageList[4].messageInfo.userName = "            blahblah4   \0   blah ";
  messageList[4].messageInfo.displayName = "displayName4";
  messageList[4].messageInfo.userId = 4;
  messageList[4].messageInfo.userMode.moderator = true;
  messageList[4].messageInfo.userMode.broadcaster = true;
  messageList[4].messageInfo.userMode.administrator = false;
  messageList[4].messageInfo.userMode.staff = false;
  messageList[4].messageInfo.userMode.system = true;
  messageList[4].messageInfo.userMode.globalModerator = true;
  messageList[4].messageInfo.userMode.banned = false;
  messageList[4].messageInfo.userMode.subscriber = false;
  messageList[4].messageInfo.nameColorARGB = 444444;
  messageList[4].messageInfo.flags.action = true;
  messageList[4].messageInfo.flags.notice = true;
  messageList[4].messageInfo.flags.ignored = false;
  messageList[4].messageInfo.flags.deleted = false;
  messageList[4].messageInfo.flags.containsBits = true;
  messageList[4].messageInfo.timestamp = 44444;
  messageList[4].messageId = "messageId4";
  messageList[4].messageInfo.numBitsSent = 444444;
  messageList[4].messageInfo.tokens.push_back(std::make_unique<TextToken>("                     "));
  messageList[4].messageInfo.tokens.push_back(std::make_unique<EmoticonToken>("     \0    \n    ", "444"));
  messageList[4].messageInfo.tokens.push_back(std::make_unique<MentionToken>("      ", " adfadf   ", false));
  messageList[4].messageInfo.tokens.push_back(std::make_unique<UrlToken>("      adf a ", false));
  messageList[4].messageInfo.tokens.push_back(std::make_unique<BitsToken>("   EF QEF QEF ", 4444));
  messageList[4].messageInfo.badges.push_back({"1", "4"});
  messageList[4].messageInfo.badges.push_back({"2", "4"});
  messageList[4].messageInfo.badges.push_back({"3", "4"});
  messageList[4].messageInfo.badges.push_back({"4", "4"});
  messageList[4].messageInfo.badges.push_back({"    ", "     b"});
  messageList[4].messageInfo.badges.push_back({"cats are cool ? ", "\0"});
  messageList[4].messageInfo.messageType = "messageType4";
  messageList[4].messageInfo.messageTags.insert(
    // clang-format off
    {{"key40", "value40"},
     {"key41", "value41"},
     {"key42", "value42"},
     {"key43", "value43"},
     {"key44", "value44"}}
    // clang-format on
  );

  // Call the callback the that will use the data before the stack frees it
  func();
}

void SetupChatRoomInfo(ChatRoomInfo& roomInfo) {
  roomInfo.id = "room-id";
  roomInfo.rolePermissions.read = RoomRole::Everyone;
  roomInfo.rolePermissions.send = RoomRole::Everyone;
  roomInfo.name = "room-name";
  roomInfo.owner.userId = 22222;
  roomInfo.topic = "topic";
  roomInfo.view.isArchived = true;
  roomInfo.view.isMuted = false;
  roomInfo.view.isUnread = true;
  roomInfo.view.lastReadAt = 77777;
  roomInfo.view.unreadMentionCount = 44;
}
}  // namespace

JNIEXPORT void JNICALL Java_tv_twitch_test_ChatTest_Test_1IChatAPIListener(
  JNIEnv* jEnv, jobject /*jThis*/, jobject jModule, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  LoadAllUtilityJavaClassInfo(jEnv);
  LoadAllCoreJavaClassInfo(jEnv);
  LoadAllChatJavaClassInfo(jEnv);

  {
    auto listener = std::make_shared<JavaChatAPIListenerProxy>(jModule);
    listener->SetListener(jListener);

    listener->ModuleStateChanged(nullptr, IModule::State::Initialized, TTV_EC_SUCCESS);

    listener->ChatUserEmoticonSetsChanged(9001, {});
  }
}

JNIEXPORT void JNICALL Java_tv_twitch_test_ChatTest_Test_1IChatChannelListener(
  JNIEnv* jEnv, jobject /*jThis*/, jobject /*jModule*/, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  LoadAllUtilityJavaClassInfo(jEnv);
  LoadAllCoreJavaClassInfo(jEnv);
  LoadAllChatJavaClassInfo(jEnv);

  {
    auto listener = std::make_shared<JavaChatChannelListenerProxy>();
    listener->SetListener(jListener);

    listener->ChatChannelStateChanged(9001, 1001, ChatChannelState::Connected, TTV_EC_SUCCESS);

    {
      RestrictionReason reason;
      reason.subscribersOnly = true;

      ChatChannelInfo channelInfo;
      channelInfo.name = "mychannel";
      channelInfo.broadcasterLanguage = "EN";
      channelInfo.localUserRestriction = reason;

      listener->ChatChannelInfoChanged(9001, 1001, channelInfo);
    }

    {
      ChatChannelRestrictions restrictions;
      restrictions.slowMode = true;
      restrictions.slowModeDuration = 600;
      restrictions.r9k = true;
      listener->ChatChannelRestrictionsChanged(9001, 1001, restrictions);
    }

    {
      UserMode mode;
      mode.staff = true;
      mode.subscriber = true;

      ChatUserInfo userInfo;
      SetChatUserInfo(userInfo, "myusername", "mydisplayname", mode, 987654);

      listener->ChatChannelLocalUserChanged(9001, 1001, userInfo);
    }

    {
      UserMode banned;
      banned.banned = true;

      UserMode admin;
      admin.administrator = true;

      UserMode broadcaster;
      broadcaster.broadcaster = true;
    }

    {
      std::vector<LiveChatMessage> messageList;

      SetupChatLiveMessageList(
        messageList, [listener, &messageList]() { listener->ChatChannelMessagesReceived(9001, 1001, messageList); });
    }

    {
      std::vector<LiveChatMessage> messageList;
      SetupChatLiveMessageList(messageList, [listener, &messageList]() {
        SubscriptionNotice notice;
        notice.userMessage = std::make_unique<MessageInfo>(messageList[0].messageInfo);
        notice.systemMessage = "This is a system sub message.";
        notice.subCumulativeMonthCount = 5;
        notice.subStreakMonthCount = 2;
        notice.type = SubscriptionNotice::Type::Resub;
        notice.plan = SubscriptionNotice::Plan::Sub2000;
        notice.planDisplayName = "$9.99 sub!";
        notice.shouldShowSubStreak = true;

        listener->ChatChannelSubscriptionNoticeReceived(9001, 1001, notice);
      });

      FirstTimeChatterNotice firstTimeChatterNotice;
      firstTimeChatterNotice.userMessage.userName = "test_name";
      firstTimeChatterNotice.userMessage.displayName = "Test_DisplayName";
      firstTimeChatterNotice.userMessage.userId = 9001;
      firstTimeChatterNotice.userMessage.nameColorARGB = 123456;
      firstTimeChatterNotice.userMessage.timestamp = 98989898;
      firstTimeChatterNotice.userMessage.userName = "test_name";
      firstTimeChatterNotice.userMessage.tokens.push_back(std::make_unique<EmoticonToken>("HeyGuys", "30259"));

      firstTimeChatterNotice.systemMessage = "Somebody is new here!";
      firstTimeChatterNotice.messageId = "37feed0f-b9c7-4c3a-b475-21c6c6d21c3d";

      listener->ChatChannelFirstTimeChatterNoticeReceived(9001, 1001, firstTimeChatterNotice);

      RaidNotice raidNotice;
      raidNotice.systemMessage = "Raid system message!";
      raidNotice.profileImageUrl =
        "https://static-cdn.jtvnw.net/user-default-pictures/b83b1794-7df9-4878-916c-88c2ad2e4f9f-profile_image-70x70.jpg";
      raidNotice.viewerCount = 9001;
      raidNotice.raidingUserInfo.userId = 12345;
      raidNotice.raidingUserInfo.userName = "testuser";
      raidNotice.raidingUserInfo.displayName = "TestUser";

      listener->ChatChannelRaidNoticeReceived(9001, 1001, raidNotice);

      UnraidNotice unraidNotice;
      unraidNotice.systemMessage = "Unraid system message!";

      listener->ChatChannelUnraidNoticeReceived(9001, 1001, unraidNotice);

      GenericMessageNotice genericMessageNotice;
      genericMessageNotice.messageId = "messageId";
      genericMessageNotice.messageInfo.messageType = "messageType";
      genericMessageNotice.messageInfo.messageTags = {{"k1", "v1"}, {"k2", "v2"}};

      listener->ChatChannelGenericNoticeReceived(9001, 1001, genericMessageNotice);
    }

    listener->ChatChannelMessagesCleared(9001, 1001);
    listener->ChatChannelUserMessagesCleared(9001, 1001, 9002);
    listener->ChatChannelHostTargetChanged(9001, 1001, "targetChannel", 999);

    std::map<std::string, std::string> params;
    params["hi"] = "bye";
    params["yes"] = "no";
    listener->ChatChannelNoticeReceived(9001, 1001, "noticeId", params);

    {
      std::vector<LiveChatMessage> messageList;
      SetupChatLiveMessageList(messageList, [listener, &messageList]() {
        SubscriptionNotice notice;
        notice.userMessage = std::make_unique<MessageInfo>(messageList[0].messageInfo);
        notice.systemMessage = "test gifted a $4.99 sub to recipient!";
        notice.subCumulativeMonthCount = 1;
        notice.subStreakMonthCount = 1;
        notice.type = SubscriptionNotice::Type::SubGift;
        notice.plan = SubscriptionNotice::Plan::Sub1000;
        notice.planDisplayName = "Channel Subscription (test)";
        notice.messageId = "d5f235d0-1942-4d0f-a3cd-f5ec1f933c83";
        notice.recipient.userName = "recipient";
        notice.recipient.displayName = "Recipient";
        notice.recipient.userId = 1002;
        notice.senderCount = 1;
        notice.shouldShowSubStreak = true;

        listener->ChatChannelSubscriptionNoticeReceived(9001, 1001, notice);
      });
    }

    listener->AutoModCaughtSentMessage(9001, 1001);
    listener->AutoModApprovedSentMessage(9001, 1001);
    listener->AutoModDeniedSentMessage(9001, 1001);

    listener->AutoModCaughtMessageForMods(
      9002, 1001, "message-id", "this is the flagged message", 9001, "sender", "profanity");
    listener->AutoModMessageApprovedByMod(9002, 9001, "message-id", 9003, "otherMod");
    listener->AutoModMessageDeniedByMod(9002, 9001, "message-id", 9003, "otherMod");

    listener->AutoModDeniedSentCheer(9001, 1001);
    listener->AutoModTimedOutSentCheer(9001, 1001);

    listener->AutoModCaughtCheerForMods(
      9002, 1001, "message-id", "this is the flagged message", 9001, "sender", "profanity");

    ModerationActionInfo modActionInfo;
    modActionInfo.moderatorName = "moderator";
    modActionInfo.targetName = "otherUser";
    modActionInfo.moderatorId = 1002;
    modActionInfo.targetId = 1003;

    listener->ChatChannelModNoticeUserTimedOut(9001, 1001, ModerationActionInfo(modActionInfo), 60, "test");
    listener->ChatChannelModNoticeUserUntimedOut(9001, 1001, ModerationActionInfo(modActionInfo));
    listener->ChatChannelModNoticeUserBanned(9001, 1001, ModerationActionInfo(modActionInfo), "spam");
    listener->ChatChannelModNoticeUserUnbanned(9001, 1001, std::move(modActionInfo));

    listener->ChatChannelMessageDeleted(9001, 1001, "messageId", "sender", "content");
    listener->ChatChannelModNoticeMessageDeleted(
      9001, 1001, ModerationActionInfo(modActionInfo), "messageId", "this is a message");

    listener->ChatChannelModNoticeClearChat(9001, 1001, 1002, "modName");
    listener->ChatChannelModNoticeEmoteOnly(9001, 1001, 1002, "modName");
    listener->ChatChannelModNoticeEmoteOnlyOff(9001, 1001, 1002, "modName");
    listener->ChatChannelModNoticeFollowersOnly(9001, 1001, 1002, "modName", 30);
    listener->ChatChannelModNoticeFollowersOnlyOff(9001, 1001, 1002, "modName");
    listener->ChatChannelModNoticeR9K(9001, 1001, 1002, "modName");
    listener->ChatChannelModNoticeR9KOff(9001, 1001, 1002, "modName");
    listener->ChatChannelModNoticeSlow(9001, 1001, 1002, "modName", 60);
    listener->ChatChannelModNoticeSlowOff(9001, 1001, 1002, "modName");
    listener->ChatChannelModNoticeSubsOnly(9001, 1001, 1002, "modName");
    listener->ChatChannelModNoticeSubsOnlyOff(9001, 1001, 1002, "modName");
  }
}

JNIEXPORT void JNICALL Java_tv_twitch_test_ChatTest_Test_1IChatUserThreadsListener(
  JNIEnv* jEnv, jobject /*jThis*/, jobject /*jModule*/, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  LoadAllUtilityJavaClassInfo(jEnv);
  LoadAllCoreJavaClassInfo(jEnv);
  LoadAllChatJavaClassInfo(jEnv);

  {
    auto listener = std::make_shared<JavaChatUserThreadsListenerProxy>();
    listener->SetListener(jListener);

    {
      std::vector<WhisperMessage> messageList;

      SetupChatWhisperMessageList(messageList,
        [listener, &messageList]() { listener->ChatThreadRealtimeMessageReceived(9001, "threadId", messageList[1]); });
    }
  }
}

JNIEXPORT void JNICALL Java_tv_twitch_test_ChatTest_Test_1IChatRaidListener(
  JNIEnv* jEnv, jobject /*jThis*/, jobject /*jModule*/, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  LoadAllUtilityJavaClassInfo(jEnv);
  LoadAllCoreJavaClassInfo(jEnv);
  LoadAllChatJavaClassInfo(jEnv);

  {
    auto listener = std::make_shared<JavaChatRaidListenerProxy>();
    listener->SetListener(jListener);

    // All of these listener functions don't do anything unique with the RaidStatus object,
    // so we'll pass in the same one to all of them.
    RaidStatus status;
    status.raidId = "aabbccdd";
    status.creatorUserId = 12826;
    status.sourceChannelId = 2222;
    status.targetChannelId = 1234;
    status.targetUserLogin = "login_name";
    status.targetUserDisplayName = "display_name";
    status.targetUserProfileImageUrl = "www.profileimage.com";
    status.transitionJitterSeconds = 1;
    status.numUsersInRaid = 1;
    status.forceRaidNowSeconds = 90;
    status.joined = true;

    listener->RaidStarted(status);
    listener->RaidUpdated(status);
    listener->RaidFired(status);
    listener->RaidCancelled(status);
  }
}

JNIEXPORT void JNICALL Java_tv_twitch_test_ChatTest_Test_1IBitsListener(
  JNIEnv* jEnv, jobject /*jThis*/, jobject /*jModule*/, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  LoadAllUtilityJavaClassInfo(jEnv);
  LoadAllCoreJavaClassInfo(jEnv);
  LoadAllChatJavaClassInfo(jEnv);

  {
    auto listener = std::make_shared<JavaBitsListenerProxy>();
    listener->SetListener(jListener);

    BitsReceivedEvent event;
    event.channelName = "bits_receiver";
    event.context = "cheer";
    event.channelId = 12345;
    event.bitsUsed = 102;
    event.totalBitsUsed = 108;
    event.badge.isNewBadgeLevel = true;
    event.badge.previousLevel = 1;
    event.badge.newLevel = 100;
    event.message.userName = "guy_who_bits";
    event.message.userId = 12826;
    event.message.flags.containsBits = true;
    event.message.numBitsSent = 102;
    ttv::RFC3339TimeToUnixTimestamp("2017-08-11T18:53:12Z", event.message.timestamp);

    auto textToken = std::make_unique<TextToken>("Chee ");
    event.message.tokens.emplace_back(std::move(textToken));

    auto bitsToken = std::make_unique<BitsToken>("cheer", 1);
    event.message.tokens.emplace_back(std::move(bitsToken));

    listener->UserReceivedBits(event);

    BitsSentEvent sendEvent;
    sendEvent.channelId = 12826;
    sendEvent.channelBitsTotal = 108;
    sendEvent.userBitsBalance = 1000;

    listener->UserSentBits(sendEvent);

    listener->UserGainedBits(sendEvent.userBitsBalance);
  }
}

JNIEXPORT void JNICALL Java_tv_twitch_test_ChatTest_Test_1IFollowersListener(
  JNIEnv* jEnv, jobject /*jThis*/, jobject /*jModule*/, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  LoadAllUtilityJavaClassInfo(jEnv);
  LoadAllCoreJavaClassInfo(jEnv);
  LoadAllChatJavaClassInfo(jEnv);

  {
    auto listener = std::make_shared<JavaFollowersListenerProxy>();
    listener->SetListener(jListener);

    FollowerAddedEvent event;
    event.displayName = "test_display_name";
    event.userName = "test_user_name";
    event.userId = 157328038;

    listener->NewFollowerAdded(event);
  }
}

JNIEXPORT void JNICALL Java_tv_twitch_test_ChatTest_Test_1IFollowingListener(
  JNIEnv* jEnv, jobject /*jThis*/, jobject /*jModule*/, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  LoadAllUtilityJavaClassInfo(jEnv);
  LoadAllCoreJavaClassInfo(jEnv);
  LoadAllChatJavaClassInfo(jEnv);

  {
    auto listener = std::make_shared<JavaFollowingListenerProxy>();
    listener->SetListener(jListener);
    listener->FollowedChannel(1000, 1001);
    listener->UnfollowedChannel(1000, 1001);
  }
}

JNIEXPORT void JNICALL Java_tv_twitch_test_ChatTest_Test_1ISubscribersListener(
  JNIEnv* jEnv, jobject /*jThis*/, jobject jModule, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  LoadAllUtilityJavaClassInfo(jEnv);
  LoadAllCoreJavaClassInfo(jEnv);
  LoadAllChatJavaClassInfo(jEnv);

  {
    auto listener = std::make_shared<JavaSubscribersListenerProxy>();
    listener->SetListener(jListener);

    SubscriberAddedEvent event;
    event.userName = "guy_who_subs";
    event.displayName = "test_name";
    event.channelName = "subs_receiver";
    event.userId = 157328038;
    event.channelId = 12345;
    ttv::RFC3339TimeToUnixTimestamp("2017-08-11T18:53:12Z", event.timestamp);
    event.subNotice.type = SubscriptionNotice::Type::Resub;
    event.subNotice.plan = SubscriptionNotice::Plan::Sub1000;
    event.subNotice.subCumulativeMonthCount = 5;
    event.subNotice.subStreakMonthCount = 2;
    event.subNotice.planDisplayName = "QA Test Subscription";
    event.subNotice.systemMessage = "";
    event.subNotice.shouldShowSubStreak = true;

    auto message = std::make_unique<MessageInfo>();

    auto textToken = std::make_unique<TextToken>(" A Twitch baby is born! ");
    message->tokens.emplace_back(std::move(textToken));

    auto emoticonToken = std::make_unique<EmoticonToken>("KappaHD", "2867");
    message->tokens.emplace_back(std::move(emoticonToken));

    auto mentionToken = std::make_unique<MentionToken>("another_user", "@another_user", false);
    message->tokens.emplace_back(std::move(mentionToken));

    auto urlToken = std::make_unique<UrlToken>("http://twitch.tv", false);
    message->tokens.emplace_back(std::move(urlToken));

    message->displayName = event.displayName;
    message->userName = event.userName;
    message->timestamp = event.timestamp;
    message->userId = event.userId;
    message->numBitsSent = 0;

    event.subNotice.userMessage = std::move(message);

    listener->NewSubscriberAdded(event);
  }
}

JNIEXPORT void JNICALL Java_tv_twitch_test_ChatTest_Test_1IChatRoomNotificationsListener(
  JNIEnv* jEnv, jobject /*jThis*/, jobject /*jModule*/, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  LoadAllUtilityJavaClassInfo(jEnv);
  LoadAllCoreJavaClassInfo(jEnv);
  LoadAllChatJavaClassInfo(jEnv);

  {
    auto listener = std::make_shared<JavaIChatRoomNotificationsListenerProxy>();
    listener->SetListener(jListener);

    listener->UserTimedOut(1001, 2002, 3333);
    listener->UserBanned(1001, 2002);
    listener->UserUnbanned(1001, 2002);

    ChatRoomInfo roomInfo;
    SetupChatRoomInfo(roomInfo);

    ChatRoomMessage message;
    message.roomId = "my-room";
    message.roomMessageId = "message-id";
    message.messageInfo.userId = 1001;
    message.messageInfo.displayName = "display-name";

    listener->RoomViewUpdated(45454, 22222, "room-id", std::move(roomInfo.view));

    RoomMentionInfo mentionInfo;
    mentionInfo.roomOwnerId = 1001;
    mentionInfo.roomOwnerName = "me";
    mentionInfo.roomOwnerLogin = "meLogin";

    mentionInfo.senderId = 2002;
    mentionInfo.senderName = "sender";
    mentionInfo.roomId = "my-room";
    mentionInfo.roomName = "room-name";
    mentionInfo.messageId = "message-id";
    mentionInfo.sentAt = 3333;

    listener->RoomMentionReceived(1001, std::move(mentionInfo));
  }
}

JNIEXPORT void JNICALL Java_tv_twitch_test_ChatTest_Test_1IChatRoomListener(
  JNIEnv* jEnv, jobject /*jThis*/, jobject /*jModule*/, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  LoadAllUtilityJavaClassInfo(jEnv);
  LoadAllCoreJavaClassInfo(jEnv);
  LoadAllChatJavaClassInfo(jEnv);

  {
    auto listener = std::make_shared<JavaIChatRoomListenerProxy>();
    listener->SetListener(jListener);

    ChatRoomMessage message;
    message.roomId = "my-room";
    message.roomMessageId = "message-id";
    message.messageInfo.userId = 1001;
    message.messageInfo.displayName = "display-name";

    listener->MessageReceived("room-id", std::move(message));
    listener->MessageEdited("room-id", std::move(message));
    listener->MessageDeleted("room-id", std::move(message));

    ChatRoomInfo roomInfo;
    SetupChatRoomInfo(roomInfo);

    listener->RoomUpdated(std::move(roomInfo));
  }
}

JNIEXPORT void JNICALL Java_tv_twitch_test_ChatTest_Test_1IChatChannelPropertyListener(
  JNIEnv* jEnv, jobject /*jThis*/, jobject /*jModule*/, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  LoadAllUtilityJavaClassInfo(jEnv);
  LoadAllCoreJavaClassInfo(jEnv);
  LoadAllChatJavaClassInfo(jEnv);

  {
    auto listener = std::make_shared<JavaIChatChannelPropertyListenerProxy>();
    listener->SetListener(jListener);

    listener->RitualsEnabled(true);

    listener->OutgoingHostChanged(12827, 0, 12826, "Twitch", 10);
    listener->IncomingHostStarted(12826, 12827, "Twitch2", 10);
    listener->IncomingHostEnded(12826, 12827, "Twitch2");

    ExtensionMessage message;
    message.chatColor = 123456;
    message.extensionClientId = "client id";
    message.extensionDisplayName = "display name";
    message.extensionVersion = "1.0";
    message.messageId = "message id";
    message.sentAt = 54321;
    message.badges.push_back({"badge", "version"});
    message.badges.push_back({"badge2", "version2"});
    message.tokens.push_back(std::make_unique<TextToken>("yo"));
    message.tokens.push_back(std::make_unique<TextToken>("??"));

    listener->ExtensionMessageReceived(std::move(message));

    ChatChannelRestrictions restrictions;

    restrictions.followersDuration = 10;
    restrictions.slowModeDuration = 20;
    restrictions.emoteOnly = true;
    restrictions.verifiedOnly = true;
    restrictions.followersOnly = true;
    restrictions.subscribersOnly = true;
    restrictions.slowMode = true;
    restrictions.r9k = true;
    restrictions.slowModeSetAt = 30;

    listener->ChatChannelRestrictionsReceived(std::move(restrictions));
  }
}

JNIEXPORT void JNICALL Java_tv_twitch_test_ChatTest_Test_1IChatCommentListener(
  JNIEnv* jEnv, jobject /*jThis*/, jobject /*jModule*/, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  LoadAllUtilityJavaClassInfo(jEnv);
  LoadAllCoreJavaClassInfo(jEnv);
  LoadAllChatJavaClassInfo(jEnv);

  {
    auto listener = std::make_shared<JavaIChatCommentListenerProxy>();
    listener->SetListener(jListener);

    std::vector<ChatComment> comments;

    ChatComment comment;
    comment.channelId = 12827;
    comment.commentId = "commentId";
    comment.commentSource = ChatCommentSource::Comment;
    comment.contentId = "vodId";
    comment.messageInfo.userId = 1000;
    comment.messageInfo.displayName = "display-name";
    comment.moreReplies = false;
    comment.parentCommentId = "commentId2";
    comment.publishedState = ChatCommentPublishedState::PendingReview;
    comment.timestampMilliseconds = 12345;
    comment.updatedAt = 54321;

    comments.push_back(comment);

    ChatComment comment2;
    comment2.channelId = 12827;
    comment2.commentId = "commentId2";
    comment2.commentSource = ChatCommentSource::Chat;
    comment2.contentId = "vodId";
    comment2.messageInfo.userId = 1001;
    comment2.messageInfo.displayName = "display-name2";
    comment2.moreReplies = true;
    comment2.parentCommentId = "";
    comment2.publishedState = ChatCommentPublishedState::Published;
    comment2.replies = comments;
    comment2.timestampMilliseconds = 12345;
    comment2.updatedAt = 54321;

    comments.push_back(comment2);

    listener->ChatCommentManagerStateChanged(12826, "vodId", IChatCommentManager::PlayingState::Playing);
    listener->ChatCommentsReceived(12826, "vodId", std::move(comments));
    listener->ChatCommentsErrorReceived("This is a very bad error", TTV_EC_API_REQUEST_FAILED);
  }
}

JNIEXPORT void JNICALL Java_tv_twitch_test_ChatTest_Test_1ISubscriptionsNotificationsListener(
  JNIEnv* jEnv, jobject /*jThis*/, jobject /*jModule*/, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  LoadAllUtilityJavaClassInfo(jEnv);
  LoadAllCoreJavaClassInfo(jEnv);
  LoadAllChatJavaClassInfo(jEnv);

  {
    auto listener = std::make_shared<JavaISubscriptionsNotificationsListenerProxy>();
    listener->SetListener(jListener);

    listener->SubscribedToChannel(1234, 5678);
  }
}

JNIEXPORT void JNICALL Java_tv_twitch_test_ChatTest_Test_1ISquadNotificationsListener(
  JNIEnv* jEnv, jobject /*jThis*/, jobject /*jModule*/, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  LoadAllUtilityJavaClassInfo(jEnv);
  LoadAllCoreJavaClassInfo(jEnv);
  LoadAllChatJavaClassInfo(jEnv);

  {
    auto listener = std::make_shared<JavaISquadNotificationsListenerProxy>();
    listener->SetListener(jListener);

    SquadMember member;
    member.channelId = 12345;
    member.userLogin = "twitch";
    member.userDisplayName = "Twitch";
    member.profileImageUrl150 = "profileimageurl.jpg";

    SquadInfo squad;
    squad.ownerId = 12826;
    squad.squadId = "squadid";
    squad.status = SquadStatus::Live;
    squad.members.push_back(member);

    listener->SquadUpdated(std::move(squad));
    listener->SquadEnded();
  }
}

JNIEXPORT void JNICALL Java_tv_twitch_test_ChatTest_Test_1IMultiviewNotificationsListener(
  JNIEnv* jEnv, jobject /*jThis*/, jobject /*jModule*/, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  LoadAllUtilityJavaClassInfo(jEnv);
  LoadAllCoreJavaClassInfo(jEnv);
  LoadAllChatJavaClassInfo(jEnv);

  {
    auto listener = std::make_shared<JavaIMultiviewNotificationsListenerProxy>();
    listener->SetListener(jListener);

    Chanlet chanlet;
    chanlet.chanletId = 12345;

    MultiviewContentAttribute attribute;
    attribute.attributeId = "attribute id";
    attribute.key = "key";
    attribute.name = "name";
    attribute.ownerChannelId = 12345;
    attribute.parentId = "parent id";
    attribute.parentKey = "parent key";
    attribute.value = "value";
    attribute.valueShortName = "val";
    attribute.imageUrl = "image url";
    attribute.createdAt = 1;
    attribute.updatedAt = 2;

    chanlet.attributes.push_back(attribute);
    chanlet.attributes.push_back(attribute);

    listener->ChanletUpdated(1001, 12345, std::move(chanlet));
  }
}

JNIEXPORT jobject JNICALL Java_tv_twitch_test_ChatTest_Test_1JniChatWhisperMessage(JNIEnv* jni, jobject) {
  ttv::WhisperMessage message;

  message.messageId = 1;
  message.threadId = "thread_id";
  message.messageUuid = "message_uuid";

  return GetJavaInstance_ChatWhisperMessage(jni, message);
}
