/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#if defined(__clang__)
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wgnu-zero-variadic-macro-arguments"
#endif

#if __clang__
#define TTV_NORETURN __attribute__((analyzer_noreturn))
#else
#define TTV_NORETURN
#endif

#if __clang_analyzer__
// clang-format off
#define TTV_UNUSED_VARS(...)                                        \
  do {                                                              \
  _Pragma("clang diagnostic push")                                  \
  _Pragma("clang diagnostic ignored \"-Wunused-comparison\"")       \
  auto __ttv_assert_tmp_value = sizeof(decltype(0, ##__VA_ARGS__)); \
  (void)__ttv_assert_tmp_value;                                     \
  _Pragma("clang diagnostic pop")                                   \
} while(0)
// clang-format on
#else
#define TTV_UNUSED_VARS(...) ((void)sizeof(decltype(0, ##__VA_ARGS__)))
#endif

namespace ttv {
namespace assertion {
namespace details {
/**
 * Assert implementation details.  This should not be called directly and is expected to be used
 * through the |TTV_ASSERT| macro to allow the expression execution to be excluded
 * when asserts are disabled.
 */
void Assert(bool assertion, const char* expr, const char* component, const char* message, const char* file, int line);

void Assert(bool assertion, const char* expr, const char* file, int line);

}  // namespace details

/**
 * The definition for the assertion handling callback used by SetAssertHandler.  The assert
 * has already failed when this is called.
 *
 * component: The sdk componen the assert belongs to
 * expr: A text representation of the expression that failed
 * file: The file the assert occurred in
 * line: The line in the file the assert occurred on
 */
using AssertHandler = bool (*)(
  const char* expr, const char* component, const char* message, const char* file, int line);

/**
 * Sets the callback that allows the application to handle a failed assertion.
 * Returning true from this will aborb the assertion and allow the program to continue.
 * Returning false will pass the failed assert to the standard assertion function for
 * the platform.
 */
AssertHandler SetAssertHandler(AssertHandler assertHandler);

bool AssertFailed(
  const char* expr, const char* component, const char* message, const char* file, int line) TTV_NORETURN;

}  // namespace assertion
}  // namespace ttv

#if TTV_ENABLE_ASSERTS
#define TTV_ASSERT(expression, ...) \
  ttv::assertion::details::Assert((expression) ? true : false, #expression, ##__VA_ARGS__, __FILE__, __LINE__)
#else
#define TTV_ASSERT(...) TTV_UNUSED_VARS(__VA_ARGS__)
#endif

#if defined(__clang__)
#pragma clang diagnostic pop
#endif
