/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/types/coretypes.h"

#include <unordered_map>

#include <memory>

namespace ttv {
/**
 * A helper class which provides data caching. Designed to store std::shared_ptrs as value_types; when storing
 * primitives as value_types, be aware that ForEach iterates over a copy of the cache.
 */
template <typename KEY_TYPE, typename VALUE_TYPE>
class Cache {
 public:
  struct CacheEntry {
    CacheEntry() : expiryTime(0), lastUsedTime(0) {}

    KEY_TYPE key;
    uint64_t expiryTime;    //!< The time the data will expire.
    uint64_t lastUsedTime;  //!< The last time the data was used.
    VALUE_TYPE data;        //!< The cached data.
  };

  using VisitorFunc = std::function<void(CacheEntry& entry)>;

 protected:
  using MapType = std::unordered_map<KEY_TYPE, CacheEntry>;

 public:
  Cache() : mExpiryAge(1000 * 60 * 60) {}

  /**
   * Sets the amount of time after which entries will expire.  Modifying this while entries exist in the cache will not
   * affect existing entries.
   */
  void SetExpiryAge(uint64_t age) { mExpiryAge = age; }

  /**
   * Returns the number of entries in the cache.
   */
  uint32_t GetSize() const { return static_cast<uint32_t>(mCache.size()); }

  /**
   * Removes items from the cache that have not been used for the given amount of time.
   */
  void PurgeUnused(uint64_t olderThan) {
    auto now = GetSystemTimeMilliseconds();
    if (olderThan > now) {
      olderThan = 0;
    } else {
      olderThan = now - olderThan;
    }

    for (auto iter = mCache.begin(); iter != mCache.end();) {
      if ((*iter).second.lastUsedTime < olderThan) {
        iter = mCache.erase(iter);
      } else {
        ++iter;
      }
    }
  }

  /**
   * Removes items from the cache whose expiry time has elapsed.
   */
  void PurgeExpired() {
    auto now = GetSystemTimeMilliseconds();

    for (auto iter = mCache.begin(); iter != mCache.end();) {
      if ((*iter).second.expiryTime <= now) {
        iter = mCache.erase(iter);
      } else {
        ++iter;
      }
    }
  }

  /**
   * Clears the cache.
   */
  void Clear() { mCache.clear(); }

  /**
   * Marks the entry as recently used.
   * @return true if the entry was found, false otherwise.
   */
  bool MarkEntryUsed(const KEY_TYPE& key) {
    auto iter = mCache.find(key);
    if (iter == mCache.end()) {
      return false;
    }

    (*iter).second.lastUsedTime = GetSystemTimeMilliseconds();

    return true;
  }

  /**
   * Marks the entry as never being unused so it will never be purged.  This is reset by MarkEntryUsed().
   * @return true if the entry was found, false otherwise.
   */
  bool MarkEntryNeverUnused(const KEY_TYPE& key) {
    auto iter = mCache.find(key);
    if (iter == mCache.end()) {
      return false;
    }

    (*iter).second.lastUsedTime = std::numeric_limits<uint64_t>::max();

    return true;
  }

  /**
   * Marks the entry as never expiring.  This is reset by setting the entry again or manipulating the expiry time
   * with other methods.
   * @return true if the entry was found, false otherwise.
   */
  bool MarkEntryNeverExpires(const KEY_TYPE& key) {
    auto iter = mCache.find(key);
    if (iter == mCache.end()) {
      return false;
    }

    (*iter).second.expiryTime = std::numeric_limits<uint64_t>::max();

    return true;
  }

  /**
   * Explicitly sets the expiry time for a given cache item.
   * @return true if the entry was found, false otherwise.
   */
  bool SetEntryExpiryTime(const KEY_TYPE& key, uint64_t time) {
    auto iter = mCache.find(key);
    if (iter == mCache.end()) {
      return false;
    }

    (*iter).second.expiryTime = time;

    return true;
  }

  /**
   * Expire the entry immediately.
   * @return true if the entry was found, false otherwise.
   */
  bool ExpireEntry(const KEY_TYPE& key) { return SetEntryExpiryTime(key, 0); }

  /**
   * Expires all entries immediately.
   */
  void ExpireAll() {
    for (auto& kvp : mCache) {
      kvp.second.expiryTime = 0;
    }
  }

  /**
   * Caches the given value.
   */
  void SetEntry(const KEY_TYPE& key, VALUE_TYPE data) {
    uint64_t now = GetSystemTimeMilliseconds();

    CacheEntry entry;

    typename MapType::iterator iter = mCache.find(key);
    if (iter != mCache.end()) {
      entry = (*iter).second;
    } else {
      entry.key = key;
      entry.lastUsedTime = now;
    }

    entry.data = data;

    // Set the expiry time, handle overflow
    if (now <= std::numeric_limits<uint64_t>::max() - mExpiryAge) {
      entry.expiryTime = now + mExpiryAge;
    } else {
      entry.expiryTime = std::numeric_limits<uint64_t>::max();
    }

    mCache[key] = entry;
  }

  /**
   * Explicitly removes the given value from the cache.
   */
  void RemoveEntry(const KEY_TYPE& key) {
    auto iter = mCache.find(key);
    if (iter == mCache.end()) {
      return;
    }

    mCache.erase(iter);
  }

  /**
   * Retrieves the value from the cache.
   * @return true if entry was found, false otherwise.
   */
  bool GetEntry(const KEY_TYPE& key, VALUE_TYPE& result) {
    auto iter = mCache.find(key);
    if (iter != mCache.end()) {
      result = (*iter).second.data;
    }

    return iter != mCache.end();
  }

  /**
   * Retrieves the set of keys for cached values.
   */
  void GetKeys(std::vector<KEY_TYPE>& result) {
    for (auto kvp : mCache) {
      result.push_back(kvp.first);
    }
  }

  /**
   * Determines if any data is cached under the given key.
   */
  bool ContainsEntry(const KEY_TYPE& key) const { return mCache.find(key) != mCache.end(); }

  /**
   * Visits each entry in the cache.
   */
  void ForEach(VisitorFunc func) {
    auto copy = mCache;

    for (auto& kvp : copy) {
      func(kvp.second);
    }
  }

  bool HasExpiredEntries() const {
    Timestamp now = static_cast<Timestamp>(GetSystemTimeMilliseconds());
    for (const auto& kvp : mCache) {
      if (kvp.second.expiryTime <= now) {
        return true;
      }
    }

    return false;
  }

  /**
   * Visit each expired entry in the cache.
   */
  void ForEachExpired(VisitorFunc func) {
    auto now = GetSystemTimeMilliseconds();

    auto copy = mCache;

    for (auto& kvp : copy) {
      if (kvp.second.expiryTime <= now) {
        func(kvp.second);
      }
    }
  }

 protected:
  MapType mCache;       //!< The mapping of key to the data.
  uint64_t mExpiryAge;  //!< The amount of time after which entries expire.
};
}  // namespace ttv
