/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/cache.h"
#include "twitchsdk/core/component.h"
#include "twitchsdk/core/mutex.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"

#include <map>
#include <memory>

namespace ttv {
class TaskRunner;
class ChannelRepository;
class GetChannelTask;
struct ChannelInfo;
}  // namespace ttv

class ttv::ChannelRepository : public Component {
 public:
  using LookupCallback = std::function<void(TTV_ErrorCode ec, const ChannelInfo& channelInfo)>;

 public:
  ChannelRepository();

  /**
   * Returns cached channel info or fetches if needed.
   */
  TTV_ErrorCode FetchChannelInfo(ChannelId channelId, LookupCallback callback);
  /**
   * Returns cached channel info or TTV_EC_NOT_AVAILABLE if not yet fetched.
   */
  TTV_ErrorCode GetChannelInfo(ChannelId channelId, ChannelInfo& result);
  TTV_ErrorCode CacheChannelInfo(const ChannelInfo& channelInfo);

  // Component overrides
  virtual void Update() override;
  virtual TTV_ErrorCode Shutdown() override;
  virtual std::string GetLoggerName() const override;

 private:
  struct LookupEntry {
    ChannelId channelId;
    LookupCallback callback;
    std::shared_ptr<GetChannelTask> task;
  };

  // Component overrides
  virtual bool CheckShutdown() override;
  virtual void CompleteShutdown() override;

  TTV_ErrorCode ScheduleLookup();
  void CompleteLookup(const ChannelInfo& channelInfo, TTV_ErrorCode ec);

  std::unique_ptr<ttv::IMutex> mMutex;
  Cache<ChannelId, ChannelInfo> mChannelInfoCache;
  std::vector<LookupEntry> mPendingChannelInfoLookups;

  bool mLookupInProgress;
};
