/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"

namespace ttv {
class IChannelListener;
class ChannelListenerProxy;
}  // namespace ttv

class ttv::IChannelListener {
 public:
  virtual ~IChannelListener() = default;

  /**
   * Called when a stream is brought up.
   * The playDelay argument contains the artificial playback delay set on the stream.
   */
  virtual void StreamUp(uint32_t playDelaySeconds) = 0;
  /**
   * Called when a stream goes down.
   */
  virtual void StreamDown() = 0;
  /**
   * Called when the stream's viewer count changes.
   */
  virtual void StreamViewerCountChanged(uint32_t viewerCount) = 0;
  /**
   * Indicates that a midroll ad pod should be played.  The total length of the ad pod should be
   * at most duration seconds.
   */
  virtual void StreamTriggeredMidroll(uint32_t durationSeconds) = 0;
  /**
   * If the channel is hosting a watch party, the server will send us periodic updates
   * containing information about the watch party.
   */
  virtual void StreamReceivedWatchPartyUpdate(const WatchPartyUpdate& update) = 0;
  /**
   * Called when a new profile image has been successfully uploaded for the current channel (from any device).
   * Note: Only successful uploads will trigger this function.
   * @param[in] images Updated profile image information.
   */
  virtual void ProfileImageUpdated(const std::vector<ProfileImage>& images) = 0;
  /**
   * Called when the broadcaster updates their stream information.
   * @param[in] info The new information for the channel's stream.
   */
  virtual void StreamInfoUpdated(StreamInfoUpdate&& info) = 0;

  /**
   * Called when the squad the channel belongs to is updated or when the channel joins the squad.
   * @param[in] squad The updated squad info.
   */
  virtual void SquadUpdated(SquadInfo&& squad) = 0;

  /**
   * Called when the squad the channel leaves a squad.
   */
  virtual void SquadLeft() = 0;

  /**
   * Called when pixel tracking has been updated
   * @param[in] refresh required
   */
  virtual void PixelTrackingUpdate(bool refresh) = 0;
};

/**
 * A default implementation of IChannelListener that allows an app to plug in lambdas.
 */
class ttv::ChannelListenerProxy : public ttv::IChannelListener {
 public:
  using StreamUpCallback = std::function<void(uint32_t durationSeconds)>;
  using StreamDownCallback = std::function<void()>;
  using StreamViewerCountChangedCallback = std::function<void(uint32_t viewerCount)>;
  using StreamTriggeredMidrollCallback = std::function<void(uint32_t durationSeconds)>;
  using StreamReceivedWatchPartyUpdateCallback = std::function<void(const WatchPartyUpdate& update)>;
  using ProfileImageUpdatedCallback = std::function<void(const std::vector<ProfileImage>& images)>;
  using StreamInfoUpdatedCallback = std::function<void(StreamInfoUpdate&& info)>;
  using SquadUpdatedCallback = std::function<void(SquadInfo&& info)>;
  using SquadLeftCallback = std::function<void()>;
  using PixelTrackingUpdateCallback = std::function<void(bool refresh)>;

  std::function<void(uint32_t viewerCount)> mViewerCountChangedHandler;

  virtual void StreamUp(uint32_t playDelaySeconds) override {
    if (streamUpCallback != nullptr) {
      streamUpCallback(playDelaySeconds);
    }
  }

  virtual void StreamDown() override {
    if (streamDownCallback != nullptr) {
      streamDownCallback();
    }
  }

  virtual void StreamViewerCountChanged(uint32_t viewerCount) override {
    if (streamViewerCountChangedCallback != nullptr) {
      streamViewerCountChangedCallback(viewerCount);
    }
  }

  virtual void StreamTriggeredMidroll(uint32_t durationSeconds) override {
    if (streamTriggeredMidrollCallback != nullptr) {
      streamTriggeredMidrollCallback(durationSeconds);
    }
  }

  virtual void StreamReceivedWatchPartyUpdate(const WatchPartyUpdate& update) override {
    if (streamReceivedWatchPartyUpdateCallback != nullptr) {
      streamReceivedWatchPartyUpdateCallback(update);
    }
  }

  virtual void ProfileImageUpdated(const std::vector<ProfileImage>& images) override {
    if (profileImageUpdatedCallback != nullptr) {
      profileImageUpdatedCallback(images);
    }
  }

  virtual void StreamInfoUpdated(StreamInfoUpdate&& info) override {
    if (streamInfoUpdatedCallback != nullptr) {
      streamInfoUpdatedCallback(std::move(info));
    }
  }

  virtual void SquadUpdated(SquadInfo&& info) override {
    if (squadUpdatedCallback != nullptr) {
      squadUpdatedCallback(std::move(info));
    }
  }

  virtual void SquadLeft() override {
    if (squadLeftCallback != nullptr) {
      squadLeftCallback();
    }
  }

  virtual void PixelTrackingUpdate(bool refresh) override {
    if (pixelTrackingUpdateCallback != nullptr) {
      pixelTrackingUpdateCallback(refresh);
    }
  }

  StreamUpCallback streamUpCallback;
  StreamDownCallback streamDownCallback;
  StreamViewerCountChangedCallback streamViewerCountChangedCallback;
  StreamTriggeredMidrollCallback streamTriggeredMidrollCallback;
  StreamReceivedWatchPartyUpdateCallback streamReceivedWatchPartyUpdateCallback;
  ProfileImageUpdatedCallback profileImageUpdatedCallback;
  StreamInfoUpdatedCallback streamInfoUpdatedCallback;
  SquadUpdatedCallback squadUpdatedCallback;
  SquadLeftCallback squadLeftCallback;
  PixelTrackingUpdateCallback pixelTrackingUpdateCallback;
};
