/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

namespace ttv {
class IChannelStatus;
}

/**
 * The interface for receiving general upates on a channel.
 * Currently receives stream and profile image upload updates.
 */
class ttv::IChannelStatus {
 public:
  using UploadProfileImageCallback = std::function<void(TTV_ErrorCode ec, const std::vector<ProfileImage>& images)>;

 public:
  virtual ~IChannelStatus() = default;

  /**
   * This should be called when the application is done with the instance.
   */
  virtual TTV_ErrorCode Dispose() = 0;

  /**
   * Uploads a new profile image for the current user asynchronously.
   * Only one request can be in progress at a time.
   * This function will call HttpRequest::SendHttpRequest: the client implementation of this must be able
   * to send raw bytes (and not apply any encoding such as UTF-16).
   * @param[in] imageBuffer A buffer containing the image data to be uploaded.
   * @param[in] size The size of the image to be uploaded (in bytes).
   * @param[in] callback The callback to be called upon completion of the upload (whether successful or not)
   * The callback has several error codes that it can return:
   *   - TTV_EC_SUCCESS: We have successfully uploaded and updated the profile image.
   *   - TTV_EC_API_REQUEST_FAILED: The API requests to upload the profile image have failed.
   *   - TTV_EC_PROFILEIMAGE_IMAGE_VALIDATION_FAILED: The profile image was uploaded, but is not a valid image.
   *   - TTV_EC_PROFILEIMAGE_FORMAT_VALIDATION_FAILED: The profile image was uploaded, but is not in a valid image
   * format.
   *   - TTV_EC_PROFILEIMAGE_SIZE_VALIDATION_FAILED: The profile image was uploaded, but it is too big to be used (over
   * 5 MB). The callback will also return a vector of the profile image in different dimensions if the upload was
   * successful.
   *   - Each ProfileImage object in the vector contains the dimensions, url, and image format.
   * @return
   *   - TTV_EC_SUCCESS: We've initialized and started the profile image upload successfully.
   *   - TTV_EC_NOT_INITIALIZED: We haven't initialized properly.
   *   - TTV_EC_NEED_TO_LOGIN: We aren't logged in properly.
   *   - TTV_EC_INVALID_ARG: If any input arguments are invalid.
   *   - TTV_EC_REQUEST_PENDING: If a request is already pending.
   *   - TTV_EC_API_REQUEST_FAILED: The API requests to upload the profile image have failed.
   */
  virtual TTV_ErrorCode UploadProfileImage(
    const char* imageBuffer, size_t imageSize, const UploadProfileImageCallback& callback) = 0;
};
