/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/coreutilities.h"
#include "twitchsdk/core/eventscheduler.h"
#include "twitchsdk/core/tracer.h"
#include "twitchsdk/core/user/userlistener.h"

#include <map>

namespace ttv {
class IComponent;
class Component;
class UserComponent;
class ComponentContainer;
class TaskRunner;
class Task;
class User;
class IMutex;
class IEventScheduler;
}  // namespace ttv

class ttv::IComponent {
 public:
  enum class State {
    Uninitialized,
    Initialized,
    ShuttingDown,
    Inert,  //!< The component doesn't require shutting down.
  };

  virtual ~IComponent() = default;

  virtual State GetState() const = 0;

  virtual TTV_ErrorCode Initialize() = 0;
  virtual void Update() = 0;
  virtual TTV_ErrorCode Shutdown() = 0;
};

class ttv::Component : public IComponent {
 protected:
  Component();

 public:
  virtual ~Component() override;

  virtual std::string GetLoggerName() const = 0;

  virtual void SetTaskRunner(std::shared_ptr<TaskRunner> taskRunner);
  std::shared_ptr<TaskRunner> GetTaskRunner() { return mTaskRunner; }
  virtual void SetMainEventScheduler(const std::shared_ptr<IEventScheduler>& eventScheduler);
  std::shared_ptr<IEventScheduler> GetMainEventScheduler() const { return mMainEventScheduler; }

  // IComponent implementation
  virtual State GetState() const override;
  virtual TTV_ErrorCode Initialize() override;
  virtual void Update() override;
  virtual TTV_ErrorCode Shutdown() override;

 protected:
  TTV_ErrorCode StartTask(std::shared_ptr<Task> task);
  void CompleteTask(Task* task);
  bool IsTaskRunning(Task* task) const;
  virtual void SetState(State state);
  virtual void SetClientState(State state);
  virtual void SetServerState(State state);
  virtual bool CheckShutdown();
  virtual void CompleteShutdown();

  ClientServerValue<State> mState;
  std::shared_ptr<TaskRunner> mTaskRunner;
  std::shared_ptr<IEventScheduler> mMainEventScheduler;
  std::vector<std::shared_ptr<Task>> mRunningTasks;

  mutable std::unique_ptr<IMutex> mTaskMutex;
  uint64_t mShutdownTimeMilliseconds;
};

class ttv::UserComponent : public ttv::Component {
 public:
  UserComponent(const std::shared_ptr<User>& user);

  virtual TTV_ErrorCode Initialize() override;
  std::shared_ptr<User> GetUser();

 protected:
  class UserListener : public IUserListener {
   public:
    UserListener(UserComponent* owner);

    // IUserListener implementation
    virtual void OnUserLogInComplete(User* source, TTV_ErrorCode ec) override;
    virtual void OnUserLogOutComplete(User* source, TTV_ErrorCode ec) override;
    virtual void OnUserInfoFetchComplete(User* source, TTV_ErrorCode ec) override;
    virtual void OnUserAuthenticationIssue(
      User* source, std::shared_ptr<const OAuthToken> oauthToken, TTV_ErrorCode ec) override;

   private:
    UserComponent* mOwner;
  };

  // IUserListener implementation
  virtual void OnUserLogInComplete(TTV_ErrorCode ec);
  virtual void OnUserLogOutComplete(TTV_ErrorCode ec);
  virtual void OnUserInfoFetchComplete(TTV_ErrorCode ec);
  virtual void OnUserAuthenticationIssue(std::shared_ptr<const OAuthToken> oauthToken, TTV_ErrorCode ec);

  virtual void CompleteShutdown() override;
  void Log(MessageLevel level, const char* format, ...);

  std::weak_ptr<User> mUser;
  bool
    mOAuthIssue;  //!< Whether or not the internal updates of the component are on hold because of an issue with the OAuth token.

 private:
  std::shared_ptr<UserListener> mUserListener;
};

class ttv::ComponentContainer : public Component {
 public:
  ComponentContainer();
  virtual ~ComponentContainer() override;

  // Component overrides
  virtual TTV_ErrorCode Initialize() override;
  virtual void Update() override;
  virtual TTV_ErrorCode Shutdown() override;
  virtual std::string GetLoggerName() const override;

  virtual TTV_ErrorCode AddComponent(std::shared_ptr<IComponent> component);
  virtual TTV_ErrorCode RemoveComponent(std::shared_ptr<IComponent> component);
  virtual TTV_ErrorCode DisposeComponent(std::shared_ptr<IComponent> component);

  virtual TTV_ErrorCode SetComponent(const std::string& name, std::shared_ptr<IComponent> component);
  virtual TTV_ErrorCode GetComponent(const std::string& name, std::shared_ptr<IComponent>& result);
  virtual TTV_ErrorCode RemoveComponent(const std::string& name);
  virtual TTV_ErrorCode DisposeComponent(const std::string& name);

  /**
   * Determines if the given component is stored in the container.
   */
  bool ContainsComponent(std::shared_ptr<IComponent> component) const;

  template <typename T>
  std::shared_ptr<T> GetComponentAs(const std::string& name) {
    std::shared_ptr<T> result;
    std::shared_ptr<IComponent> component;
    TTV_ErrorCode ec = GetComponent(name, component);

    if (TTV_SUCCEEDED(ec) && component != nullptr) {
      result = std::static_pointer_cast<T>(component);
    }

    return result;
  }

  template <typename T>
  std::shared_ptr<T> GetComponent() {
    std::shared_ptr<T> result;
    std::shared_ptr<IComponent> component;
    TTV_ErrorCode ec = GetComponent(T::GetComponentName(), component);

    if (TTV_SUCCEEDED(ec) && component != nullptr) {
      result = std::static_pointer_cast<T>(component);
    }

    return result;
  }

 protected:
  // Component overrides
  virtual bool CheckShutdown() override;
  virtual void CompleteShutdown() override;

  mutable std::unique_ptr<IMutex> mMutex;
  std::vector<std::shared_ptr<IComponent>> mComponents;
  std::map<std::string, std::shared_ptr<IComponent>> mComponentMap;

  std::vector<std::shared_ptr<IComponent>> mDisposeComponents;
};
