/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/assertion.h"
#include "twitchsdk/core/mutex.h"

#include <atomic>
#include <queue>

namespace ttv {
template <typename TYPE>
class ConcurrentQueue;
}

/**
 * An implementation of a queue which allows concurrent pushes and pops.
 */
template <typename TYPE>
class ttv::ConcurrentQueue {
 public:
  /**
   * Constructs a queue instance.  The required mutex will be created using
   * the mutex factory.
   */
  ConcurrentQueue() : mQueueSize(0) {
#ifdef WIN32
#pragma push_macro("CreateMutex")
#undef CreateMutex
    TTV_ErrorCode ec = ttv::CreateMutex(mMutex, "ConcurrentQueue");
#pragma pop_macro("CreateMutex")
#else
    TTV_ErrorCode ec = ttv::CreateMutex(mMutex, "ConcurrentQueue");
#endif
    TTV_ASSERT(TTV_SUCCEEDED(ec) && mMutex != nullptr);
  }

  /**
   * Constructs a queue instance with a preallocated mutex.  This allows the queue
   * to be used before the mutex factory may be set.
   */
  ConcurrentQueue(std::unique_ptr<IMutex>&& mutex) : mMutex(std::move(mutex)) { TTV_ASSERT(mMutex != nullptr); }

 public:
  void clear() {
    AutoMutex lock(mMutex.get());
    while (!mQueue.empty()) {
      mQueue.pop();
    }
    mQueueSize.store(mQueue.size(), std::memory_order_release);
  }

  bool empty() const {
    AutoMutex lock(mMutex.get());
    return mQueue.empty();
  }

  /**
   * Pushes the item onto the back of the queue in a thread-safe manner.
   */
  void push(const TYPE& item) {
    AutoMutex lock(mMutex.get());
    mQueue.emplace(item);
    mQueueSize.store(mQueue.size(), std::memory_order_release);
  }

  /**
   * Pushes the item onto the back of the queue in a thread-safe manner.
   */
  void push(TYPE&& item) {
    AutoMutex lock(mMutex.get());
    mQueue.emplace(std::forward<TYPE>(item));
    mQueueSize.store(mQueue.size(), std::memory_order_release);
  }

  /**
   * Attempts to pop from the front of the queue in a thread-safe manner.  If an item is available then
   * the result will be set to the value and true will be returned.  Otherwise, false will be returned.
   *
   * NOTE: Be careful if ITEM doesn't support move assignment since it may call the assignment operator
   * and destructor of the value while under the lock.  If this is the case, be sure that
   * the assignment operator and destructor will not in turn call back into the ConcurrentQueue
   * instance or a deadlock may occur.
   */
  bool try_pop(TYPE& result) {
    // NOTE: Reset the value before entering the lock to avoid resources potentially being
    // cleaned up while under the lock during assignment
    result = {};

    bool ret = false;

    AutoMutex lock(mMutex.get());
    if (!mQueue.empty()) {
      result = std::move(mQueue.front());
      mQueue.pop();
      ret = true;
    }

    mQueueSize.store(mQueue.size(), std::memory_order_release);
    return ret;
  }

  /**
   * Retrieves the size of the queue.  This is thread-safe and will get the current size at the time of reading,
   * however, the size can update immediately after being called by other threads so this should be used to get an
   * approximate size.
   */
  size_t unsafe_size() const { return mQueueSize.load(std::memory_order_consume); }

 private:
  std::queue<TYPE> mQueue;
  std::unique_ptr<IMutex> mMutex;
  std::atomic_size_t mQueueSize;
};
