/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/corelistener.h"
#include "twitchsdk/core/eventscheduler.h"
#include "twitchsdk/core/eventsource.h"
#include "twitchsdk/core/module.h"
#include "twitchsdk/core/pollingeventscheduler.h"
#include "twitchsdk/core/pubsub/pubsubclient.h"
#include "twitchsdk/core/trackingcontext.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"
#include "twitchsdk/core/user/userlistener.h"

#include <memory>

namespace ttv {
class CoreAPI;
class IChannelListener;
class ICoreApiClient;
class ICoreAPIListener;
class ChannelRepository;
class IChannelStatus;
class IDashboardActivityListener;
class IDashboardActivityStatus;
class IGenericSubscriberListener;
class IGenericSubscriberStatus;
class SettingRepository;
class TaskRunner;
class User;
class UserRepository;
struct CoreAPIInternalData;

/**
 * Sets the Twitch client-id to be used in the system.  This must be set before initializing any modules and must not be
 * set again until all modules are fully shut down.
 */
TTV_ErrorCode SetClientId(const std::string& clientId);
/**
 * Retrieves the currently set client-id.
 */
TTV_ErrorCode GetClientId(std::string& clientId);
}  // namespace ttv

/**
 * An event interface for modules that are dependent on CoreAPI.
 */
class ttv::ICoreApiClient {
 public:
  ICoreApiClient();
  virtual ~ICoreApiClient() = default;

  virtual std::string GetClientName() = 0;

  virtual void GetRequiredOAuthScopes(std::vector<std::string>& scopes);

  virtual void CoreUserLoggedIn(std::shared_ptr<User> user);
  virtual void CoreUserLoggedOut(std::shared_ptr<User> user);
};

/**
 * Provides the core services need by other subsystems.
 *
 * The following properties must be set before calling Initialize().
 * - SetListener()
 */
class ttv::CoreAPI : public ModuleBase {
 public:
  using FetchUserInfoCallback = std::function<void(const ErrorDetails& errorDetails, const UserInfo& userInfo)>;
  using FetchChannelInfoCallback = std::function<void(TTV_ErrorCode ec, const ChannelInfo& channelInfo)>;
  using FetchStreamInfoCallback = std::function<void(TTV_ErrorCode ec, const StreamInfo& streamInfo)>;
  using LogInCallback = std::function<void(const ErrorDetails& errorDetails, const UserInfo& userInfo)>;
  using LogOutCallback = std::function<void(TTV_ErrorCode ec)>;
  using UploadProfileImageCallback = std::function<void(TTV_ErrorCode ec, const std::vector<ProfileImage>& images)>;

 public:
  CoreAPI();

  // IModule implementation
  virtual std::string GetModuleName() const override;
  virtual TTV_ErrorCode Initialize(const InitializeCallback& callback) override;
  virtual TTV_ErrorCode Shutdown(const ShutdownCallback& callback) override;
  virtual TTV_ErrorCode Update() override;

  TTV_ErrorCode SetListener(std::shared_ptr<ICoreAPIListener> listener);

  std::shared_ptr<UserRepository> GetUserRepository() const { return mUserRepository; }
  std::shared_ptr<ChannelRepository> GetChannelRepository() const { return mChannelRepository; }
  std::shared_ptr<SettingRepository> GetSettingRepository() const { return mSettingRepository; }
  std::shared_ptr<TrackingContext> GetTrackingContext() const { return mTrackingContext; }
  std::shared_ptr<IEventScheduler> GetMainEventScheduler() const { return mMainEventScheduler; }

  /**
   * Retrieves the currently set client-id.
   */
  TTV_ErrorCode GetClientId(std::string& clientId);
  /**
   * Sets the language used by the local client.  This is a 2 language character code.
   */
  TTV_ErrorCode SetLocalLanguage(const std::string& language);
  /**
   * Retrieves the currently set local language.  This is a 2 language character code.
   */
  TTV_ErrorCode GetLocalLanguage(std::string& language);
  /**
   * Asynchonously fetches the user information for the user with a given id. The result of the fetch will be returned
   * in the callback.
   */
  TTV_ErrorCode FetchUserInfoById(UserId userId, FetchUserInfoCallback callback);
  /**
   * Asynchronously fetches the user information for the named user.  The result of the fetch will be returned in the
   * callback.
   */
  TTV_ErrorCode FetchUserInfoByName(const std::string& userName, FetchUserInfoCallback callback);
  /**
   * Asynchonously fetches the channel information for the given channel id.
   */
  TTV_ErrorCode FetchChannelInfoById(ChannelId channelId, const FetchChannelInfoCallback& callback);
  /**
   * Asynchonously fetches the channel information for the named channel.
   */
  TTV_ErrorCode FetchChannelInfoByName(const std::string& channelName, const FetchChannelInfoCallback& callback);
  /**
   * Asynchonously fetches the stream information for the given channel id.
   * @param[in] channelId The id of the channel to fetch stream information for.
   * @return:
   *   - TTV_EC_SUCCESS if we've successfully fetched the stream info.
   *   - TTV_EC_WEBAPI_RESULT_NO_STREAMINFO if we've successfully fetched, but there is no stream info available (i.e.
   * the stream is down).
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON if we were unable to process the fetch response.
   */
  TTV_ErrorCode FetchStreamInfoById(ChannelId channelId, const FetchStreamInfoCallback& callback);
  /**
   * Asynchonously fetches the stream information for the given channel name.
   * @param[in] channelName The name of the channel to fetch stream information for.
   * @return:
   *   - TTV_EC_SUCCESS if we've successfully fetched the stream info.
   *   - TTV_EC_WEBAPI_RESULT_NO_STREAMINFO if we've successfully fetched, but there is no stream info available (i.e.
   * the stream is down).
   *   - TTV_EC_WEBAPI_RESULT_INVALID_JSON if we were unable to process the fetch response.
   */
  TTV_ErrorCode FetchStreamInfoByName(const std::string& channelName, const FetchStreamInfoCallback& callback);
  /**
   * An IChannelListener object can be passed in to receive updates for the channel.
   * @param[in] userId The user id to create the channel listener for.  Can be 0 if for not specific user.
   * @param[in] channelId The channel id to listen on.  Cannot be 0.
   * @param[in] listener The listener to use for callbacks. Can be null if listener is not needed.
   * @param[out] result The resulting IChannelStatus implementation for the given user/channel to receive updates on.
   * @return:
   *   - TTV_EC_SUCCESS if we've initialized and created FollowersStatus successfully.
   *   - TTV_EC_NOT_INITIALIZED if we haven't initialized properly.
   *   - TTV_EC_NEED_TO_LOGIN if we aren't logged in properly.
   *   - TTV_EC_INVALID_ARG: If any input arguments are invalid.
   */
  TTV_ErrorCode CreateChannelStatus(UserId userId, ChannelId channelId,
    const std::shared_ptr<IChannelListener>& listener, std::shared_ptr<IChannelStatus>& result);

  /**
   * An IDashboardActivityListener object can be passed in to receive updates for the channel.
   * @param[in] userId The user id to listen for dashboard activities, cannot be 0.
   * @param[in] channelId The channel id to listen on.  Cannot be 0.
   * @param[in] listener The listener to use for callbacks.
   * @param[out] result The resulting IDashboardActivityStatus implementation for the given user/channel to receive
   * updates on.
   * @return:
   *   - TTV_EC_SUCCESS if we've initialized and created the Status successfully.
   *   - TTV_EC_NOT_INITIALIZED if we haven't initialized properly.
   *   - TTV_EC_NEED_TO_LOGIN if we aren't logged in properly.
   *   - TTV_EC_INVALID_ARG: If any input arguments are invalid.
   */
  TTV_ErrorCode CreateDashboardActivityStatus(UserId userId, ChannelId channelId,
    const std::shared_ptr<IDashboardActivityListener>& listener, std::shared_ptr<IDashboardActivityStatus>& result);

  /**
   * An IGenericSubscriberListener object can be passed in to receive updates for a pubsub topic
   * @param[in] userId The user id to create the pubsub topic listener for.
   * @param[in] subscriberTopic The topic to subscriber for pubsub events.
   * @param[in] listener The listener to use for callbacks.
   * @param[out] result The resulting IGenericSubscriberStatus implementation to receive updates on.
   * @return:
   *   - TTV_EC_SUCCESS if we've initialized and created the Status successfully.
   *   - TTV_EC_NOT_INITIALIZED if we haven't initialized properly.
   *   - TTV_EC_NEED_TO_LOGIN if we aren't logged in properly.
   *   - TTV_EC_INVALID_ARG: If any input arguments are invalid.
   */
  TTV_ErrorCode CreateGenericSubscriberStatus(UserId userId, const std::string& subscriberTopic,
    const std::shared_ptr<IGenericSubscriberListener>& listener, std::shared_ptr<IGenericSubscriberStatus>& result);

  /**
   * Retrieves the required scopes from all registered modules.
   */
  TTV_ErrorCode GetRequiredOAuthScopes(std::vector<std::string>& modules, std::vector<std::string>& scopes);

  TTV_ErrorCode LogIn(const std::string& oauthToken, LogInCallback callback);
  TTV_ErrorCode LogOut(UserId userId, LogOutCallback callback);

  TTV_ErrorCode ConnectPubSub(UserId userId);
  TTV_ErrorCode DisconnectPubSub(UserId userId);

  // Reference counting so that CoreAPI isn't shutdown while other systems are using it
  TTV_ErrorCode RegisterClient(std::shared_ptr<ICoreApiClient> client);
  TTV_ErrorCode UnregisterClient(std::shared_ptr<ICoreApiClient> client);

  TTV_ErrorCode SetGlobalSetting(const std::string& key, const std::string& value);
  TTV_ErrorCode RemoveGlobalSetting(const std::string& key);
  TTV_ErrorCode GetGlobalSetting(const std::string& key, std::string& value);

  // This will get all subscribed topics from all users
  TTV_ErrorCode GetSubscribedPubsubTopics(std::vector<std::string>& subscribedTopics);

  /**
   * Crashes with an abort to demonstrate a crash in native code
   * This is helpful for testing platform-specific crash reporters.
   */
  TTV_ErrorCode CrashAbort();

 private:
  class PubSubListener : public PubSubClient::IListener {
   public:
    PubSubListener(CoreAPI* api);

    // PubSubClient::Listener implementation
    virtual void OnStateChanged(PubSubClient* source, PubSubState state, TTV_ErrorCode ec) override;

   private:
    CoreAPI* mOwner;
  };

  class UserListener : public IUserListener {
   public:
    UserListener(CoreAPI* owner);

    // IUserListener implementation
    virtual void OnUserLogInComplete(User* source, TTV_ErrorCode ec) override;
    virtual void OnUserLogOutComplete(User* source, TTV_ErrorCode ec) override;
    virtual void OnUserInfoFetchComplete(User* source, TTV_ErrorCode ec) override;
    virtual void OnUserAuthenticationIssue(
      User* source, std::shared_ptr<const OAuthToken> oauthToken, TTV_ErrorCode ec) override;

   private:
    CoreAPI* mOwner;
  };

  enum class EventSchedulerState { Running, ShuttingDown, ShutDown };

  // PubSubClient::Listener implementation
  void OnPubSubStateChanged(PubSubClient* source, PubSubState state, TTV_ErrorCode ec);

  // IUserListener implementation
  void OnUserLogInComplete(User* source, TTV_ErrorCode ec);
  void OnUserLogOutComplete(User* source, TTV_ErrorCode ec);
  void OnUserInfoFetchComplete(User* source, TTV_ErrorCode ec);
  void OnUserAuthenticationIssue(User* source, std::shared_ptr<const OAuthToken> oauthToken, TTV_ErrorCode ec);

  // ModuleBase overrides
  virtual bool CheckShutdown() override;
  virtual void CompleteShutdown() override;

  static TTV_ErrorCode DisposeChannelStatus(
    const std::shared_ptr<IChannelStatus>& channelStatus, const std::shared_ptr<CoreAPIInternalData>& internalData);
  static TTV_ErrorCode DisposeDashboardActivityStatus(
    const std::shared_ptr<IDashboardActivityStatus>& status, const std::shared_ptr<CoreAPIInternalData>& internalData);
  static TTV_ErrorCode DisposeGenericSubscriberStatus(
    const std::shared_ptr<IGenericSubscriberStatus>& status, const std::shared_ptr<CoreAPIInternalData>& internalData);

  void InitializeAnonymousUser();

  std::shared_ptr<ChannelRepository> mChannelRepository;
  std::shared_ptr<UserRepository> mUserRepository;
  std::shared_ptr<TrackingContext> mTrackingContext;
  std::shared_ptr<UserListener> mUserListener;
  std::shared_ptr<PubSubListener> mPubSubListener;
  std::shared_ptr<TaskRunner> mTaskRunner;
  std::shared_ptr<SettingRepository> mSettingRepository;
  std::shared_ptr<CoreAPIInternalData> mInternalData;
  std::shared_ptr<IEventScheduler> mMainEventScheduler;
  std::shared_ptr<PollingEventScheduler> mPollingEventScheduler;

  std::vector<std::shared_ptr<ICoreApiClient>> mApiClients;
  std::string mLocalLanguage;
};
