/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include <algorithm>
#include <memory>
#include <vector>

namespace ttv {
// <T> in these is intended to be a bitfield.
template <typename T>
void AddFlag(T& x, T flag) {
  x = static_cast<T>(x | flag);
}

template <typename T>
void RemoveFlag(T& x, T flag) {
  x = static_cast<T>(x & (~flag));
}

template <typename T>
void SetFlag(T& x, T flag, bool value) {
  if (value) {
    ttv::AddFlag(x, flag);
  } else {
    ttv::RemoveFlag(x, flag);
  }
}

template <typename T, typename F>
bool HasFlag(T& x, F flag) {
  return static_cast<uint32_t>(static_cast<uint32_t>(x) & static_cast<uint32_t>(flag)) == static_cast<uint32_t>(flag);
}

template <typename Base, typename Derived>
class Cloneable : public Base {
  virtual std::unique_ptr<Base> Clone() const override {
    return std::make_unique<Derived>(*static_cast<const Derived*>(this));
  }
};

template <typename FunctionType>
class CallbackQueue {
 public:
  using LookupId = uint64_t;

 public:
  template <typename FunctionRef>
  void Push(FunctionRef&& callback) {
    Push(std::forward<FunctionRef>(callback), 0);
  }

  template <typename FunctionRef>
  void Push(FunctionRef&& callback, LookupId lookupId) {
    if (callback != nullptr) {
      LookupEntry entry;
      entry.callback = std::forward<FunctionRef>(callback);
      entry.lookupId = lookupId;

      mLookups.push_back(std::move(entry));
    }
  }

  template <typename... Args>
  void Flush(Args&&... args) {
    for (const auto& lookup : mLookups) {
      lookup.callback(std::forward<Args>(args)...);
    }
    mLookups.clear();
  }

  FunctionType Erase(LookupId token) {
    auto iter =
      std::find_if(mLookups.begin(), mLookups.end(), [token](const LookupEntry& x) { return x.lookupId == token; });

    if (iter != mLookups.end()) {
      FunctionType callback = std::move(iter->callback);

      mLookups.erase(iter);
      return std::move(callback);
    }

    return nullptr;
  }

  bool Empty() const { return mLookups.empty(); }

 private:
  struct LookupEntry {
    FunctionType callback;
    LookupId lookupId;
  };

  std::vector<LookupEntry> mLookups;
};

/**
 * Provides double-buffering of a value that is typically needed between 2 threads.
 * It can be used as a single value via overloaded operators and in this case
 * it will return the client value.
 */
template <typename T>
struct ClientServerValue {
  T client;
  T server;

  ClientServerValue() {}

  ClientServerValue(const T& val) {
    client = val;
    server = val;
  }

  operator T() { return client; }

  void operator=(const T& val) {
    client = val;
    server = val;
  }

  bool operator==(const T& other) { return client == other; }

  bool operator!=(const T& other) { return client != other; }
};

template <typename Arg>
struct MakeVoid {
  using Type = void;
};

template <typename Arg>
using VoidType = typename MakeVoid<Arg>::Type;
}  // namespace ttv
