/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

// A few things were accidentally left out of the C++11 standard, but should have been there:
// std::make_unique
// std::static_pointer_cast for unique_ptrs
// std::dynamic_pointer_cast for unique_ptrs
// std::const_pointer_cast for unique_ptrs

// TODO: This should also have std::allocate_unique

#include <type_traits>

#include <memory>

namespace std {
template <typename DesiredType, typename CurrentType>
inline unique_ptr<DesiredType> static_pointer_cast(unique_ptr<CurrentType>&& currentPointer) {
  return unique_ptr<DesiredType>(static_cast<DesiredType*>(currentPointer.release()));
}

template <typename DesiredType, typename CurrentType>
inline unique_ptr<DesiredType> dynamic_pointer_cast(unique_ptr<CurrentType>&& currentPointer) {
  return unique_ptr<DesiredType>(dynamic_cast<DesiredType*>(currentPointer.release()));
}

template <typename DesiredType, typename CurrentType>
inline unique_ptr<DesiredType> const_pointer_cast(unique_ptr<CurrentType>&& currentPointer) {
  return unique_ptr<DesiredType>(const_cast<DesiredType*>(currentPointer.release()));
}

}  // namespace std

// It is intended to only be used in the preprocessing step, before const variables are considered.

#define VC2012 170050727
#define VC2012_NOVEMBER_CTP 170051025
#define VC2012_UPDATE1 170051106
#define VC2012_UPDATE2 170060315

#if defined(_MSC_VER)
// VC does not support emplace_back until VC2012
// (VC10 has an emplace_back, but it only takes one parameter and doesn't actually do in-place construction. So for all
// intents and purposes, it is useless)
#if _MSC_FULL_VER >= VC2012
#define TTVSDK_EMPLACE_BACK_SUPPORTED

// VC does not support variadic templates until VC2012 Nov CTP
// However, VC2012 Update 1 and 2 (which came later) do not support it.
// So only doing a greater-than-or-equals on the CTP version number will not work.
// I spoke with Stephan T. Lavavej at Microsoft and he said I can do equality to the CTP
// and also the next major release of VC.
#if _MSC_FULL_VER == VC2012_NOVEMBER_CTP || _MSC_VER >= 1800
#define TTVSDK_VARIADIC_TEMPLATES_SUPPORTED
#endif
#endif
#elif defined(__GNUC__)
// GCC 4.4+ supports variadic templates
#if __GNUC__ > 4 || (__GNUC__ == 4 && __GNUC_MINOR__ >= 4)
#define TTVSDK_VARIADIC_TEMPLATES_SUPPORTED
#endif
#else
// Unknown compiler...assume features are not supported
#endif

#if defined(TTVSDK_VARIADIC_TEMPLATES_SUPPORTED)
namespace std {

// non-array implementation (as checked by the enable_if)
template <typename PointedAtType, typename... ParameterTypes>
inline typename std::enable_if<!std::is_array<PointedAtType>::value, std::unique_ptr<PointedAtType>>::type make_unique(
  ParameterTypes&&... parameters) {
  return std::unique_ptr<PointedAtType>(new PointedAtType(std::forward<ParameterTypes>(parameters)...));
}

// array implementation (as checked by the enable_if)
template <typename PointedAtType>
inline typename std::enable_if<std::is_array<PointedAtType>::value, std::unique_ptr<PointedAtType>>::type make_unique(
  std::size_t ElementCount) {
  typedef typename std::remove_extent<PointedAtType>::type ElementType;
  return std::unique_ptr<PointedAtType>(new ElementType[ElementCount]);
}

}  // namespace std
#else   // defined TTVSDK_VARIADIC_TEMPLATES_SUPPORTED
namespace std {

// non-array implementations (as checked by the enable_if)
template <typename PointedAtType>
inline typename std::enable_if<!std::is_array<PointedAtType>::value, std::unique_ptr<PointedAtType>>::type
make_unique() {
  return std::unique_ptr<PointedAtType>(new PointedAtType());
}

template <typename PointedAtType, typename Parameter1Type>
inline typename std::enable_if<!std::is_array<PointedAtType>::value, std::unique_ptr<PointedAtType>>::type make_unique(
  Parameter1Type&& parameter1) {
  return std::unique_ptr<PointedAtType>(new PointedAtType(std::forward<Parameter1Type>(parameter1)));
}

template <typename PointedAtType, typename Parameter1Type, typename Parameter2Type>
inline typename std::enable_if<!std::is_array<PointedAtType>::value, std::unique_ptr<PointedAtType>>::type make_unique(
  Parameter1Type&& parameter1, Parameter2Type&& parameter2) {
  return std::unique_ptr<PointedAtType>(
    new PointedAtType(std::forward<Parameter1Type>(parameter1), std::forward<Parameter2Type>(parameter2)));
}

template <typename PointedAtType, typename Parameter1Type, typename Parameter2Type, typename Parameter3Type>
inline typename std::enable_if<!std::is_array<PointedAtType>::value, std::unique_ptr<PointedAtType>>::type make_unique(
  Parameter1Type&& parameter1, Parameter2Type&& parameter2, Parameter3Type&& parameter3) {
  return std::unique_ptr<PointedAtType>(new PointedAtType(std::forward<Parameter1Type>(parameter1),
    std::forward<Parameter2Type>(parameter2), std::forward<Parameter3Type>(parameter3)));
}

template <typename PointedAtType, typename Parameter1Type, typename Parameter2Type, typename Parameter3Type,
  typename Parameter4Type>
inline typename std::enable_if<!std::is_array<PointedAtType>::value, std::unique_ptr<PointedAtType>>::type make_unique(
  Parameter1Type&& parameter1, Parameter2Type&& parameter2, Parameter3Type&& parameter3, Parameter4Type&& parameter4) {
  return std::unique_ptr<PointedAtType>(
    new PointedAtType(std::forward<Parameter1Type>(parameter1), std::forward<Parameter2Type>(parameter2),
      std::forward<Parameter3Type>(parameter3), std::forward<Parameter4Type>(parameter4)));
}

template <typename PointedAtType, typename Parameter1Type, typename Parameter2Type, typename Parameter3Type,
  typename Parameter4Type, typename Parameter5Type>
inline typename std::enable_if<!std::is_array<PointedAtType>::value, std::unique_ptr<PointedAtType>>::type make_unique(
  Parameter1Type&& parameter1, Parameter2Type&& parameter2, Parameter3Type&& parameter3, Parameter4Type&& parameter4,
  Parameter5Type&& parameter5) {
  return std::unique_ptr<PointedAtType>(new PointedAtType(std::forward<Parameter1Type>(parameter1),
    std::forward<Parameter2Type>(parameter2), std::forward<Parameter3Type>(parameter3),
    std::forward<Parameter4Type>(parameter4), std::forward<Parameter5Type>(parameter5)));
}

// array implementation (as checked by the enable_if)
template <typename PointedAtType>
inline typename std::enable_if<std::is_array<PointedAtType>::value, std::unique_ptr<PointedAtType>>::type make_unique(
  std::size_t ElementCount) {
  typedef typename std::remove_extent<PointedAtType>::type ElementType;
  return std::unique_ptr<PointedAtType>(new ElementType[ElementCount]);
}

}  // namespace std
#endif  // ! defined TTVSDK_VARIADIC_TEMPLATES_SUPPORTED
