/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/eventscheduler.h"
#include "twitchsdk/core/mutex.h"
#include "twitchsdk/core/result.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"

#include <deque>
#include <vector>

namespace ttv {
class EventQueue;
}

/**
 * Thread-safe data structure to manage and execute tasks/trigger events.
 * Useful as a helper for IEventScheduler implementations to maintain a queue of tasks in a thread-safe manner.
 *
 * Tasks can be added into the EventQueue to be run immediately or some later time from now with InsertTask().
 * Each Task in the EventQueue has a unique identifier TaskId (increments up starting from 1) that can be used to remove
 * the task later from the queue (without running it) with RemoveTask(). The EventQueue is sorted in ascending timestamp
 * order with the 'oldest' task coming first. If the timestamps for tasks are the same, tasks added earlier into the
 * queue will be run first.
 *
 */
class ttv::EventQueue {
 public:
  /**
   * @param[in] name The name of the queue (used for logging purposes).
   */
  EventQueue(const std::string& name);
  ~EventQueue();

 public:
  /**
   * Adds a task into the EventQueue with the current time + some delay.
   *
   * @param[in] taskParams Parameters (taskFunction, delay, name) of the task to be added into the EventQueue.
   * @return
   *   - The unique id of the task that can be used to cancel the task later with EventQueue::RemoveTask.
   */
  TaskId InsertTask(TaskParams&& taskParams);

  /**
   * Removes the task with the specified TaskId from the EventQueue (and does not run the task).
   * If called on a task currently being ran by the EventQueue, the task will not be aborted/cancelled.
   *
   * @param[in] taskId The task to be removed from the EventQueue.
   * @return
   *   - true: The task was successly removed.
   *   - false: A task with the specified TaskId was not found, and not removed.
   */
  bool RemoveTask(TaskId taskId);

  /**
   * Removes all tasks in the EventQueue (and does not run them).
   * Does not cancel/remove any task that is currently running.
   */
  void Clear();

  /**
   * Wait until a task in the event queue can be run, remove that task and then run it in the current execution context.
   *
   * WaitForEvent() will wait until the first/'oldest' task in the EventQueue has a timestamp less than the current
   * timestamp. If there already exists such a task, then WaitForEvent() will not wait, and will immediately remove that
   * task, run it and then return.
   */
  void WaitForEvent();

  /**
   * Wait (only for a given timeout) until a task in the event queue can be run, remove that task and then run it in the
   * current execution context.
   *
   * WaitForEventWithTimeout() will wait until the first/'oldest' task in the EventQueue has a timestamp less than the
   * current timestamp OR until the timeout is hit. If there already exists such a task, then WaitForEvent() will not
   * wait, and will immediately remove that task, run it and then return. If the timeout is hit,
   * WaitForEventWithTimeout() will return without running a task.
   *
   * @param[in] timeout The maxiumum amount to wait for an event to be run.
   * @return
   *   - true: If an event/task was succesfully run.
   *   - false: If an event/task was not run, i.e the timeout was hit.
   */
  bool WaitForEventWithTimeout(uint64_t timeout);

 private:
  struct Task {
    TaskFunc taskFunc;             //!< The task to be run
    std::string taskName;          //!< The name of the task (used for logging purposes)
    uint64_t invocationTimestamp;  //!< The timestamp at which to run the task
    TaskId taskId;                 //!< Unique id for the task to be run

    bool operator<(const Task& other) const {
      if (this->invocationTimestamp == other.invocationTimestamp) {
        return this->taskId < other.taskId;
      }

      return this->invocationTimestamp < other.invocationTimestamp;
    }
  };

  std::deque<Task> mQueue;
  std::vector<TaskFunc> mCancelledTasks;
  std::unique_ptr<IConditionMutex> mMutex;
  std::string mName;
  TaskId mCurrentTaskId;
};
