/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/internal/pch.h"

#include "twitchsdk/core/result.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"

namespace ttv {
class IEventScheduler;
class IMainEventSchedulerFactory;
class IBackgroundEventSchedulerFactory;
struct TaskParams;

/**
 * Sets the IMainEventSchedulerFactory to be used as the MainEventSchedulerFactory.
 * Must set the MainEventSchedulerFactory before trying to create a MainEventScheduler with CreateMainEventScheduler.
 */
void SetMainEventSchedulerFactory(const std::shared_ptr<IMainEventSchedulerFactory>& factory);
/**
 * Sets the IBackgroundEventSchedulerFactory to be used as the BackgroundEventSchedulerFactory.
 * Must set the BackgroundEventSchedulerFactory before trying to create a BackgroundEventScheduler with
 * CreateBackgroundEventScheduler.
 */
void SetBackgroundEventSchedulerFactory(const std::shared_ptr<IBackgroundEventSchedulerFactory>& factory);

/**
 * Creates a MainEventScheduler and should only be called once by the SDK in CoreAPI.
 * This returns an IEventScheduler where tasks can be scheduled to run on the main threading context.
 * @return
 *   - TTV_EC_SUCCESS: MainEventScheduler was successfully created.
 *   - TTV_EC_NOT_INITIALIZED: Factory has not been set with SetMainEventSchedulerFactory.
 */
TTV_ErrorCode CreateMainEventScheduler(std::shared_ptr<IEventScheduler>& result);
TTV_ErrorCode CreateMainEventScheduler(const std::string& name, std::shared_ptr<IEventScheduler>& result);
/**
 * Creates a BackgroundEventScheduler. Multiple BackgroundEventSchedulers can be created in various contexts.
 * This returns an IEventScheduler where tasks can be scheduled to run on a background threading context.
 * @return
 *   - TTV_EC_SUCCESS: BackgroundEventScheduler was successfully created.
 *   - TTV_EC_NOT_INITIALIZED: Factory has not been set with SetBackgroundEventSchedulerFactory.
 */
TTV_ErrorCode CreateBackgroundEventScheduler(std::shared_ptr<IEventScheduler>& result);
TTV_ErrorCode CreateBackgroundEventScheduler(const std::string& name, std::shared_ptr<IEventScheduler>& result);
}  // namespace ttv

/**
 * Task parameters for adding IEventScheduler/EventQueue tasks.
 * Supports inline curly brace initialization when calling IEventScheduler::ScheduleTask and EventQueue::InsertTask.
 * taskFunc, taskName, and delay make up the task context - any missing parameters will be set to the default.
 *   - TaskFunc&& taskFunc              Required - The task to be added into the IEventScheduler/EventQueue.
 *   - const std::string& taskName;     Optional - Name of the task to be used for logging purposes (default is "").
 *   - uint64_t delayMilliseconds       Optional - The delay in milliseconds to run the task (default is 0).
 */
struct ttv::TaskParams {
  TaskParams(TaskFunc&& taskFuncArg) : taskFunc(std::move(taskFuncArg)), delayMilliseconds(0) {}

  TaskParams(TaskFunc&& taskFuncArg, const std::string& taskNameArg)
      : taskFunc(std::move(taskFuncArg)), taskName(taskNameArg), delayMilliseconds(0) {}

  TaskParams(TaskFunc&& taskFuncArg, uint64_t delayMillisecondsArg)
      : taskFunc(std::move(taskFuncArg)), delayMilliseconds(delayMillisecondsArg) {}

  TaskParams(TaskFunc&& taskFuncArg, uint64_t delayMillisecondsArg, const std::string& taskNameArg)
      : taskFunc(std::move(taskFuncArg)), taskName(taskNameArg), delayMilliseconds(delayMillisecondsArg) {}

  TaskFunc taskFunc;           //!< Required - The task to be added into the IEventScheduler/EventQueue.
  std::string taskName;        //!< Optional - Name of the task to be used for logging purposes (default is "").
  uint64_t delayMilliseconds;  //!< Optional - The delay in milliseconds to run the task (default is 0).
};

/**
 * Interface that describes a serial dispatch queue where tasks/events can be scheduled to be run immediately or with
 * some delay. Tasks (code blocks) added to the IEventScheduler will be run in ascending timestamp priority order, with
 * older tasks executing first. When tasks with the same timestamp are added into the queue, the task added earlier will
 * be executed first. Each task that is added into IEventScheduler can optionally get back a TaskId that uniquely
 * identifies the task in IEventScheduler. This TaskId can be used later to cancel the task (if it already hasn't been
 * executed).
 *
 * For IEventScheduler implementations, Shutdown() must be called before the IEventScheduler is destroyed.
 */
class ttv::IEventScheduler {
 public:
  /**
   * The state of the IEventScheduler
   */
  enum class EventSchedulerState { Running, ShuttingDown, ShutDown };

  virtual ~IEventScheduler() = default;

  /**
   * Adds a task into the IEventScheduler with the current time + some delay.
   * The task will be run before other pending tasks with timestamp `current time + delay` or later.
   * Should be called when the IEventScheduler is in the Running state.
   *
   * @param[in] taskParams Parameters (taskFunction, delay, name) of the task to be added into the IEventScheduler.
   * @return
   *   - The unique id of the task that can be used to cancel the task later with `CancelTask`.
   *   - TTV_EC_NOT_INITIALIZED: The IEventScheduler has not yet been initialized or has already been shutdown.
   *   - TTV_EC_INVALID_ARG: taskFunction is nullptr and the task was not added into the EventScheduler.
   */
  virtual Result<TaskId> ScheduleTask(TaskParams&& taskParams) = 0;

  /**
   * Cancels the pending task with given TaskId in the IEventScheduler (and does not run the task).
   * Should be called when the IEventScheduler is in the Running state.
   *
   * @param[in] taskId The task to be cancelled in IEventScheduler
   * @return
   *   - TTV_EC_SUCCESS: The task was successfully cancelled.
   *   - TTV_EC_NOT_INITIALIZED: The IEventScheduler has not yet been initialized or has already been shutdown.
   *   - TTV_EC_OPERATION_FAILED: The task was not found in the IEventScheduler and was failed to be cancelled.
   */
  virtual TTV_ErrorCode CancelTask(TaskId taskId) = 0;

  /**
   * Asynchronously shutdown the IEventScheduler.
   * This must be called before IEventScheduler is destroyed.
   * All tasks in the IEventScheduler should be cancelled in the normal task execution context for the IEventScheduler.
   * Should be called when the IEventScheduler is in the Running state.
   * After calling Shutdown, the IEventScheduler will move to the ShuttingDown state and then eventually the ShutDown
   * state once shutdownTask is fired.
   *
   * @param[in] shutdownTask The task to be called by the IEventScheduler once the IEventScheduler is shutdown.
   * @return
   *   - TTV_EC_SUCCESS: The request to Shutdown the IEventScheduler was sent successfully.
   *   - TTV_EC_NOT_INITIALIZED: The IEventScheduler has already been shutdown.
   */
  virtual TTV_ErrorCode Shutdown(TaskFunc&& shutdownTask) = 0;

  /**
   * Returns the state of the IEventScheduler.
   */
  virtual EventSchedulerState GetState() = 0;
};

/**
 * Factory for creating MainEventSchedulers.
 * The MainEventScheduler is generally for tasks/actions that should be run in the main threading context.
 */
class ttv::IMainEventSchedulerFactory {
 public:
  virtual ~IMainEventSchedulerFactory();

  /**
   * Creates a MainEventScheduler and should only be called once by the SDK in CoreAPI.
   * This returns an IEventScheduler where tasks can be scheduled to run on the main threading context.
   * @return
   *   - TTV_EC_SUCCESS: MainEventScheduler was successfully created.
   */
  virtual TTV_ErrorCode CreateMainEventScheduler(std::shared_ptr<IEventScheduler>& result) = 0;
  virtual TTV_ErrorCode CreateMainEventScheduler(const std::string& name, std::shared_ptr<IEventScheduler>& result) = 0;
};

/**
 * Factory for creating BackgroundEventSchedulers.
 * The BackgroundEventScheduler is generally for tasks/actions that should be run in a background threading context.
 */
class ttv::IBackgroundEventSchedulerFactory {
 public:
  virtual ~IBackgroundEventSchedulerFactory();

  /**
   * Creates a BackgroundEventScheduler. Multiple BackgroundEventSchedulers can be created in various contexts.
   * This returns an IEventScheduler where tasks can be scheduled to run on a background threading context.
   * @return
   *   - TTV_EC_SUCCESS: BackgroundEventScheduler was successfully created.
   */
  virtual TTV_ErrorCode CreateBackgroundEventScheduler(std::shared_ptr<IEventScheduler>& result) = 0;
  virtual TTV_ErrorCode CreateBackgroundEventScheduler(
    const std::string& name, std::shared_ptr<IEventScheduler>& result) = 0;
};
