/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include <functional>
#include <memory>
#include <vector>

namespace ttv {
template <typename T>
class EventSource;
}

template <typename T>
class ttv::EventSource {
 public:
  void AddListener(const std::shared_ptr<T>& listener) {
    if (listener != nullptr) {
      auto iter = FindListener(listener);
      if (iter == mListeners.end()) {
        mListeners.push_back(listener);
      }
    }
  }

  bool RemoveListener(const std::shared_ptr<T>& listener) {
    if (listener != nullptr) {
      auto iter = FindListener(listener);
      if (iter != mListeners.end()) {
        mListeners.erase(iter);
        return true;
      }
    }

    return false;
  }

  void ClearListeners() { mListeners.clear(); }

  void Invoke(std::function<void(const std::shared_ptr<T>&)> callback) {
    std::vector<std::shared_ptr<T>> listeners;
    CaptureListeners(listeners);
    Invoke(callback, listeners);
  }

  void Invoke(
    std::function<void(const std::shared_ptr<T>&)> callback, const std::vector<std::shared_ptr<T>>& listeners) {
    for (const auto& listener : listeners) {
      callback(listener);
    }
  }

  bool Empty() {
    Tidy();
    return mListeners.empty();
  }

  void CaptureListeners(std::vector<std::shared_ptr<T>>& listeners) {
    for (IterType iter = mListeners.begin(); iter != mListeners.end();) {
      auto locked = iter->lock();

      if (locked == nullptr) {
        iter = mListeners.erase(iter);
      } else {
        listeners.push_back(locked);
        ++iter;
      }
    }
  }

 private:
  using ListenerType = typename std::weak_ptr<T>;
  using ListType = typename std::vector<ListenerType>;
  using IterType = typename ListType::iterator;

  void Tidy() {
    for (IterType iter = mListeners.begin(); iter != mListeners.end();) {
      auto locked = iter->lock();

      if (locked == nullptr) {
        iter = mListeners.erase(iter);
      } else {
        ++iter;
      }
    }
  }

  IterType FindListener(const std::shared_ptr<T>& listener) {
    IterType iter = mListeners.begin();

    for (; iter != mListeners.end();) {
      auto locked = iter->lock();

      if (locked == nullptr) {
        iter = mListeners.erase(iter);
      } else if (locked == listener) {
        break;
      } else {
        ++iter;
      }
    }

    return iter;
  }

  ListType mListeners;
};
