/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/types/coretypes.h"

namespace ttv {
class IEventTracker;

/**
 * Registers the tracking event handler for the whole SDK.
 */
void SetEventTracker(const std::shared_ptr<IEventTracker>& eventTracker);

/**
 * Gets the currently registered event handler.
 */
std::shared_ptr<IEventTracker> GetEventTracker();

/**
 * Sends a tracking event to the globally registered event handler.
 * @param[in] eventName The name of the event to send.
 * @param[in] properties A map of the event properties to send with the event.
 * @return An error code indicating the result of sending the tracking event.
 *     TTV_EC_SUCCESS - Sending the event succeeded
 *     TTV_EC_NOT_AVAILABLE - No event tracker is registered.
 *     TTV_EC_HTTPREQUEST_ERROR - The event failed to send due to an HTTP error.
 */
TTV_ErrorCode TrackEvent(const std::string& eventName, const std::map<std::string, TrackingValue>& properties);
}  // namespace ttv

/**
 * An interface clients can implement that allows the SDK to send tracking events.
 */
class ttv::IEventTracker {
 public:
  virtual ~IEventTracker() = default;

  /**
   * Sends a tracking event. This method may be called from any thread, so it must handle these events in a thread-safe
   * manner.
   * @param[in] eventName The name of the event to send.
   * @param[in] properties A map of the event properties to send with the event.
   * @return An error code indicating the result of sending the tracking event.
   *     TTV_EC_SUCCESS - Sending the event succeeded
   *     TTV_EC_HTTPREQUEST_ERROR - The event failed to send due to an HTTP error.
   */
  virtual TTV_ErrorCode TrackEvent(
    const std::string& eventName, const std::map<std::string, TrackingValue>& properties) = 0;
};
