/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"

#include <functional>
#include <iostream>
#include <map>
#include <memory>
#include <vector>

namespace ttv {
class HttpRequest;

enum HttpRequestType {
  HTTP_INVALID_REQUEST = -1,

  HTTP_GET_REQUEST,
  HTTP_PUT_REQUEST,
  HTTP_POST_REQUEST,
  HTTP_DELETE_REQUEST,

  HTTP_REQUEST_COUNT
};

enum { kHttpStatusCode_Success = 200 };

struct HttpParam {
  HttpParam(const std::string& name, const std::string& value);
  HttpParam(const std::string& name, int value);

  std::string paramName;
  std::string paramValue;
};

using HttpRequestHeadersCallback =
  std::function<bool(uint statusCode, const std::map<std::string, std::string>& headers, void* userData)>;
using HttpRequestCallback = std::function<void(uint statusCode, const std::vector<char>& body, void* userData)>;

void SetHttpRequest(const std::shared_ptr<HttpRequest>& http);

/**
 * Retrieves the currently configured HttpRequest instance.
 */
std::shared_ptr<HttpRequest> GetHttpRequest();

TTV_ErrorCode HttpThreadInit();

TTV_ErrorCode SendHttpRequest(const std::string& requestName, const std::string& url,
  const std::vector<HttpParam>& requestHeaders, const uint8_t* requestBody, size_t requestBodySize,
  HttpRequestType httpReqType, uint timeOutInSecs, HttpRequestHeadersCallback headersCallback,
  HttpRequestCallback responseCallback, void* userData);

TTV_ErrorCode SendHttpRequest(const std::string& requestName, const std::string& url,
  const std::vector<HttpParam>& requestHeaders, const uint8_t* requestBody, size_t requestBodySize,
  HttpRequestType httpReqType, uint timeOutInSecs, uint numRetries, HttpRequestHeadersCallback headersCallback,
  HttpRequestCallback responseCallback, void* userData);

TTV_ErrorCode SendHttpRequest(const std::string& requestName, const std::string& url,
  const std::vector<HttpParam>& urlParams, const std::vector<HttpParam>& requestHeaders, const uint8_t* requestBody,
  size_t requestBodySize, HttpRequestType httpReqType, uint timeOutInSecs, uint numRetries,
  HttpRequestHeadersCallback headersCallback, HttpRequestCallback responseCallback, void* userData);

TTV_ErrorCode HttpThreadShutdown();

/**
 * Determines if the given url represents a Twitch endpoint.
 */
bool IsTwitchEndpoint(const std::string& url);

class HttpRequest {
 public:
  HttpRequest();
  virtual ~HttpRequest();

  // Must be called once by each thread that will be making HTTP requests
  virtual TTV_ErrorCode ThreadInit();

  // Handles automatic retries of failed requests
  virtual TTV_ErrorCode SendHttpRequest(const std::string& requestName, const std::string& url,
    const std::vector<HttpParam>& requestHeaders, const uint8_t* requestBody, size_t requestBodySize,
    HttpRequestType httpReqType, uint timeOutInSecs, uint numRetries, HttpRequestHeadersCallback headersCallback,
    HttpRequestCallback responseCallback, void* userData);

  virtual TTV_ErrorCode ThreadShutdown();

  virtual TTV_ErrorCode SendHttpRequest(const std::string& requestName, const std::string& url,
    const std::vector<HttpParam>& requestHeaders, const uint8_t* requestBody, size_t requestBodySize,
    HttpRequestType httpReqType, uint timeOutInSecs, HttpRequestHeadersCallback headersCallback,
    HttpRequestCallback responseCallback, void* userData) = 0;
};
}  // namespace ttv
