/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/httprequest.h"

#include <sstream>

namespace ttv {
class TaskRunner;
class Task;

std::string BuildUrlEncodedRequestParams(const std::vector<std::pair<std::string, std::string>>& requestParams);
std::string BuildUrlEncodedRequestParams(const std::vector<HttpParam>& requestParams);
std::string BuildUrlEncodedRequestParams(const std::map<std::string, std::string>& requestParams);
std::string BuildHttpHeader(const std::vector<ttv::HttpParam>& headerParams);
bool ContainsHttpParameter(const std::vector<HttpParam>& headers, const std::string& name);
void UrlEncode(const std::string& inputString, std::stringstream& outputStream);
std::string UrlEncode(const std::string& inputString);
void UrlDecode(const std::string& inputString, std::stringstream& outputStream);
void UrlDecode(const std::string& input, std::string& result);
std::string UrlDecode(const std::string& inputString);

TTV_ErrorCode SplitHttpParameters(
  const std::string& parameterString, std::vector<std::pair<std::string, std::string>>& result);
TTV_ErrorCode SplitHttpParameters(const std::string& parameterString, std::map<std::string, std::string>& result);

/**
 * Generate all possible hosts (i.e. by removing subdomains) for a given hostname.
 * Example: chat.twitch.tv -> {chat.twitch.tv, *.twitch.tv, twitch.tv}.
 */
TTV_ErrorCode GenerateSslVerificationHosts(const std::string& originalHost, std::vector<std::string>& result);

/**
 * Returns whether or not a given hostname is in the format of a 32-bit IPv4 address.
 * Checks for dotted decimal representation (a.b.c.d).
 * Does not check if the IP address is valid (will still return true even if numbers > 255).
 */
bool IsHostAnIpAddress(const std::string& hostName);

class Uri;
class PagedRequestFetcher;
}  // namespace ttv

class ttv::Uri {
 public:
  Uri();
  Uri(const std::string& url);

  void SetUrl(const std::string& url);
  std::string GetUrl() const;

  std::string GetPath() const { return mPath; }
  void GetPathComponents(std::vector<std::string>& result) const;
  void SetPath(const std::string& path) { mPath = path; }

  std::string GetProtocol() const { return mProtocol; }
  void SetProtocol(const std::string& protocol) { mProtocol = protocol; }

  std::string GetHostName() const { return mHostName; }
  void SetHostName(const std::string& host) { mHostName = host; }

  std::string GetPort() const { return mPort; }
  void SetPort(const std::string& port) { mPort = port; }

  bool GetPort(uint32_t& result) const;

  void ClearParams();
  void SetParam(const std::string& param, const char* value);
  void SetParam(const std::string& param, const std::string& value);
  void SetParam(const std::string& param, uint32_t value);
  void SetParam(const std::string& param, int32_t value);
  void SetParam(const std::string& param, uint64_t value);
  void SetParam(const std::string& param, int64_t value);
  void SetParam(const std::string& param, double value);
  void SetParam(const std::string& param, bool value);

  std::map<std::string, std::string>& GetParams() { return mParams; }
  const std::map<std::string, std::string>& GetParams() const { return mParams; }

  bool ContainsParam(const std::string& param) const;

  operator std::string() const;
  bool operator==(const Uri& other) const;
  bool operator!=(const Uri& other) const;

 private:
  void DisassembleUrl(const std::string& url);
  std::string AssembleUrl() const;

  std::string mProtocol;
  std::string mHostName;
  std::string mPort;
  std::string mPath;
  std::map<std::string, std::string> mParams;
};

/**
 * A helper class that fetches all pages of an endpoint.  It is assumed that only status codes in the range [200, 299]
 * will be returned when a successful page is returned.
 */
class ttv::PagedRequestFetcher {
 public:
  /**
   * Called by the fetcher to create and schedule the next task with a task runner.
   */
  using CreateTaskCallback = std::function<TTV_ErrorCode(const std::string& cursor, std::shared_ptr<Task>& task)>;
  /**
   * Called when page fetching is complete.
   */
  using CompleteCallback = std::function<void(TTV_ErrorCode ec)>;

 public:
  PagedRequestFetcher();

  /**
   * Kicks off the fetching of pages.
   */
  TTV_ErrorCode Start(
    const std::string& initialCursor, CreateTaskCallback createTaskCallback, CompleteCallback completeCallback);
  /**
   * Cancels the fetching of further pages.
   */
  void Cancel();
  /**
   * Called by the client to notify the fetcher that the fetch of the current page has completed.
   */
  void FetchComplete(TTV_ErrorCode ec, const std::string& cursor);

  /**
   * Whether or not fetches are in progress.
   */
  bool InProgress() const { return mCurrentTask != nullptr; }

 protected:
  TTV_ErrorCode FetchPage();

  CreateTaskCallback mCreateTaskCallback;
  CompleteCallback mCompleteCallback;
  std::shared_ptr<Task> mCurrentTask;
  std::string mCursor;
  bool mCancel;
};
