/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2019 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/internal/graphql/utilities/channelinfo.h"
#include "twitchsdk/core/json/jsonserialization.h"

namespace ttv {
namespace graphql {

template <typename GQLType>
StreamInfo GQLUserToStreamInfo(const GQLType& user) {
  StreamInfo result;

  result.channelInfo = GQLUserToChannelInfo(user);

  if (user.stream.HasValue()) {
    const auto& stream = user.stream.Value();

    std::string templateStr = stream.previewImageURL.ValueOrDefault("");

    // Format an image template URL to support the desired sizes
    // Ex: https://static-cdn.jtvnw.net/previews-ttv/live_user_giantwaffle-{width}x{height}.jpg
    if (!templateStr.empty()) {
      const auto widthStrLen = strlen(kWidthString);
      const auto heightStrLen = strlen(kHeightString);

      auto pos = templateStr.find(kWidthString);
      if (pos != std::string::npos) {
        std::string smallStr = templateStr;
        std::string mediumStr = templateStr;
        std::string largeStr = templateStr;

        smallStr.replace(pos, widthStrLen, "80");
        mediumStr.replace(pos, widthStrLen, "320");
        largeStr.replace(pos, widthStrLen, "640");

        auto smallPos = smallStr.find(kHeightString);
        auto mediumPos = mediumStr.find(kHeightString);
        auto largePos = largeStr.find(kHeightString);

        if (smallPos != std::string::npos && mediumPos != std::string::npos && largePos != std::string::npos) {
          smallStr.replace(smallPos, heightStrLen, "45");
          mediumStr.replace(mediumPos, heightStrLen, "180");
          largeStr.replace(largePos, heightStrLen, "360");

          result.previewImages.smallUrl = smallStr;
          result.previewImages.mediumUrl = mediumStr;
          result.previewImages.largeUrl = largeStr;
          result.previewImages.templateUrl = templateStr;
        }
      }
    }

    if (stream.game.HasValue()) {
      result.game = stream.game.Value().name;
    }

    if (stream.archiveVideo.HasValue()) {
      ParseNum(stream.archiveVideo.Value().id, result.archiveVideoId);
    }

    result.averageFPS = static_cast<double>(stream.averageFPS.ValueOrDefault(0.f));
    ParseNum(stream.id, result.streamId);
    result.delay = static_cast<uint64_t>(stream.delayLengthSeconds.ValueOrDefault(0));
    result.viewerCount = static_cast<uint64_t>(stream.viewersCount.ValueOrDefault(0));
    result.videoHeight = static_cast<uint32_t>(stream.height.ValueOrDefault(0));
    result.createdAtTimestamp = stream.createdAt.ValueOrDefault(0);

    std::string type = stream.type.ValueOrDefault("");

    if (type == "watch_party") {
      result.streamType = StreamType::WatchParty;
      result.broadcastPlatform = BroadcastPlatform::WatchParty;
    } else if (type == "premiere") {
      result.streamType = StreamType::Premiere;
      result.broadcastPlatform = BroadcastPlatform::Premiere;
    } else if (type == "rerun") {
      result.streamType = StreamType::Rerun;
      result.broadcastPlatform = BroadcastPlatform::Rerun;
    } else if (type == "playlist") {
      result.streamType = StreamType::Playlist;
      result.broadcastPlatform = BroadcastPlatform::Playlist;
    } else if (type == "mobile") {
      result.streamType = StreamType::Unknown;  // Not defined for StreamType
      result.broadcastPlatform = BroadcastPlatform::Mobile;
    } else if (type == "xbox") {
      result.streamType = StreamType::Unknown;  // Not defined for StreamType
      result.broadcastPlatform = BroadcastPlatform::Xbox;
    } else if (type == "ps4") {
      result.streamType = StreamType::Unknown;  // Not defined for StreamType
      result.broadcastPlatform = BroadcastPlatform::PS4;
    } else if (type == "live") {
      result.streamType = StreamType::Live;
      result.broadcastPlatform = BroadcastPlatform::Live;
    } else {
      result.streamType = StreamType::Unknown;
      result.broadcastPlatform = BroadcastPlatform::Unknown;
    }

    result.isPlaylist = (result.streamType == StreamType::Playlist);
  }

  return result;
}
}  // namespace graphql
}  // namespace ttv
