#pragma once

#include "twitchsdk/core/json/value.h"
#include "twitchsdk/core/optional.h"
#include "twitchsdk/core/types/coretypes.h"

#include <string>
#include <vector>

namespace ttv {
struct UserInfo;

/**
 * Parse a string from the given Json object. Returns false if the object does not contain the given key.
 */
bool ParseString(const ttv::json::Value& root, const char* key, std::string& result);

/**
 * Parse a string from the given Json object. In case of errors or a null value, assigns 'defaultValue' to 'result'.
 * Returns true if the value is null. Returns false if the value is missing or not a string.
 */
bool ParseString(const ttv::json::Value& root, const std::string& key, std::string& result, const char* defaultValue);

/**
 * Parse a bool from the given Json object. Returns false if the object does not contain the given key.
 */
bool ParseBool(const ttv::json::Value& root, const char* key, bool& result);

/**
 * Parse a bool from the given Json object. In case of errors or a null value, assigns 'defaultValue' to 'result'.
 * Returns true if the value is null. Returns false if the value is missing or not a string.
 */
bool ParseBool(const ttv::json::Value& root, const char* key, bool& result, bool defaultValue);

/**
 * Parse a uint64_t from the given Json object. Returns false if the object does not contain the given key.
 */
bool ParseUInt(const ttv::json::Value& root, const char* key, uint64_t& result);

/**
 * Parse a uint32_t from the given Json object. Returns false if the object does not contain the given key.
 */
bool ParseUInt(const ttv::json::Value& root, const char* key, uint32_t& result);

/**
 * Parse a UInt64_t from the given Json object. In case of errors or a null value, assigns 'defaultValue' to 'result'.
 * Returns true if the value is null. Returns false if the value is missing or not a string.
 */
bool ParseUInt(const ttv::json::Value& root, const char* key, uint64_t& result, uint64_t defaultValue);

/**
 * Parse a Int64_t from the given Json object. Returns false if the object does not contain the given key.
 */
bool ParseInt(const ttv::json::Value& root, const char* key, int64_t& result);

/**
 * Parse a Int64_t from the given Json object. In case of errors or a null value, assigns 'defaultValue' to 'result'.
 * Returns true if the value is null. Returns false if the value is missing or not a string.
 */
bool ParseInt(const ttv::json::Value& root, const char* key, int64_t& result, int64_t defaultValue);

/**
 * Parse a Double from the given Json object. Returns false if the object does not contain the given key.
 */
bool ParseDouble(const ttv::json::Value& root, const char* key, double& result);

/**
 * Parse a Double from the given Json object. In case of errors or a null value, assigns 'defaultValue' to 'result'.
 * Returns true if the value is null. Returns false if the value is missing or not a string.
 */
bool ParseDouble(const ttv::json::Value& root, const char* key, double& result, double defaultValue);

/**
 * Attempts to parse the given string as json and will return the parsed document if possible.
 */
bool ParseDocument(const std::string& body, ttv::json::Value& result);
/**
 * Attempts to parse the given byte array as json and will return the parsed document if possible.
 */
bool ParseDocument(const char* begin, const char* end, ttv::json::Value& result);
/**
 * Attempts to parse the given byte array as json and will return the parsed document if possible.
 */
bool ParseDocument(const uint8_t* begin, const uint8_t* end, ttv::json::Value& result);
/**
 * Attempts to parse the given byte array as json and will return the parsed document if possible.
 */
bool ParseDocument(const std::vector<char>& body, ttv::json::Value& result);

bool JsonArrayToVector(const ttv::json::Value& jArray, std::vector<float>& result);
bool JsonArrayToVector(const ttv::json::Value& jArray, std::vector<std::string>& result);

bool ParseUserId(const json::Value& jId, UserId& result);
bool ParseChannelId(const json::Value& jId, ChannelId& result);
bool ParseGameId(const json::Value& jId, GameId& result);
bool ParseTimestamp(const json::Value& jTimestamp, Timestamp& result);
bool ParseTimestamp(const ttv::json::Value& root, const char* key, Timestamp& result);
bool ParseColor(const std::string& str, Color& result);
bool ParseColor(const ttv::json::Value& root, const char* key, Color& result);
bool ParseColor(const ttv::json::Value& root, const char* key, Color& result, Color defaultValue);

bool GenerateColorString(Color color, std::string& colorString);

bool ParseUInt32(const json::Value& jValue, uint32_t& result);
bool ParseFloat(const json::Value& jValue, float& result);

/**
 * Parses the standard user object returned from Kraken.
 */
bool ParseChannelInfo(const ttv::json::Value& root, ChannelInfo& model);
bool ParseStreamInfo(const ttv::json::Value& root, StreamInfo& model);

/**
 * This will find a json value matching the key.  If no key is
 * found an empty a null Optional is returned.
 */
ttv::Optional<ttv::json::Value> FindValueByPath(const ttv::json::Value& root, const std::vector<std::string>& keyPath);

}  // namespace ttv
