/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/json/jsonserialization.h"

namespace ttv {
namespace json {
namespace description {
struct UserInfo;                                //!< Default description for UserInfo objects
struct ChannelInfo;                             //!< Default description for ChannelInfo objects
struct StreamInfo;                              //!< Default description for StreamInfo objects
struct StreamInfoUpdate;                        //!< Default description for StreamInfoUpdate objects
struct PreviewImages;                           //!< Default description for PreviewImages objects
struct BroadcastPlatform;                       //!< Default description for StreamInfo::BroadcastPlatform enum objects
struct StreamType;                              //!< Default description for StreamInfo::StreamType enum objects
struct SquadInfo;                               //!< Default description for squad streaming information through pubsub
struct SquadMember;                             //!< Default description for squad members through pubsub
struct SquadStatus;                             //!< Default description for squad streaming status through pubsub
struct DashboardActivityHeader;                 //!< Default description for dashboard activity header
struct DashboardActivityUser;                   //!< Default description for dashboard activity user
struct DashboardActivityFragment;               //!< Default description for dashboard activity fragement data
struct DashboardActivityBitsUsage;              //!< Default description for dashboard activity bits event
struct DashboardActivityHost;                   //!< Default description for dashboard activity hosting event
struct DashboardActivityRaiding;                //!< Default description for dashboard activity raiding event
struct DashboardActivitySubscription;           //!< Default description for dashboard activity subscription event
struct DashboardActivityResubscriptionSharing;  //!< Default description for dashboard activity resubscription event
struct
  DashboardActivitySubscriptionGiftingCommunity;  //!< Default description for dashboard activity community gifting event
struct
  DashboardActivitySubscriptionGiftingIndividual;  //!< Default description for dashboard activity individual gifting event
}  // namespace description
}  // namespace json
}  // namespace ttv

struct ttv::json::description::UserInfo {
  template <typename UserInfoType>
  static auto BindFields(UserInfoType& info) {
    return std::make_tuple(make_field<RequiredField>("_id", info.userId),
      make_field<RequiredField>("name", info.userName), make_field("display_name", info.displayName),
      make_field("bio", info.bio), make_field("logo", info.logoImageUrl),
      make_field<DateSchema, OptionalField>("created_at", info.createdTimestamp));
  }
};

struct ttv::json::description::ChannelInfo {
  template <typename ChannelInfoType>
  static auto BindFields(ChannelInfoType& channelInfo) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("_id", channelInfo.channelId),
      make_field<OptionalField>("broadcaster_language", channelInfo.broadcasterLanguage),
      make_field<DateSchema, OptionalField>("created_at", channelInfo.createdAtTimestamp),
      make_field<OptionalField>("description", channelInfo.description),
      make_field<OptionalField>("display_name", channelInfo.displayName),
      make_field<OptionalField>("followers", channelInfo.numFollowers),
      make_field<OptionalField>("game", channelInfo.game), make_field<OptionalField>("language", channelInfo.language),
      make_field<OptionalField>("logo", channelInfo.logoImageUrl),
      make_field<OptionalField>("mature", channelInfo.mature), make_field<OptionalField>("name", channelInfo.name),
      make_field<OptionalField>("partner", channelInfo.partner),
      make_field<OptionalField>("profile_banner", channelInfo.profileBannerImageUrl),
      make_field<OptionalField>("status", channelInfo.status),
      make_field<DateSchema, OptionalField>("updated_at", channelInfo.updatedAtTimestamp),
      make_field<OptionalField>("url", channelInfo.channelUrl),
      make_field<OptionalField>("video_banner", channelInfo.videoBannerImageUrl),
      make_field<OptionalField>("views", channelInfo.numViews));
  }
};

struct ttv::json::description::StreamInfo {
  template <typename StreamInfoType>
  static auto BindFields(StreamInfoType& streamInfo) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("_id", streamInfo.streamId),
      make_field<OptionalField>("average_fps", streamInfo.averageFPS),
      make_field<OptionalField>("broadcast_platform", streamInfo.broadcastPlatform),
      make_field<OptionalField>("channel", streamInfo.channelInfo),
      make_field<DateSchema, OptionalField>("created_at", streamInfo.createdAtTimestamp),
      make_field<OptionalField>("delay", streamInfo.delay), make_field<OptionalField>("game", streamInfo.game),
      make_field<OptionalField>("is_playlist", streamInfo.isPlaylist),
      make_field<OptionalField>("delay", streamInfo.delay),
      make_field<OptionalField>("preview", streamInfo.previewImages),
      make_field<OptionalField>("stream_type", streamInfo.streamType),
      make_field<OptionalField>("video_height", streamInfo.videoHeight),
      make_field<OptionalField>("viewers", streamInfo.viewerCount));
  }
};

struct ttv::json::description::StreamInfoUpdate {
  template <typename StreamInfoUpdateType>
  static auto BindFields(StreamInfoUpdateType& streamInfoUpdate) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("status", streamInfoUpdate.title),
      make_field<RequiredField>("game", streamInfoUpdate.game),
      make_field<RequiredField>("game_id", streamInfoUpdate.gameId));
  }
};

struct ttv::json::description::PreviewImages {
  template <typename PreviewImagesType>
  static auto BindFields(PreviewImagesType& images) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("large", images.largeUrl),
      make_field<OptionalField>("medium", images.mediumUrl), make_field<OptionalField>("small", images.smallUrl),
      make_field<OptionalField>("template", images.templateUrl));
  }
};

struct ttv::json::description::BroadcastPlatform {
  static auto EnumMap() {
    using namespace ttv::json;

    return std::make_tuple(make_enum_mapping("watch_party", ttv::BroadcastPlatform::WatchParty),
      make_enum_mapping("premiere", ttv::BroadcastPlatform::Premiere),
      make_enum_mapping("rerun", ttv::BroadcastPlatform::Rerun),
      make_enum_mapping("playlist", ttv::BroadcastPlatform::Playlist),
      make_enum_mapping("mobile", ttv::BroadcastPlatform::Mobile),
      make_enum_mapping("xbox", ttv::BroadcastPlatform::Xbox), make_enum_mapping("ps4", ttv::BroadcastPlatform::PS4),
      make_enum_mapping("live", ttv::BroadcastPlatform::Live));
  }

  static auto GetFallbackValue() { return ttv::BroadcastPlatform::Unknown; }
};

struct ttv::json::description::StreamType {
  static auto EnumMap() {
    using namespace ttv::json;

    return std::make_tuple(make_enum_mapping("watch_party", ttv::StreamType::WatchParty),
      make_enum_mapping("premiere", ttv::StreamType::Premiere), make_enum_mapping("rerun", ttv::StreamType::Rerun),
      make_enum_mapping("playlist", ttv::StreamType::Playlist), make_enum_mapping("live", ttv::StreamType::Live));
  }

  static auto GetFallbackValue() { return ttv::StreamType::Unknown; }
};

struct ttv::json::description::SquadMember {
  template <typename SquadMemberType>
  static auto BindFields(SquadMemberType& member) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("id", member.channelId),
      make_field<RequiredField>("login", member.userLogin),
      make_field<RequiredField>("display_name", member.userDisplayName),
      make_field<RequiredField>("profile_image_url_150", member.profileImageUrl150));
  }
};

struct ttv::json::description::SquadStatus {
  static auto EnumMap() {
    using namespace ttv::json;

    return std::make_tuple(make_enum_mapping("PENDING", ttv::SquadStatus::Pending),
      make_enum_mapping("LIVE", ttv::SquadStatus::Live), make_enum_mapping("ENDED", ttv::SquadStatus::Ended));
  }

  static auto GetFallbackValue() { return ttv::SquadStatus::Unknown; }
};

struct ttv::json::description::SquadInfo {
  template <typename SquadInfoType>
  static auto BindFields(SquadInfoType& squad) {
    using namespace ttv::json;

    return std::make_tuple(make_field<RequiredField>("id", squad.squadId),
      make_field<OptionalField>("owner_id", squad.ownerId), make_field<OptionalField>("members", squad.members),
      make_field<RequiredField>("status", squad.status));
  }
};

struct ttv::json::description::DashboardActivityHeader {
  template <typename Impl>
  static auto BindFields(Impl& impl) {
    using namespace ttv::json;

    return std::make_tuple(
      make_field<RequiredField>("id", impl.id), make_field<DateSchema, OptionalField>("timestamp", impl.timestamp));
  }
};

struct ttv::json::description::DashboardActivityUser {
  template <typename Impl>
  static auto BindFields(Impl& impl) {
    using namespace ttv::json;

    return std::make_tuple(make_field<OptionalField>("id", impl.userId), make_field<OptionalField>("login", impl.login),
      make_field<OptionalField>("display_name", impl.displayName));
  }
};

struct ttv::json::description::DashboardActivityFragment {
  template <typename Impl>
  static auto BindFields(Impl& impl) {
    using namespace ttv::json;

    return std::make_tuple(make_field<OptionalField>(MakeKeyPath("emoticon", "id"), impl.emoticonId),
      make_field<OptionalField>(MakeKeyPath("emoticon", "set_id"), impl.emoticonSetId),
      make_field<OptionalField>("text", impl.text));
  }
};

struct ttv::json::description::DashboardActivityBitsUsage {
  template <typename Impl>
  static auto BindFields(Impl& impl) {
    using namespace ttv::json;

    return std::make_tuple(make_field<OptionalField>("bits_amount", impl.amount),
      make_field<OptionalField>("bits_anonymous", impl.anonymous));
  }
};

struct ttv::json::description::DashboardActivityHost {
  template <typename Impl>
  static auto BindFields(Impl& impl) {
    using namespace ttv::json;

    return std::make_tuple(make_field<OptionalField>("hosting_viewer_count", impl.viewerCount));
  }
};

struct ttv::json::description::DashboardActivityRaiding {
  template <typename Impl>
  static auto BindFields(Impl& impl) {
    using namespace ttv::json;

    return std::make_tuple(make_field<OptionalField>("raiding_viewer_count", impl.viewerCount));
  }
};

struct ttv::json::description::DashboardActivitySubscription {
  template <typename Impl>
  static auto BindFields(Impl& impl) {
    using namespace ttv::json;

    return std::make_tuple(make_field<OptionalField>("subscription_tier", impl.tier));
  }
};

struct ttv::json::description::DashboardActivityResubscriptionSharing {
  template <typename Impl>
  static auto BindFields(Impl& impl) {
    using namespace ttv::json;

    return std::make_tuple(make_field<OptionalField>("subscription_tier", impl.tier),
      make_field<OptionalField>("subscription_cumulative_tenure_months", impl.cumulativeTenureMonths),
      make_field<OptionalField>("subscription_custom_message_text", impl.customMessage));
  }
};

struct ttv::json::description::DashboardActivitySubscriptionGiftingCommunity {
  template <typename Impl>
  static auto BindFields(Impl& impl) {
    using namespace ttv::json;

    return std::make_tuple(make_field<OptionalField>("subscription_gift_anonymous", impl.anonymous),
      make_field<OptionalField>("subscription_gift_quantity", impl.quantity),
      make_field<OptionalField>("subscription_gift_tier", impl.tier));
  }
};

struct ttv::json::description::DashboardActivitySubscriptionGiftingIndividual {
  template <typename Impl>
  static auto BindFields(Impl& impl) {
    using namespace ttv::json;

    return std::make_tuple(make_field<OptionalField>("subscription_gift_anonymous", impl.anonymous),
      make_field<OptionalField>("subscription_gift_tier", impl.tier));
  }
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::UserInfo> {
  using Type = ObjectSchema<ttv::json::description::UserInfo>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::ChannelInfo> {
  using Type = ObjectSchema<ttv::json::description::ChannelInfo>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::PreviewImages> {
  using Type = ObjectSchema<ttv::json::description::PreviewImages>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::StreamInfo> {
  using Type = ObjectSchema<ttv::json::description::StreamInfo>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::StreamInfoUpdate> {
  using Type = ObjectSchema<ttv::json::description::StreamInfoUpdate>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::BroadcastPlatform> {
  using Type = EnumSchema<ttv::json::description::BroadcastPlatform>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::StreamType> {
  using Type = EnumSchema<ttv::json::description::StreamType>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::SquadStatus> {
  using Type = EnumSchema<ttv::json::description::SquadStatus>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::SquadMember> {
  using Type = ObjectSchema<ttv::json::description::SquadMember>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::SquadInfo> {
  using Type = ObjectSchema<ttv::json::description::SquadInfo>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::DashboardActivityHeader> {
  using Type = ObjectSchema<ttv::json::description::DashboardActivityHeader>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::DashboardActivityUser> {
  using Type = ObjectSchema<ttv::json::description::DashboardActivityUser>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::DashboardActivityFragment> {
  using Type = ObjectSchema<ttv::json::description::DashboardActivityFragment>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::DashboardActivityBitsUsage> {
  using Type = ObjectSchema<ttv::json::description::DashboardActivityBitsUsage>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::DashboardActivityHost> {
  using Type = ObjectSchema<ttv::json::description::DashboardActivityHost>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::DashboardActivityRaiding> {
  using Type = ObjectSchema<ttv::json::description::DashboardActivityRaiding>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::DashboardActivitySubscription> {
  using Type = ObjectSchema<ttv::json::description::DashboardActivitySubscription>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::DashboardActivityResubscriptionSharing> {
  using Type = ObjectSchema<ttv::json::description::DashboardActivityResubscriptionSharing>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::DashboardActivitySubscriptionGiftingCommunity> {
  using Type = ObjectSchema<ttv::json::description::DashboardActivitySubscriptionGiftingCommunity>;
};

template <>
struct ttv::json::DefaultSchemaProvider<ttv::DashboardActivitySubscriptionGiftingIndividual> {
  using Type = ObjectSchema<ttv::json::description::DashboardActivitySubscriptionGiftingIndividual>;
};
