/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/value.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/core/optional.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/dashboardtypes.h"

#include <string.h>

#include <array>

namespace ttv {
namespace json {
/**
 * Public convenience functions for serializing and unserializing objects using the default schema for the passed-in
 * object.
 */

/**
 * Translates an object into JSON using its default schema.
 * @param[in] object The input object to serialize into JSON.
 * @param[out] value The output JSON value, or a null value if parsing failed.
 * @return Whether parsing succeeded.
 */
template <typename ObjectType>
bool ToJson(const ObjectType& object, ttv::json::Value& value);

/**
 * Translates an object into a JSON string using the default schema.
 * @param[in] object The input object to serialize into JSON.
 * @param[out] jsonString The output JSON string. This value is not modified if parsing fails.
 * @return Whether parsing succeed.
 */
template <typename ObjectType>
bool ToJsonString(const ObjectType& object, std::string& jsonString);

/**
 * Deserializes JSON into an object using the object's default schema.
 * @param[in] value The JSON value to deserialize.
 * @param[out] object The deserialized object, or a default constructed object if the parsing fails.
 * @return Whether parsing succeeded.
 */
template <typename ObjectType>
bool ToObject(const ttv::json::Value& value, ObjectType& object);

/**
 * Deserializes JSON into an object using the object's default schema.
 * @param[in] jsonString The JSON string to deserialize.
 * @param[out] object The deserialized object, or a default constructed object if the parsing fails.
 * @return Whether parsing succeeded.
 */
template <typename ObjectType>
bool ToObject(const std::string& jsonString, ObjectType& object);

/**
 * Deserializes JSON into an object using the object's default schema.
 * @param[in] jsonString The null-terminated JSON string to deserialize.
 * @param[out] object The deserialized object, or a default constructed object if the parsing fails.
 * @return Whether parsing succeeded.
 */
template <typename ObjectType>
bool ToObject(const char* jsonString, ObjectType& object);

/**
 * Schemas
 *
 * Schemas are types that define ways to translate between objects and JSON.
 * They should conform to the following concept:
 *
 * ```
 * struct Schema
 * {
 *     static bool Parse(const ttv::json::Value& value, ObjectType& output);
 *     static bool Emit(const ObjectType& input, ttv::json::Value& value);
 * };
 * ```
 *
 * The Parse function takes in a JSON value and outputs an object.
 * The Emit function takes in an object and outputs a JSON object.
 *
 * Schemas are implemented for fundamental types, such as integers, strings, and booleans. They are
 * also implemented to translate JSON arrays into std::vectors, and there are utilities for simple
 * object parsing as well (see ObjectSchema below). For objects that need some sort of specialized
 * processing or translation, specialized schemas can be created. However, for simple objects that
 * have a simple one-to-one relationship between JSON fields and C++ struct fields, using ObjectSchema
 * is recommended.
 */
struct IntegerSchema;
struct UnsignedIntegerSchema;
struct FloatingPointSchema;
struct StringSchema;
struct BooleanSchema;
struct DateSchema;
struct ColorSchema;

template <typename ElementSchema>
struct ArraySchema;

template <typename ElementSchema, typename ElementType>
struct OptionalSchema;

/**
 * Enum Schema
 *
 * The EnumSchema is a reusable schema that takes in an EnumDescription type, which conforms to the
 * following concept:
 *
 * ```
 * struct EnumDescription
 * {
 *     static auto EnumMap() const;
 *     static auto GetDefaultValue() const; // Optional
 * };
 * ```
 *
 * The implementation of the static method "EnumField" should return a tuple of "EnumMapping" objects
 * that map the expected strings from the JSON and how they map to individual enumeration values.
 *
 * The implementation of the static method "GetDefaultValue" is optional. If it is implemented, it will
 * be used when a string is found that does not match any specified in the EnumMap, and parsing will
 * will succeed. If it is not implemented, parsing will fail if the string is not found in the EnumMap.
 */
template <typename EnumDescription>
struct EnumSchema;

/**
 * A class used by EnumDescriptions to indicate how a particular string appearing in JSON corresponds
 * to an enumeration value. Instantiating these objects directly is not recommended. Instead, use the
 * make_enum_mapping function instead.
 */
template <typename EnumType>
class EnumMapping;

/**
 * Create an EnumMapping between a string value expected in JSON and the corresponding enumeration value.
 * @param string The expected string to find in JSON.
 * @param value The corresponding enumeration value.
 */
template <typename EnumType>
auto make_enum_mapping(const char* string, EnumType value);

/**
 * Case-insensitive version due to GraphQL not obeserving their schema.
 */
template <typename EnumType>
auto make_enum_mapping_case_insensitive(const char* string, EnumType value);

/**
 * Object Schema
 *
 * The ObjectSchema is a reusable schema that takes in an ObjectDescription type, which conforms
 * to the following concept:
 *
 * ```
 * struct ObjectDescription
 * {
 *     template <typename ObjectType>
 *     static auto BindFields(ObjectType& info);
 * };
 * ```
 *
 * It must provide an implementation of the static method "BindFields", which provides a tuple
 * of JavaField objects (documented below) that map JSON fields to C++ fields. The method is
 * templated so that it can bind to the fields of const and non-const objects.
 *
 */
template <typename ObjectDescription>
struct ObjectSchema;

/**
 * Value types indicating whether a field in a JSON object is required or optional, to be used with
 * make_field below.
 */
struct RequiredField;
struct OptionalField;

/**
 * A class used by the object schema to map between JSON fields and C++ fields. Instantiating these
 * objects directly is not recommended, use the make_field functions documented below.
 */
template <typename FieldType, typename RequiredType, typename SchemaType, size_t KeyPathSize>
class JsonField;

/**
 * A class used by the object schema to map variant types to the correct schema.
 */
template <typename DesiredType, typename RequiredType, typename SchemaType, size_t KeyPathSize>
class VariantSelector;

/**
 * Create a JSONField explicitly defining the schema to use and whether the field is required.
 *
 * @tparam RequiredType Used to specify if parsing should be aborted if this field fails to parse.
 * Should be of type RequiredField or OptionalField.
 *
 * @tparam SchemaType The scheme used to serialize or deserialize.
 * @tparam ObjectType The C++ object type to bind to (automatically inferred from arguments).
 */
template <typename SchemaType, typename RequiredType, typename ObjectType>
auto make_field(const char* key, ObjectType& field);

template <typename SchemaType, typename RequiredType, typename ObjectType, size_t ArraySize>
auto make_field(const std::array<const char*, ArraySize>& keyPath, ObjectType& field);

/**
 * Create a JSONField with the default schema and explicitly define whether it is required.
 *
 * @tparam Required Used to specify if parsing should be aborted if this field fails to parse.
 * @tparam ObjectType The C++ object type to bind to (automatically inferred from arguments).
 */
template <typename RequiredType, typename ObjectType>
auto make_field(const char* key, ObjectType& field);

template <typename RequiredType, typename ObjectType, size_t ArraySize>
auto make_field(const std::array<const char*, ArraySize>& keyPath, ObjectType& field);

/**
 * Create a JSONField with the default schema that is not required.
 *
 * @tparam ObjectType The C++ object type to bind to (automatically inferred from arguments).
 */
template <typename ObjectType>
auto make_field(const char* key, ObjectType& field);

template <typename ObjectType, size_t ArraySize>
auto make_field(const std::array<const char*, ArraySize>& keyPath, ObjectType& field);

/**
 * Creates a VariantSelector with the default schema.  This is currently not supporting
 * Arrays of Variant.  Additional development will be needed to support Arrays of Variants.
 */
template <typename SchemaType, typename RequiredType, typename ObjectType>
auto make_variant_selector(ObjectType& field);

template <typename RequiredType, typename ObjectType>
auto make_variant_selector(ObjectType& field);

template <typename ObjectType>
auto make_variant_selector(ObjectType& field);

/**
 * Partial specializations of the DefaultSchemaProvider struct provide the default schema for
 * a given type. These specializations conform to the following concept:
 *
 * ```
 * template <typename ObjectType>
 * struct DefaultSchemaProvider
 * {
 *     using Type = SchemaType;
 * };
 * ```
 *
 * Where ObjectType is the type to serialize/deserialize, and the SchemaType is the desired
 * schema to use by default for serializing and deserializing objects of ObjectType.
 */
template <typename ObjectType, typename Enable = void>
struct DefaultSchemaProvider;

/**
 * A convenience alias to make for easier syntax when getting the DefaultSchema for a type.
 */
template <typename ObjectType>
using DefaultSchema = typename DefaultSchemaProvider<ObjectType>::Type;

/**
 * Helper function to create an array of proper size based on the number of key path values passed in.
 */
template <typename... Args>
std::array<const char*, sizeof...(Args)> MakeKeyPath(Args... args);
}  // namespace json
}  // namespace ttv

template <typename ObjectType>
bool ttv::json::ToJson(const ObjectType& object, ttv::json::Value& value) {
  return DefaultSchema<std::decay_t<ObjectType>>::Emit(object, value);
}

template <typename ObjectType>
bool ttv::json::ToJsonString(const ObjectType& object, std::string& jsonString) {
  ttv::json::Value value;
  if (ToJson(object, value)) {
    ttv::json::FastWriter writer;
    jsonString = writer.write(value);
    return true;
  } else {
    return false;
  }
}

template <typename ObjectType>
bool ttv::json::ToObject(const ttv::json::Value& value, ObjectType& object) {
  return DefaultSchema<std::decay_t<ObjectType>>::Parse(value, object);
}

template <typename ObjectType>
bool ttv::json::ToObject(const std::string& jsonString, ObjectType& object) {
  ttv::json::Value value;
  ttv::json::Reader reader;
  if (reader.parse(jsonString, value)) {
    return ToObject(value, object);
  } else {
    return false;
  }
}

template <typename ObjectType>
bool ttv::json::ToObject(const char* jsonString, ObjectType& object) {
  return ToObject(std::string{jsonString}, object);
}

struct ttv::json::IntegerSchema {
  template <typename IntegralType>
  static bool Parse(const ttv::json::Value& value, IntegralType& output) {
    if (value.isNull()) {
      return false;
    } else if (value.isString()) {
      return ParseNum(value.asString(), output);
    } else if (value.isIntegral()) {
      output = static_cast<IntegralType>(value.asInt());
      return true;
    } else {
      return false;
    }
  }

  template <typename IntegralType>
  static bool Emit(IntegralType input, ttv::json::Value& value) {
    value = static_cast<int64_t>(input);
    return true;
  }
};

struct ttv::json::UnsignedIntegerSchema {
  template <typename UnsignedIntegralType>
  static bool Parse(const ttv::json::Value& value, UnsignedIntegralType& output) {
    if (value.isNull()) {
      return false;
    } else if (value.isString()) {
      return ParseNum(value.asString(), output);
    } else if (value.isUInt()) {
      output = static_cast<UnsignedIntegralType>(value.asUInt());
      return true;
    } else if (value.isInt()) {
      if (value.asInt() < 0) {
        return false;
      }

      output = static_cast<UnsignedIntegralType>(value.asUInt());
      return true;
    } else {
      return false;
    }
  }

  template <typename UnsignedIntegralType>
  static bool Emit(UnsignedIntegralType input, ttv::json::Value& value) {
    value = static_cast<uint64_t>(input);
    return true;
  }
};

struct ttv::json::FloatingPointSchema {
  template <typename FloatingPointType>
  static bool Parse(const ttv::json::Value& value, FloatingPointType& output) {
    if (value.isNull()) {
      return false;
    } else if (value.isNumeric()) {
      output = static_cast<FloatingPointType>(value.asDouble());
      return true;
    } else {
      return false;
    }
  }

  template <typename FloatingPointType>
  static bool Emit(FloatingPointType input, ttv::json::Value& value) {
    value = static_cast<double>(input);
    return true;
  }
};

struct ttv::json::StringSchema {
  static bool Parse(const ttv::json::Value& value, std::string& output) {
    if (value.isNull() || !value.isString()) {
      return false;
    } else {
      output = value.asString();
      return true;
    }
  }

  static bool Emit(const std::string& input, ttv::json::Value& value) {
    value = input;
    return true;
  }
};

struct ttv::json::BooleanSchema {
  static bool Parse(const ttv::json::Value& value, bool& output) {
    if (value.isNull() || !value.isBool()) {
      return false;
    } else {
      output = value.asBool();
      return true;
    }
  }

  static bool Emit(const bool& input, ttv::json::Value& value) {
    value = input;
    return true;
  }
};

struct ttv::json::DateSchema {
  static bool Parse(const ttv::json::Value& value, ttv::Timestamp& output) {
    if (value.isNull()) {
      return false;
    } else if (value.isNumeric()) {
      output = static_cast<Timestamp>(value.asUInt());
      return true;
    } else if (value.isString()) {
      return ttv::RFC3339TimeToUnixTimestamp(value.asString(), output);
    } else {
      return false;
    }
  }

  static bool Emit(ttv::Timestamp input, ttv::json::Value& value) {
    value = ttv::UnixTimestampToRFC3339String(input);
    return true;
  }
};

struct ttv::json::ColorSchema {
  static bool Parse(const ttv::json::Value& value, ttv::Color& output) {
    output = 0xFF000000;

    if (value.isNull()) {
      return false;
    } else if (value.isString()) {
      return ParseColor(value.asString(), output);
    } else {
      return false;
    }
  }

  // Emit(ttv::Color, ttv::json::Value&) intentionally unimplemented
};

template <typename ElementSchema>
struct ttv::json::ArraySchema {
  template <typename ContainerType>
  static bool Parse(const ttv::json::Value& value, ContainerType& output) {
    if (value.isNull() || !value.isArray()) {
      return false;
    }

    for (const auto& element : value) {
      output.emplace_back();
      if (!ElementSchema::Parse(element, output.back())) {
        output.clear();
        return false;
      }
    }

    return true;
  }

  template <typename ContainerType>
  static bool Emit(const ContainerType& input, ttv::json::Value& value) {
    for (const auto& element : input) {
      ttv::json::Value elementValue;
      if (ElementSchema::Emit(element, elementValue)) {
        value.append(elementValue);
      } else {
        value = ttv::json::nullValue;
        return false;
      }
    }
    return true;
  }
};

template <typename ElementSchema, typename ElementType>
struct ttv::json::OptionalSchema {
  static bool Parse(const ttv::json::Value& value, ttv::Optional<ElementType>& output) {
    if (value.isNull()) {
      return true;
    }

    ElementType element;
    if (ElementSchema::Parse(value, element)) {
      output = element;
    } else {
      output.Clear();
      return false;
    }

    return true;
  }

  static bool Emit(const ttv::Optional<ElementType>& input, ttv::json::Value& value) {
    if (input.HasValue()) {
      if (!ElementSchema::Emit(*input, value)) {
        value = ttv::json::nullValue;
        return false;
      }
    } else {
      value = ttv::json::nullValue;
    }

    return true;
  }
};

template <typename EnumDescription>
struct ttv::json::EnumSchema {
  template <typename EnumType>
  static bool Parse(const ttv::json::Value& value, EnumType& output) {
    if (value.isNull() || !value.isString()) {
      return false;
    } else {
      return FindEnumMatch(EnumDescription::EnumMap(), value, output);
    }
  }

  template <typename EnumType>
  static bool Emit(EnumType input, ttv::json::Value& value) {
    return FindStringMatch(EnumDescription::EnumMap(), input, value);
  }

 private:
  template <typename MapTuple, typename EnumType>
  static bool FindEnumMatch(const MapTuple& tuple, const ttv::json::Value& value, EnumType& output) {
    return FindEnumMatchFromIndex<0>(tuple, value, output);
  }

  template <size_t index, typename MapTuple, typename EnumType>
  static std::enable_if_t<(index < std::tuple_size<MapTuple>::value), bool> FindEnumMatchFromIndex(
    const MapTuple& tuple, const ttv::json::Value& value, EnumType& output) {
    auto mapping = std::get<index>(tuple);
    if (mapping.Match(value)) {
      output = mapping.GetValue();
      return true;
    } else {
      return FindEnumMatchFromIndex<index + 1>(tuple, value, output);
    }
  }

  template <size_t index, typename MapTuple, typename EnumType>
  static std::enable_if_t<(index >= std::tuple_size<MapTuple>::value), bool> FindEnumMatchFromIndex(
    const MapTuple& /*tuple*/, const ttv::json::Value& /*value*/, EnumType& output) {
    return ApplyDefault<EnumDescription>(output, 0);
  }

  template <typename Description, typename EnumType, typename = decltype(Description::GetFallbackValue())>
  static bool ApplyDefault(EnumType& output, int /*n*/) {
    // This overload will be selected if the description has defined the GetFallbackValue() static method.
    output = Description::GetFallbackValue();
    return true;
  }

  template <typename Description, typename EnumType>
  static bool ApplyDefault(EnumType& /*output*/, long /*n*/) {
    // This overload will be selected if the description has not defined the GetFallbackValue() static method.
    // The second argument type is of type long, so that when a numeric literal is passed in, it will prefer
    // to bind to the above method (with an int parameter), if it is available, instead of this one.
    return false;
  }

  template <typename MapTuple, typename EnumType>
  static bool FindStringMatch(const MapTuple& tuple, EnumType input, ttv::json::Value& value) {
    return FindStringMatchFromIndex<0>(tuple, input, value);
  }

  template <size_t index, typename MapTuple, typename EnumType>
  static std::enable_if_t<(index < std::tuple_size<MapTuple>::value), bool> FindStringMatchFromIndex(
    const MapTuple& tuple, EnumType input, ttv::json::Value& value) {
    auto mapping = std::get<index>(tuple);
    if (mapping.GetValue() == input) {
      value = mapping.GetString();
      return true;
    } else {
      return FindStringMatchFromIndex<index + 1>(tuple, input, value);
    }
  }

  template <size_t index, typename MapTuple, typename EnumType>
  static std::enable_if_t<(index >= std::tuple_size<MapTuple>::value), bool> FindStringMatchFromIndex(
    const MapTuple& /*tuple*/, EnumType /*input*/, ttv::json::Value& /*value*/) {
    return false;
  }
};

template <typename EnumType>
class ttv::json::EnumMapping {
 public:
  constexpr EnumMapping(const char* string, EnumType value, bool caseSensitive = true)
      : mString(string), mValue(value), mCaseSensitive(caseSensitive) {}

  bool Match(const ttv::json::Value& type) const {
    if (mCaseSensitive) {
      return type == mString;
    } else {
      if (type.isString()) {
        std::string input = type.asString();
#ifdef _MSC_VER
        return _stricmp(input.c_str(), mString) == 0;
#else
        return strcasecmp(input.c_str(), mString) == 0;
#endif
      }

      return false;
    }
  }

  constexpr EnumType GetValue() const { return mValue; }

  constexpr const char* GetString() const { return mString; }

 private:
  const char* mString;
  EnumType mValue;
  bool mCaseSensitive;
};

template <typename EnumType>
auto ttv::json::make_enum_mapping(const char* string, EnumType value) {
  return EnumMapping<EnumType>{string, value};
}

template <typename EnumType>
auto ttv::json::make_enum_mapping_case_insensitive(const char* string, EnumType value) {
  return EnumMapping<EnumType>{string, value, false};
}

template <typename ObjectDescription>
struct ttv::json::ObjectSchema {
  template <typename OutputType>
  static bool Parse(const ttv::json::Value& value, OutputType& output) {
    if (value.isNull() || !value.isObject()) {
      return false;
    }

    if (ParseValues(value, ObjectDescription::BindFields(output))) {
      return true;
    } else {
      output = {};
      return false;
    }
  }

  template <typename InputType>
  static bool Emit(const InputType& input, ttv::json::Value& value) {
    if (EmitValues(value, ObjectDescription::BindFields(input))) {
      return true;
    } else {
      value = ttv::json::nullValue;
      return false;
    }
  }

 private:
  template <typename FieldsTuple>
  static bool ParseValues(const ttv::json::Value& value, FieldsTuple&& tuple) {
    return ParseValuesAtIndex<0>(value, std::forward<FieldsTuple>(tuple));
  }

  template <size_t index, typename FieldsTuple>
  static std::enable_if_t<(index < std::tuple_size<FieldsTuple>::value), bool> ParseValuesAtIndex(
    const ttv::json::Value& value, FieldsTuple&& tuple) {
    auto field = std::get<index>(tuple);
    if (!field.Parse(value)) {
      return false;
    } else {
      return ParseValuesAtIndex<index + 1>(value, std::forward<FieldsTuple>(tuple));
    }
  }

  template <size_t index, typename FieldsTuple>
  static std::enable_if_t<(index >= std::tuple_size<FieldsTuple>::value), bool> ParseValuesAtIndex(
    const ttv::json::Value& /*value*/, FieldsTuple&& /*tuple*/) {
    return true;
  }

  template <typename FieldsTuple>
  static bool EmitValues(ttv::json::Value& value, FieldsTuple&& tuple) {
    return EmitValuesAtIndex<0>(value, std::forward<FieldsTuple&&>(tuple));
  }

  template <size_t index, typename FieldsTuple>
  static std::enable_if_t<(index < std::tuple_size<FieldsTuple>::value), bool> EmitValuesAtIndex(
    ttv::json::Value& value, FieldsTuple&& tuple) {
    auto field = std::get<index>(tuple);
    if (!field.Emit(value) && decltype(field)::IsRequired) {
      return false;
    } else {
      return EmitValuesAtIndex<index + 1>(value, std::forward<FieldsTuple>(tuple));
    }
  }

  template <size_t index, typename FieldsTuple>
  static std::enable_if_t<(index >= std::tuple_size<FieldsTuple>::value), bool> EmitValuesAtIndex(
    ttv::json::Value& /*value*/, FieldsTuple&& /*tuple*/) {
    return true;
  }
};

struct ttv::json::RequiredField {
  static constexpr bool IsRequired = true;
};

struct ttv::json::OptionalField {
  static constexpr bool IsRequired = false;
};

template <typename FieldType, typename RequiredType, typename SchemaType>
class ttv::json::JsonField<FieldType, RequiredType, SchemaType, 1> {
 public:
  static constexpr bool IsRequired = RequiredType::IsRequired;

  JsonField(const char* key, FieldType& object) : mKey(key), mValue(object) {}

  bool Parse(const ttv::json::Value& parent) {
    if (!SchemaType::Parse(parent[mKey], mValue) && IsRequired) {
      return false;
    }
    return true;
  }

  bool Emit(ttv::json::Value& parent) { return SchemaType::Emit(mValue, parent[mKey]); }

 private:
  const char* mKey;
  FieldType& mValue;
};

template <typename DesiredType, typename RequiredType, typename SchemaType>
class ttv::json::VariantSelector<DesiredType, RequiredType, SchemaType, 1> {
 public:
  static constexpr bool IsRequired = RequiredType::IsRequired;

  VariantSelector(DesiredType& object) : mValue(object) {}

  bool Parse(const ttv::json::Value& parent) {
    if (!SchemaType::SelectSchema(parent, mValue) && IsRequired) {
      return false;
    }
    return true;
  }

  // TODO Emit(...) this is purposely left empty, additional
  // development needed to ToJson conversion

 private:
  DesiredType& mValue;
};

template <typename FieldType, typename RequiredType, typename SchemaType, size_t ArraySize>
class ttv::json::JsonField {
  static_assert(ArraySize != 0, "Key path for JSON field must have at least one key.");

 public:
  static constexpr bool IsRequired = RequiredType::IsRequired;

  JsonField(const std::array<const char*, ArraySize>& keyPath, FieldType& object) : mKeyPath(keyPath), mValue(object) {}

  bool Parse(const ttv::json::Value& parent) { return ParseHelper<0>(parent); }

  bool Emit(ttv::json::Value& parent) { return EmitHelper<0>(parent); }

 private:
  template <size_t index>
  std::enable_if_t<(index < ArraySize - 1), bool> ParseHelper(const ttv::json::Value& parent) {
    const auto& jIntermediate = parent[mKeyPath[index]];

    if (jIntermediate.isNull()) {
      return !IsRequired;
    } else if (!jIntermediate.isObject()) {
      return false;
    } else {
      return ParseHelper<index + 1>(jIntermediate);
    }
  }

  template <size_t index>
  std::enable_if_t<(index == ArraySize - 1), bool> ParseHelper(const ttv::json::Value& parent) {
    if (!SchemaType::Parse(parent[mKeyPath[index]], mValue) && IsRequired) {
      return false;
    }

    return true;
  }

  template <size_t index>
  std::enable_if_t<(index < ArraySize - 1), bool> EmitHelper(ttv::json::Value& parent) {
    auto& jIntermediate = parent[mKeyPath[index]];

    return EmitHelper<index + 1>(jIntermediate);
  }

  template <size_t index>
  std::enable_if_t<(index == ArraySize - 1), bool> EmitHelper(ttv::json::Value& parent) {
    return SchemaType::Emit(mValue, parent[mKeyPath[index]]);
  }

  std::array<const char*, ArraySize> mKeyPath;
  FieldType& mValue;
};

template <typename SchemaType, typename RequiredType, typename ObjectType>
auto ttv::json::make_field(const char* key, ObjectType& field) {
  return ttv::json::JsonField<ObjectType, RequiredType, SchemaType, 1>{key, field};
}

template <typename SchemaType, typename RequiredType, typename ObjectType, size_t ArraySize>
auto ttv::json::make_field(const std::array<const char*, ArraySize>& keyPath, ObjectType& field) {
  return ttv::json::JsonField<ObjectType, RequiredType, SchemaType, ArraySize>{keyPath, field};
}

template <typename RequiredType, typename ObjectType>
auto ttv::json::make_field(const char* key, ObjectType& field) {
  return make_field<DefaultSchema<std::decay_t<ObjectType>>, RequiredType, ObjectType>(key, field);
}

template <typename RequiredType, typename ObjectType, size_t ArraySize>
auto ttv::json::make_field(const std::array<const char*, ArraySize>& keyPath, ObjectType& field) {
  return make_field<DefaultSchema<std::decay_t<ObjectType>>, RequiredType, ObjectType, ArraySize>(keyPath, field);
}

template <typename ObjectType>
auto ttv::json::make_field(const char* key, ObjectType& field) {
  return make_field<DefaultSchema<std::decay_t<ObjectType>>, OptionalField, ObjectType>(key, field);
}

template <typename ObjectType, size_t ArraySize>
auto ttv::json::make_field(const std::array<const char*, ArraySize>& keyPath, ObjectType& field) {
  return make_field<DefaultSchema<std::decay_t<ObjectType>>, OptionalField, ObjectType, ArraySize>(keyPath, field);
}

template <typename SchemaType, typename RequiredType, typename ObjectType>
auto ttv::json::make_variant_selector(ObjectType& field) {
  return ttv::json::VariantSelector<ObjectType, RequiredType, SchemaType, 1>{field};
}

template <typename RequiredType, typename ObjectType>
auto ttv::json::make_variant_selector(ObjectType& field) {
  return make_variant_selector<DefaultSchema<std::decay_t<ObjectType>>, RequiredType, ObjectType>(field);
}

template <typename ObjectType>
auto ttv::json::make_variant_selector(ObjectType& field) {
  return make_variant_selector<DefaultSchema<std::decay_t<ObjectType>>, OptionalField, ObjectType>(field);
}

template <typename ObjectType, typename Enable>
struct ttv::json::DefaultSchemaProvider {
  static_assert(sizeof(ObjectType) < 0, "Default schema not found for object.");
};

template <typename UnsignedIntegralType>
struct ttv::json::DefaultSchemaProvider<UnsignedIntegralType,
  std::enable_if_t<std::is_integral<UnsignedIntegralType>::value && std::is_unsigned<UnsignedIntegralType>::value>> {
  using Type = UnsignedIntegerSchema;
};

template <typename IntegralType>
struct ttv::json::DefaultSchemaProvider<IntegralType,
  std::enable_if_t<std::is_integral<IntegralType>::value && !std::is_unsigned<IntegralType>::value>> {
  using Type = IntegerSchema;
};

template <typename FloatingPointType>
struct ttv::json::DefaultSchemaProvider<FloatingPointType,
  std::enable_if_t<std::is_floating_point<FloatingPointType>::value>> {
  using Type = FloatingPointSchema;
};

template <>
struct ttv::json::DefaultSchemaProvider<std::string> {
  using Type = StringSchema;
};

template <>
struct ttv::json::DefaultSchemaProvider<bool> {
  using Type = BooleanSchema;
};

template <typename ElementType>
struct ttv::json::DefaultSchemaProvider<std::vector<ElementType>> {
  using Type = ArraySchema<DefaultSchema<ElementType>>;
};

template <typename ElementType>
struct ttv::json::DefaultSchemaProvider<ttv::Optional<ElementType>> {
  using Type = OptionalSchema<DefaultSchema<ElementType>, ElementType>;
};

template <typename... Args>
std::array<const char*, sizeof...(Args)> ttv::json::MakeKeyPath(Args... args) {
  return std::array<const char*, sizeof...(Args)>{{args...}};
}
