/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/types/coretypes.h"

#include <unordered_map>

#include <list>

namespace ttv {
/**
 * A helper class which provides data caching. When full, evicts based on the least-recently-used entry.
 * Designed to store std::shared_ptrs as value_types; when storing primitives as value_types, be aware that ForEach
 * iterates over a copy of the cache. Not thread safe.
 */
template <typename KEY_TYPE, typename VALUE_TYPE>
class LruCache {
 public:
  using VisitorFunc = std::function<void(VALUE_TYPE& entry)>;

 protected:
  using KeyValuePair = typename std::pair<KEY_TYPE, VALUE_TYPE>;
  using ListIterator = typename std::list<KeyValuePair>::iterator;

 public:
  LruCache(size_t maxSize) : mMaxSize(maxSize) { TTV_ASSERT(maxSize != 0); }

  /**
   * Returns the number of entries in the cache.
   */
  uint32_t GetSize() const { return static_cast<uint32_t>(mListIteratorMap.size()); }

  /**
   * Clears the cache.
   */
  void Clear() {
    mListIteratorMap.clear();
    mCacheItemsList.clear();
  }

  /**
   * Caches the given value.
   */
  void SetEntry(const KEY_TYPE& key, VALUE_TYPE data) {
    auto iter = mListIteratorMap.find(key);
    if (iter != mListIteratorMap.end()) {
      mCacheItemsList.erase(iter->second);
      mListIteratorMap.erase(iter);
    }

    mCacheItemsList.push_front(KeyValuePair(key, data));
    mListIteratorMap[key] = mCacheItemsList.begin();

    if (mListIteratorMap.size() > mMaxSize) {
      // rbegin() will return an interator to the last element in the non-reversed list.
      auto last = mCacheItemsList.rbegin();
      mListIteratorMap.erase(last->first);
      mCacheItemsList.pop_back();
    }
  }

  /**
   * Explicitly removes the given value from the cache.
   */
  void RemoveEntry(const KEY_TYPE& key) {
    auto iter = mListIteratorMap.find(key);
    if (iter == mListIteratorMap.end()) {
      return;
    }

    mCacheItemsList.erase(iter->second);
    mListIteratorMap.erase(iter);
  }

  /**
   * Retrieves the value from the cache.
   * @return true if entry was found, false otherwise.
   */
  bool GetEntry(const KEY_TYPE& key, VALUE_TYPE& result) {
    auto iter = mListIteratorMap.find(key);
    if (iter != mListIteratorMap.end()) {
      // Move the value to the front of the list, most recently used.
      mCacheItemsList.splice(mCacheItemsList.begin(), mCacheItemsList, iter->second);
      result = iter->second->second;
    }

    return iter != mListIteratorMap.end();
  }

  /**
   * Retrieves the set of keys for cached values.
   */
  void GetKeys(std::vector<KEY_TYPE>& result) {
    for (auto kvp : mListIteratorMap) {
      result.push_back(kvp.first);
    }
  }

  /**
   * Determines if any data is cached under the given key.
   */
  bool ContainsEntry(const KEY_TYPE& key) const { return mListIteratorMap.find(key) != mListIteratorMap.end(); }

  /**
   * Visits each entry in the cache.
   */
  void ForEach(VisitorFunc func) {
    auto copy = mCacheItemsList;

    for (auto& value : copy) {
      func(value->second);
    }
  }

 protected:
  std::list<KeyValuePair> mCacheItemsList;
  std::unordered_map<KEY_TYPE, ListIterator> mListIteratorMap;
  size_t mMaxSize;
};
}  // namespace ttv
